/* convert Ascii in intermediate Yiddish form into other forms:
	Unicode, UTF-8 Unicode, MS Windows Hebrew, Mac Hebrew

Author: Raphael Finkel, 1997
See the COPYRIGHT file enclosed with this distribution.

NOTES:
	It appears than in the middle of a UTF-8 page, one may place codes of the
	form &#nnnn; where nnnn is the decimal equivalent of the 2-byte Unicode-2
	encoding.  This works Feb 1998 in Netscape Navigator 4 for some fonts (like
	standard Greek), not others (Hebrew).  The codes that are understood do
	*not* work when actually put in UTF-8.
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>

/* constants */
#define BUFLEN 10000
#define TRUE 1
#define FALSE 0

/* types */
typedef int Boolean; /* for clarity only */

/* variables */
char *CodeTable[256]; /* Unicode, Microsoft Windows, or Macintosh codes */
enum {Undef, Unicode, Mac, MS, QText} Style = Undef;
char *StyleString;
Boolean UTF = FALSE; /* in Unicode/UTF mode, we output UTF-8 */
Boolean Reverse = FALSE; /* in Reverse mode, each input line is reversed before
	conversion and <br> is placed between lines */
Boolean HTML = TRUE; /* generate HTML?  If not, omit HTML tags */
Boolean DynFont = FALSE; /* generate HTML tag for dynamic font? */
char *NewLine; /* in some codes, \n; in others, empty */
Boolean exact = FALSE; /* exact mode: preserve spacing */
Boolean Combiners = TRUE; /* whether we want combining chars (as opposed to
							 precombined combinations */
Boolean Entities = FALSE; /* output entities, not pure Unicode */
Boolean Ligatures = FALSE; /* output separate letters, not ligatures */

void InitUnicodes() {
	/* you can find these at
	http://www.blueneptune.com/~tseng/Unicode/normal/U+0590.html
	*/
	int ch;
	NewLine = "\n";
	for (ch = 0; ch <= 0xFF; ch++) { /* default Ascii conversion */
		CodeTable[ch] = 0;
	}
	/* specialized conversions */
	CodeTable['|'] = "";  /* remove prophylactic | */
	CodeTable['#'] = "\xd0";  /* HEBREW LETTER ALEF */
	CodeTable['B'] = "\xd1\xbf";  /* HEBREW LETTER BET,
									HEBREW POINT RAFE */
	CodeTable['b'] = "\xd1";  /* HEBREW LETTER BET */
	CodeTable['g'] = "\xd2";  /* HEBREW LETTER GIMEL */
	CodeTable['G'] = "\xd2\xbf";  /* HEBREW LETTER GIMEL,
									HEBREW POINT RAFE */
	CodeTable['d'] = "\xd3";  /* HEBREW LETTER DALET */
	CodeTable['h'] = "\xd4";  /* HEBREW LETTER HE */
	CodeTable['w'] = "\xd5";  /* HEBREW LETTER VAV */
	CodeTable['u'] = "\xd5\xbc";  /* HEBREW LETTER VAV
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['z'] = "\xd6";  /* HEBREW LETTER ZAYIN */
	CodeTable['H'] = "\xd7";  /* HEBREW LETTER HET */
	CodeTable['t'] = "\xd8";  /* HEBREW LETTER TET */
	CodeTable['y'] = "\xd9";  /* HEBREW LETTER YOD */
	CodeTable['X'] = "\xda";  /* HEBREW LETTER FINAL KAF */
	CodeTable['x'] = "\xdb";  /* HEBREW LETTER KAF */
	CodeTable['K'] = "\xdb\xbc";  /* HEBREW LETTER KAF,
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['l'] = "\xdc";  /* HEBREW LETTER LAMED */
	CodeTable['M'] = "\xdd";  /* HEBREW LETTER FINAL MEM */
	CodeTable['m'] = "\xde";  /* HEBREW LETTER MEM */
	CodeTable['N'] = "\xdf";  /* HEBREW LETTER FINAL NUN */
	CodeTable['n'] = "\xe0";  /* HEBREW LETTER NUN */
	CodeTable['s'] = "\xe1";  /* HEBREW LETTER SAMEKH */
	CodeTable['e'] = "\xe2";  /* HEBREW LETTER AYIN */
	CodeTable['F'] = "\xe3";  /* HEBREW LETTER FINAL PE */
	CodeTable['f'] = "\xe4\xbf";  /* HEBREW LETTER PE,
									HEBREW POINT RAFE */
	CodeTable['P'] = "\xe4";  /* HEBREW LETTER PE */
	CodeTable['p'] = "\xe4\xbc";  /* HEBREW LETTER PE,
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['C'] = "\xe5";	 /* HEBREW LETTER FINAL TSADI */
	CodeTable['c'] = "\xe6";  /* HEBREW LETTER TSADI */
	CodeTable['k'] = "\xe7";  /* HEBREW LETTER QOF */
	CodeTable['r'] = "\xe8";  /* HEBREW LETTER RESH */
	CodeTable['Q'] = "\xe9\xc2";  /* HEBREW LETTER SHIN,
									HEBREW POINT SIN DOT */
	CodeTable['S'] = "\xe9";  /* HEBREW LETTER SHIN */
	CodeTable['O'] = "\xf1";  /* HEBREW LIGATURE YIDDISH VAV YOD */
	CodeTable['T'] = "\xea";  /* HEBREW LETTER TAV */
	CodeTable['W'] = "\xea\xbc";  /* HEBREW LETTER TAV
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['Z'] = "\xd6\xbf";  /* HEBREW LETTER ZAYIN,
									HEBREW POINT RAFE */
	CodeTable['v'] = "\xf0";  /* HEBREW LIGATURE YIDDISH DOUBLE VAV */
	CodeTable['A'] = "\xf2";  /* HEBREW LIGATURE YIDDISH DOUBLE YOD */
	CodeTable['I'] = "\xf2\xb7";  /* HEBREW LIGATURE YIDDISH DOUBLE YOD,
									HEBREW POINT PATAH */
	CodeTable['a'] = "\xd0\xb7";  /* HEBREW LETTER ALEF,
									HEBREW POINT PATAH */
	CodeTable['o'] = "\xd0\xb8";  /* HEBREW LETTER ALEF,
									HEBREW POINT QAMATS */
	CodeTable['i'] = "\xd9\xb4";  /* HEBREW LETTER YOD,
									HEBREW POINT HIRIQ */
	CodeTable['-'] = "\xbe";  /* HEBREW hyphen */
	CodeTable[''] = "\x20\x1e";  /* HEBREW open quote */
	CodeTable[''] = "\x20\x1c";  /* HEBREW close quote */
	CodeTable[0x80] = "\xb7";  /* HEBREW POINT PATAH */
	CodeTable[0x81] = "\xb8";  /* HEBREW POINT QAMATS */
	CodeTable[0x82] = "\xb6";  /* HEBREW POINT SEGOL */
	CodeTable[0x83] = "\xb4";  /* HEBREW POINT HIRIQ */
	CodeTable[0x84] = "\xbb";  /* HEBREW POINT QUBUTS */
	CodeTable[0x85] = "\xb0";  /* HEBREW POINT SHEVA */
	CodeTable[0x86] = "\xb5";  /* HEBREW POINT TSERE */
	CodeTable[0x87] = "\xb9";  /* HEBREW POINT HOLAM */
	CodeTable[0x88] = "\xbc";  /* HEBREW POINT DAGESH OR MAPIQ */
	CodeTable[0x89] = "\xb2";  /* HEBREW POINT HATAF PATAH */
	CodeTable[0x90] = "\xb3";  /* HEBREW POINT HATAF QAMATS */
	CodeTable[0x91] = "\xb1";  /* HEBREW POINT HATAF SEGOL */
	CodeTable[0x92] = "\x20\x15";  /* quotation dash */
	CodeTable[0x93] = "\x20\x1e";  /* double comma */
	CodeTable[0x94] = "\x20\x1c";  /* double quote */
	CodeTable[0x95] = "\x20\x26";  /* three dots */
	if (!Combiners) {
		CodeTable['B'] = "\xfb\x4c"; /* HEBREW LETTER BET WITH RAFE */
		CodeTable['u'] = "\xfb\x35"; /* HEBREW LETTER VAV WITH DAGESH */
		CodeTable['K'] = "\xfb\x3b"; /* HEBREW LETTER KAF WITH DAGESH */
		CodeTable['f'] = "\xfb\x4e"; /* HEBREW LETTER PE WITH RAFE */
		CodeTable['p'] = "\xfb\x44"; /* HEBREW LETTER PE WITH DAGESH */
		CodeTable['Q'] = "\xfb\x2b"; /* HEBREW LETTER SHIN WITH SIN DOT */
		CodeTable['W'] = "\xfb\x4a"; /* HEBREW LETTER TAV WITH DAGESH */
		CodeTable['I'] = "\xfb\x1f"; /* HEBREW LIGATURE YIDDISH YOD YOD PATAH */
		CodeTable['a'] = "\xfb\x2e"; /* HEBREW LETTER ALEF WITH PATAH */
		CodeTable['o'] = "\xfb\x2f"; /* HEBREW LETTER ALEF WITH QAMATS */
		CodeTable['i'] = "\xfb\x1d"; /* HEBREW LETTER YOD WITH HIRIQ */
	}
	if (!Ligatures) {
		CodeTable['O'] = "\xd5\xd9";  /* HEBREW VAV, HEBREW YOD */
		CodeTable['v'] = "\xd5\xd5";  /* HEBREW LETTER VAV, twice */
		CodeTable['A'] = "\xd9\xd9";  /* HEBREW LETTER YOD, twice */
		CodeTable['I'] = "\xf2\xb7";  /* HEBREW LIGATURE YIDDISH DOUBLE YOD,
									HEBREW POINT PATAH */
	}
} /* InitUnicodes */

void InitMSCodes() {
	/* you can find these at
	ftp://unicode.org/pub/UNIX/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP1255.TXT
	*/
	int ch;
	NewLine = " ";
	for (ch = 0; ch <= 0xFF; ch++) { /* default Ascii conversion */
		CodeTable[ch] = 0;
	}
	CodeTable['~'] = " ";  /* non-breaking space */
	/* specialized conversions */
	CodeTable['#'] = "\xe0";  /* HEBREW LETTER ALEF */
	CodeTable['B'] = "\xe1\xcf";  /* HEBREW LETTER BET,
									HEBREW POINT RAFE */
	CodeTable['b'] = "\xe1";  /* HEBREW LETTER BET */
	CodeTable['g'] = "\xe2";  /* HEBREW LETTER GIMEL */
	CodeTable['d'] = "\xe3";  /* HEBREW LETTER DALET */
	CodeTable['h'] = "\xe4";  /* HEBREW LETTER HE */
	CodeTable['w'] = "\xe5";  /* HEBREW LETTER VAV */
	CodeTable['u'] = "\xe5\xcc";  /* HEBREW LETTER VAV
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['z'] = "\xe6";  /* HEBREW LETTER ZAYIN */
	CodeTable['H'] = "\xe7";  /* HEBREW LETTER HET */
	CodeTable['t'] = "\xe8";  /* HEBREW LETTER TET */
	CodeTable['y'] = "\xe9";  /* HEBREW LETTER YOD */
	CodeTable['X'] = "\xea";  /* HEBREW LETTER FINAL KAF */
	CodeTable['x'] = "\xeb";  /* HEBREW LETTER KAF */
	CodeTable['K'] = "\xeb\xcc";  /* HEBREW LETTER KAF,
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['l'] = "\xec";  /* HEBREW LETTER LAMED */
	CodeTable['M'] = "\xed";  /* HEBREW LETTER FINAL MEM */
	CodeTable['m'] = "\xee";  /* HEBREW LETTER MEM */
	CodeTable['N'] = "\xef";  /* HEBREW LETTER FINAL NUN */
	CodeTable['n'] = "\xf0";  /* HEBREW LETTER NUN */
	CodeTable['s'] = "\xf1";  /* HEBREW LETTER SAMEKH */
	CodeTable['e'] = "\xf2";  /* HEBREW LETTER AYIN */
	CodeTable['F'] = "\xf3";  /* HEBREW LETTER FINAL PE */
	CodeTable['f'] = "\xf4\xcf";  /* HEBREW LETTER PE,
									HEBREW POINT RAFE */
	CodeTable['P'] = "\xf4";  /* HEBREW LETTER PE */
	CodeTable['p'] = "\xf4\xcc";  /* HEBREW LETTER PE,
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['C'] = "\xf5";	 /* HEBREW LETTER FINAL TSADI */
	CodeTable['c'] = "\xf6";  /* HEBREW LETTER TSADI */
	CodeTable['k'] = "\xf7";  /* HEBREW LETTER QOF */
	CodeTable['r'] = "\xf8";  /* HEBREW LETTER RESH */
	CodeTable['Q'] = "\xf9\xd2";  /* HEBREW LETTER SHIN,
									HEBREW POINT SIN DOT */
	CodeTable['S'] = "\xf9";  /* HEBREW LETTER SHIN */
	CodeTable['O'] = "\xd5";  /* HEBREW LIGATURE YIDDISH VAV YOD */
	CodeTable['T'] = "\xfa";  /* HEBREW LETTER TAV */
	CodeTable['W'] = "\xfa\xcc";  /* HEBREW LETTER TAV
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['v'] = "\xd4";  /* HEBREW LIGATURE YIDDISH DOUBLE VAV */
	CodeTable['A'] = "\xd6";  /* HEBREW LIGATURE YIDDISH DOUBLE YOD */
	CodeTable['I'] = "\xd6\xc7";  /* HEBREW LIGATURE YIDDISH DOUBLE YOD,
									HEBREW POINT PATAH */
	CodeTable['a'] = "\xe0\xc7";  /* HEBREW LETTER ALEF,
									HEBREW POINT PATAH */
	CodeTable['o'] = "\xe0\xc8";  /* HEBREW LETTER ALEF,
									HEBREW POINT QAMATS */
	CodeTable['i'] = "\xe9\xc4";  /* HEBREW LETTER YOD,
									HEBREW POINT HIRIQ */
	CodeTable[0x80] = "\xc7"; /* pasekh */
	CodeTable[0x81] = "\xc8"; /* komets */
	CodeTable[0x82] = "\xc6"; /* segol */
	CodeTable[0x83] = "\xc4"; /* khirik */
	CodeTable[0x84] = "\xcb"; /* kubuts */
	CodeTable[0x85] = "\xc0"; /* sheva */
	CodeTable[0x86] = "\xc5"; /* tseyre */
	CodeTable[0x87] = "\xc9"; /* kholem */
	CodeTable[0x88] = "\xcc"; /* dagesh */
	CodeTable[0x89] = "\xc3"; /* khatafpasekh */
	CodeTable[0x90] = "\xc2"; /* khatafkomets */
	CodeTable[0x91] = "\xc1"; /* katafsegol */
} /* InitMSCodes */

void InitMacCodes() {
	/* you can find these at
	ftp://unicode.org/pub/UNIX/MAPPINGS/VENDORS/APPLE/HEBREW.TXT
	*/
	int ch;
	NewLine = " ";
	for (ch = 0; ch <= 0xFF; ch++) { /* default Ascii conversion */
		CodeTable[ch] = 0;
	}
	CodeTable['~'] = " ";  /* non-breaking space */
	/* specialized conversions */
	CodeTable['!'] = "\xa1";  /* RTO + ! */
	CodeTable['"'] = "\xa2";  /* RTO + " */
	CodeTable['$'] = "\xa4";  /* RTO + $ */
	CodeTable['%'] = "\xa5";  /* RTO + % */
	CodeTable['\''] = "\xa7";  /* RTO + ' */
	CodeTable[')'] = "\xa8";  /* RTO + ( */
	CodeTable['('] = "\xa9";  /* RTO + ) */
	CodeTable['['] = "\xfc";  /* RTO + ] */
	CodeTable[']'] = "\xfe";  /* RTO + [ */
	CodeTable['{'] = "\xfb";  /* RTO + } */
	CodeTable['}'] = "\xfd";  /* RTO + { */
	CodeTable['*'] = "\xaa";  /* RTO + * */
	CodeTable['+'] = "\xab";  /* RTO + + */
	CodeTable[','] = "\xac";  /* RTO + , */
	CodeTable['-'] = "\xad";  /* RTO + - */
	CodeTable['.'] = "\xae";  /* RTO + . */
	CodeTable[':'] = "\xba";  /* RTO + : */
	CodeTable[';'] = "\xbb";  /* RTO + ; */
	CodeTable['?'] = "\xbf";  /* RTO + ? */
	CodeTable['='] = "\xbd";  /* RTO + = */
	CodeTable['#'] = "\xe0";  /* HEBREW LETTER ALEF */
	CodeTable['B'] = "\xe1\xd8";  /* HEBREW LETTER BET,
									HEBREW POINT RAFE */
	CodeTable['b'] = "\xe1";  /* HEBREW LETTER BET */
	CodeTable['g'] = "\xe2";  /* HEBREW LETTER GIMEL */
	CodeTable['d'] = "\xe3";  /* HEBREW LETTER DALET */
	CodeTable['h'] = "\xe4";  /* HEBREW LETTER HE */
	CodeTable['w'] = "\xe5";  /* HEBREW LETTER VAV */
	CodeTable['u'] = "\xe5\xc6";  /* HEBREW LETTER VAV
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['z'] = "\xe6";  /* HEBREW LETTER ZAYIN */
	CodeTable['H'] = "\xe7";  /* HEBREW LETTER HET */
	CodeTable['t'] = "\xe8";  /* HEBREW LETTER TET */
	CodeTable['y'] = "\xe9";  /* HEBREW LETTER YOD */
	CodeTable['X'] = "\xea";  /* HEBREW LETTER FINAL KAF */
	CodeTable['x'] = "\xeb";  /* HEBREW LETTER KAF */
	CodeTable['K'] = "\xeb\xc6";  /* HEBREW LETTER KAF,
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['l'] = "\xec";  /* HEBREW LETTER LAMED */
	CodeTable['M'] = "\xed";  /* HEBREW LETTER FINAL MEM */
	CodeTable['m'] = "\xee";  /* HEBREW LETTER MEM */
	CodeTable['N'] = "\xef";  /* HEBREW LETTER FINAL NUN */
	CodeTable['n'] = "\xf0";  /* HEBREW LETTER NUN */
	CodeTable['s'] = "\xf1";  /* HEBREW LETTER SAMEKH */
	CodeTable['e'] = "\xf2";  /* HEBREW LETTER AYIN */
	CodeTable['F'] = "\xf3";  /* HEBREW LETTER FINAL PE */
	CodeTable['f'] = "\xf4\xd8";  /* HEBREW LETTER PE,
									HEBREW POINT RAFE */
	CodeTable['P'] = "\xf4";  /* HEBREW LETTER PE */
	CodeTable['p'] = "\xf4\xc6";  /* HEBREW LETTER PE,
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['C'] = "\xf5";	 /* HEBREW LETTER FINAL TSADI */
	CodeTable['c'] = "\xf6";  /* HEBREW LETTER TSADI */
	CodeTable['k'] = "\xf7";  /* HEBREW LETTER QOF */
	CodeTable['r'] = "\xf8";  /* HEBREW LETTER RESH */
	CodeTable['Q'] = "\xd7";  /* HEBREW LETTER SHIN WITH SIN DOT */
	CodeTable['S'] = "\xf9";  /* HEBREW LETTER SHIN */
	CodeTable['O'] = "\xe5\xe9";  /* HEBREW LETTER VAV, HEBREW LETTER YOD */
	CodeTable['T'] = "\xfa";  /* HEBREW LETTER TAV */
	CodeTable['W'] = "\xfa\xc6";  /* HEBREW LETTER TAV
									HEBREW POINT DAGESH OR MAPIQ */
	CodeTable['v'] = "\xe5\xe5";  /* HEBREW LETTER VAV [twice] */
	CodeTable['A'] = "\xe9\xe9";  /* HEBREW LETTER YOD [twice] */
	CodeTable['I'] = "\x81";  /* HEBREW LIGATURE YIDDISH YOD YOD PATAH */
	/* CodeTable['I'] = "\xe9\xe9\xcc";  HEBREW LETTER YOD, YOD, PATAH */
	CodeTable['a'] = "\xe0\xcc";  /* HEBREW LETTER ALEF,
									HEBREW POINT PATAH */
	CodeTable['o'] = "\xe0\xcb";  /* HEBREW LETTER ALEF,
									HEBREW POINT QAMATS */
	CodeTable['i'] = "\xe9\xcf";  /* HEBREW LETTER YOD,
									HEBREW POINT HIRIQ */
	CodeTable[0x80] = "\xcc"; /* pasekh */
	CodeTable[0x81] = "\xcb"; /* komets */
	CodeTable[0x82] = "\xce"; /* segol */
	CodeTable[0x83] = "\xcf"; /* khirik */
	CodeTable[0x84] = "\xdc"; /* kubuts */
	CodeTable[0x85] = "\xd9"; /* sheva */
	CodeTable[0x86] = "\xcd"; /* tseyre */
	CodeTable[0x87] = "\xdd"; /* kholem */
	CodeTable[0x88] = "\xc6"; /* dagesh */
	CodeTable[0x89] = "\xda"; /* khatafpasekh */
	CodeTable[0x90] = "\xdf"; /* khatafkomets */
	CodeTable[0x91] = "\xdb"; /* katafsegol */
} /* InitMacCodes */

void InitQCodes() {
	/* QText fonts */
	int ch;
	NewLine = " ";
	for (ch = 0; ch <= 0xFF; ch++) { /* default Ascii conversion */
		CodeTable[ch] = 0;
	}
	CodeTable['~'] = " ";  /* non-breaking space */
	/* specialized conversions */
	CodeTable[')'] = "(";
	CodeTable['('] = ")";
	CodeTable['['] = "]";
	CodeTable[']'] = "[";
	CodeTable['#'] = "\xe0";  /* YIDDISH LETTER ALEF */
	CodeTable['B'] = "\xd9";  /* YIDDISH LETTER VEYS */
	CodeTable['b'] = "\xe1";  /* YIDDISH LETTER BEYS */
	CodeTable['g'] = "\xe2";  /* YIDDISH LETTER GIMEL */
	CodeTable['d'] = "\xe3";  /* YIDDISH LETTER DALED */
	CodeTable['h'] = "\xe4";  /* YIDDISH LETTER HE */
	CodeTable['w'] = "\xe5";  /* YIDDISH LETTER VOV */
	CodeTable['u'] = "\xda";  /* YIDDISH LETTER MELUPN-VOV */
	CodeTable['z'] = "\xe6";  /* YIDDISH LETTER ZAYIN */
	CodeTable['H'] = "\xe7";  /* YIDDISH LETTER KHES */
	CodeTable['t'] = "\xe8";  /* YIDDISH LETTER TES */
	CodeTable['y'] = "\xe9";  /* YIDDISH LETTER YUD */
	CodeTable['X'] = "\xea";  /* YIDDISH LETTER FINAL KHOF */
	CodeTable['x'] = "\xeb";  /* YIDDISH LETTER KHOF */
	CodeTable['K'] = "\xdd";     /* YIDDISH LETTER KOF */
	CodeTable['l'] = "\xec";  /* YIDDISH LETTER LAMED */
	CodeTable['M'] = "\xed";  /* YIDDISH LETTER SHLOS-MEM */
	CodeTable['m'] = "\xee";  /* YIDDISH LETTER MEM */
	CodeTable['N'] = "\xef";  /* YIDDISH LETTER LANGE-NUN */
	CodeTable['n'] = "\xf0";  /* YIDDISH LETTER NUN */
	CodeTable['s'] = "\xf1";  /* YIDDISH LETTER SAMEKH */
	CodeTable['e'] = "\xf2";  /* YIDDISH LETTER AYIN */
	CodeTable['F'] = "\xf3";  /* YIDDISH LETTER LANGE-PE */
	CodeTable['f'] = "\xdf";  /* YIDDISH LETTER FE */
	CodeTable['p'] = "\xde";  /* YIDDISH LETTER PE */
	CodeTable['C'] = "\xf5";	 /* YIDDISH LETTER LANGE-TSADIK */
	CodeTable['c'] = "\xf6";  /* YIDDISH LETTER TSADIK */
	CodeTable['k'] = "\xf7";  /* YIDDISH LETTER KUF */
	CodeTable['r'] = "\xf8";  /* YIDDISH LETTER REYSH */
	CodeTable['Q'] = "\xfb";  /* YIDDISH LETTER SIN */
	CodeTable['S'] = "\xf9";  /* YIDDISH LETTER SHIN */
	CodeTable['O'] = "\xd5";  /* YIDDISH LETTER VOV-YOD */
	CodeTable['T'] = "\xfa";  /* YIDDISH LETTER SOF */
	CodeTable['W'] = "\xfc";  /* YIDDISH LETTER TOF */
	CodeTable['v'] = "\xd4";  /* YIDDISH LETTER TSVEY-VOV */
	CodeTable['A'] = "\xd6";  /* YIDDISH LETTER TSVEY-YUD */
	CodeTable['I'] = "\xdc";  /* YIDDISH LETTER PASEKH-TSVEY-YUD */
	CodeTable['a'] = "\xd7";  /* YIDDISH LETTER PASEKH-ALEF */
	CodeTable['o'] = "\xd8";  /* YIDDISH LETTER KOMETS-ALEF */
	CodeTable['i'] = "\xdb";  /* YIDDISH LETTER PINTL-YUD */
} /* InitQCodes */

void Ascii2OutFormat(char *line) {
	char *ChPtr;
	if (Style == Unicode && !UTF) {
		for (ChPtr = line; *ChPtr; ChPtr++)
			fprintf(stdout, "%c%c", 0, *ChPtr);
	} else {
		fprintf(stdout, "%s", line);
	}
} /* Ascii2OutFormat */

void Unicode2OutFormat(unsigned char ch1, unsigned char ch2) {
	/* ch1 is the high (earlier) byte */
	unsigned long mychar = ((long) ch1 << 8) | ch2;
	if (Entities) { /* HTML entities format */
		fprintf(stdout, "&#%lu;", mychar);
	} else if (UTF) { /* UTF-8 format */
		if (mychar <= 0x7F) { /* 7 sig bits; plain 7-bit ascii */
			fprintf(stdout, "%c", (char) mychar);
		} else if (mychar <= 0x7FF) { /* 11 sig bits; Hebrew is in this range */
			fprintf(stdout, "%c%c",
				0300 | (int) ((mychar >> 6)&037), /* upper 5 bits */
				0200 | (int) (mychar & 077)); /* lower 6 bits */
		} else if (mychar <= 0xFFFF) { /* 16 sig bits */
			fprintf(stdout, "%c%c%c",
				0340 | (int) ((mychar >> 12)&017), /* upper 4 bits */
				0200 | (int) ((mychar >> 6)&077), /* next 6 bits */
				0200 | (int) (mychar & 077)); /* lowest 6 bits */
		} else { /* don't even try; beyond Unicode-2 */
			fprintf(stderr, "cannot convert 0x%x,0x%x to UTF-8\n", ch1, ch2);
		}
	} else { /* Unicode format */
		fprintf(stdout, "%c%c", ch1, ch2);
	}
} /* Unicode2OutFormat */

void Invert(char *string) {
	/* reverse the characters of string in place */
	char *first = string, *last = string + strlen(string) - 1;
	while (first < last) {
		char tmp = *first;
		*first = *last;
		*last = tmp;
		first++;
		last--;
	} /* while first < last */
} /* Invert */
		
void OneLine(char *line) {
	char *ChPtr, *ptr;
	if (*line == 0) { /* empty line */
		if (HTML) {
			char buf[BUFLEN];
			sprintf(buf, "\n<p%s>\n", UTF ? " align=right": "");
			Ascii2OutFormat(buf);
		} else {	
			Ascii2OutFormat("\n");
		}
		return;
	}
	if (*line == '\\' && *(line+1) == '\\') { /* TeX command: start new line */
		Ascii2OutFormat("\n");
		if (HTML && !Reverse) Ascii2OutFormat("<br>");
		return;
	}
	if ((ChPtr = strchr(line, '\\'))) { /* TeX commands */
		if (strncmp(ChPtr, "\\relax {\\roman ", 15) == 0) {
			 /* protected line; output straight */
			(void) strcpy(ChPtr, ChPtr+15); /* remove head */
			while ((ChPtr = strrchr(line,'}'))) { /* remove tail */
				(void) strcpy(ChPtr, ChPtr+1);
			}
			while ((ChPtr = strrchr(line,'~'))) { /* remove ~ chars */
				*ChPtr = ' ';
			}
			Invert(line); /* put it back in straight order */
			Ascii2OutFormat(line);
			Ascii2OutFormat(NewLine);
		} else if (strncmp(ChPtr, "\\english{", 9) == 0) { /* not Yiddish */
			(void) strcpy(ChPtr, ChPtr+9); /* remove head */
			while ((ChPtr = strrchr(line,'}'))) { /* remove tail */
				(void) strcpy(ChPtr, ChPtr+1);
			}
			while ((ChPtr = strrchr(line,'~'))) { /* remove ~ chars */
				*ChPtr = ' ';
			}
			Invert(line); /* put it back in straight order */
			Ascii2OutFormat(line);
			Ascii2OutFormat(NewLine);
		} else if (strncmp(ChPtr, "\\exact", 6) == 0) { /* exact mode */
			exact = TRUE;
		} else if (HTML && exact && strncmp(ChPtr, "\\hspace", 7) == 0) {
			Ascii2OutFormat("&nbsp;");
		} else if (HTML && strncmp(ChPtr, "\\vspace", 7) == 0) {
			Ascii2OutFormat("<br clear=all><p>");
			Ascii2OutFormat("<IMG SRC=../../rnb-line.gif WIDTH=60% HEIGHT=4 ");
			Ascii2OutFormat("alt=\"\"><p><br clear=all>\n");
		}
		return; /* ignore other TeX lines */
	}
	switch (Style) {
		case Unicode: /* must do it two at a time */
			for (ChPtr = line; *ChPtr; ChPtr++) {
				if (CodeTable[(unsigned char) (*ChPtr)]) {
					/* Yiddish; needs conversion */
					if (Reverse) {
						ptr = CodeTable[(unsigned char) (*ChPtr)];
						for (ptr += strlen(ptr);
						  ptr != CodeTable[(unsigned char) (*ChPtr)];) {
							ptr -= 2;
							Unicode2OutFormat(*ptr, *(ptr + 1));
						}
					} else {
						for (ptr = CodeTable[(unsigned char) (*ChPtr)]; *ptr;
						 ptr = ptr+2)
							Unicode2OutFormat(*ptr, *(ptr + 1));
					}
				} else { /* ordinary ascii; prefix with a null. */
					char Buf[2] = {*ChPtr, 0};
					// fprintf(stdout, "point 2");
					if (*ChPtr == '~') { /* nonbreaking space */
						Ascii2OutFormat("&nbsp;");
					} else if (exact && HTML && *ChPtr == ' ') { /* space */
						Ascii2OutFormat("&nbsp;");
					} else {
						/* we don't need to reverse open parens and brackets;
						they are weakly directional */
						Ascii2OutFormat(Buf);
					}
				}
			}
			break;
		default: /* one char at a time */
			// fprintf(stdout, "point 1");
			for (ChPtr = line; *ChPtr; ChPtr++) {
				if (exact && HTML && *ChPtr == ' ') { /* space */
					Ascii2OutFormat("&nbsp;");
				} else if (CodeTable[(unsigned char) (*ChPtr)]) {
						for (ptr = CodeTable[(unsigned char) (*ChPtr)]; *ptr;
						 ptr++)
							fprintf(stdout, "%c", *ptr);
				} else { /* ordinary ascii */
					fprintf(stdout, "%c", *ChPtr);
				}
			}
	} /* switch Style */
	if (Style != QText) {
		Ascii2OutFormat(NewLine);
	}
} /* OneLine */

void OneFile(FILE *MyFile) {
	char Buffer[BUFLEN];
	if (Style == Unicode && !UTF) { /* raw Unicode */
		fprintf(stdout, "%c%c", '\xfe', '\xff'); /* non-breaking space to warn
			user agent of byte order */
	}
	if (HTML) { /* HTML headers */
		char HeaderBuffer[BUFLEN];
		Ascii2OutFormat("<html lang=yi");
		if (Reverse) {
			Ascii2OutFormat(">");
		} else {
			Ascii2OutFormat(" dir=RTL>");
		}
		Ascii2OutFormat("<head>\n");
		if (UTF) { /* UTF-8  */
			Ascii2OutFormat("<META HTTP-EQUIV=\"Content-Type\" ");
			Ascii2OutFormat("CONTENT=\"text/html; charset=UTF-8\">\n");
		} else if (Style == MS) {
			Ascii2OutFormat("<META HTTP-EQUIV=\"Content-Type\" ");
			Ascii2OutFormat("CONTENT=\"text/html; charset=windows-1255\">\n");
		}
		if (DynFont) {
			Ascii2OutFormat("\n<LINK REL=FONTDEF ");
			Ascii2OutFormat("SRC=http://www.cs.uky.edu/~srthes0/yiddish.pfr>\n");
			Ascii2OutFormat("<SCRIPT SRC=http://www.akruti.com/tdserver.js ");
			Ascii2OutFormat("TYPE=text/javascript>\n</SCRIPT>\n");
		}
		sprintf(HeaderBuffer, "<title>\n%s form\n</title>\n", StyleString);
		Ascii2OutFormat(HeaderBuffer);
		Ascii2OutFormat("</head>\n<body><p>\n");
		if (DynFont) {
			Ascii2OutFormat("<DIV ALIGN=right><FONT FACE=QDavid>\n");
		}
	} /* HTML headers */
   	if (Style == Unicode) {
		// Unicode2OutFormat('\x20', '\x2b'); // RTL embedding
		Unicode2OutFormat('\x20', '\x0f'); // RTL mark
	}
	while (fgets(Buffer, BUFLEN, MyFile) != NULL) {
		Buffer[strlen(Buffer)-1] = 0; /* remove the \n */
		if (Reverse) Invert(Buffer);
		OneLine(Buffer);
		if (Reverse && HTML && Buffer[0]!='\\') Ascii2OutFormat("<br>");
	}
	if (Style == Unicode) {
		// Unicode2OutFormat('\x20', '\x2c'); // pop directional formatting
		Unicode2OutFormat('\x20', '\x0f'); // RTL mark
	}
	if (HTML) {
		Ascii2OutFormat("</body></html>\n");
	}
} /* OneFile */

void Usage(char *argv[]) {
	/* 
	fprintf(stderr, "Usage: %s [-ntupewcqrh]\n", argv[0]);
	fprintf(stderr,
		"(Unicode/UTF-8/UTF-8 precombined/Unicode entities/\n"
		"Unicode entities precombined/MS Windows/Mac/QText),\n"
		" (reversed) (no HTML)\n");
	*/
	fprintf(stderr, "Usage: %s [-u[lpe8]wcq[rh]]\n", argv[0]);
	fprintf(stderr,
		"\tUnicode(+ligatures, +precombined, +entitites, +UTF8)\n"
		"MS Windows/Mac/QText),\n"
		" (+reverse) (-HTML)\n");
	exit(1);
} /* Usage */

int main(int argc, char *argv[]) {
	char ThisOpt;
	UTF = FALSE;
	Combiners = TRUE;
	Entities = FALSE;

	while ((ThisOpt = getopt(argc, argv, "u,l,p,e,8,w,c,q,r,h")) != (char) -1) {
		switch (ThisOpt) {
			case 'u': Style = Unicode; break;
			case 'l': Ligatures = TRUE; break;
			case 'p': Combiners = FALSE; break;
			case 'e': Entities = TRUE; break;
			case '8': UTF = TRUE; break;
			case 'w': Style = MS; break;
			case 'c': Style = Mac; break;
			case 'q': Style = QText; break;
			case 'r': Reverse = TRUE; break;
			case 'h': HTML = FALSE; break;
			case '?': Usage(argv); break;
		}
	} /* while getting options */
	switch (Style) {
		case Unicode:
			InitUnicodes();
			StyleString = "Unicode";
			break;
		case MS:
			InitMSCodes();
			StyleString = "MS-Hebrew";
			break;
		case Mac:
			InitMacCodes();
			StyleString = "Mac Hebrew";
			break;
		case QText:
			InitQCodes();
			StyleString = "QText";
			break;
		case Undef:
			Usage(argv);
			return(1);
	} /* switch Style */
	OneFile(stdin);
	return(0);
}
