#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#ifdef SunOS
#include <sys/types.h> 
#include <sys/ddi.h>
#endif

#include "xmevents.h"
#include "xmdefs.h"
#include "xmerror.h"

#define AWAIT_REPLY_TABLE_SIZE 16 /* Max replies we will wait for.  
                                     Can increase  */

static Boolean xmWaitForData(int, int);
static void xmShip(register XMSession *, int, int);
static void xmPutAwaitTable(register XMSession *, CARD8, CARD32);
static void xmCheckAwaitingReply(register XMSession *, CARD16, int);

int 
xmReadServer(register XMSession *session, int index)
{
  unsigned int serv_data_size; /* Server length size is 4 bytes */
  unsigned int data_size;
  char header[SERVER_HEADER_SIZE];
  int header_size;
  int socket;
  Boolean more_data;

  session->curr_buffer_len = 0;
  session->num_packets = 0; 
  more_data = TRUE;

  socket = session->xservers[index]->socketd;

  while(more_data) {

    /* This loop is entered only if there is some data available on this socket.
       If the size of the data is zero, there is something wrong with the 
       connection. */

    /* First read the header */

    if ((header_size = read(socket, header, SERVER_HEADER_SIZE)) <= 0)
      return 0;

		/* Find out the sequence number of the last request processed by the 
			 master server */
    if (index == MASTER_XSERVER)
      session->seq_num_last_processed = GETCARD16(header + SEQ_NUM_OFFSET);


    /* In data coming from an X server, only Replies can have
       more than SERVER_HEADER_SIZE bytes of data.  Events and Errors
       will have exactly SERVER_HEADER_SIZE bytes of data.  The first
       byte of the header will tell us if the packet is a Reply. */

    if (*header == XSERVER_REPLY) {
      /* Length of data is available at offset 4.  It holds
         number of bytes in units of 4 bytes. */

      serv_data_size = GETCARD32(header + SERV_SIZE_OFFSET);
      data_size = serv_data_size << 2;
      
      /* If the header says the size of the packet is absurdly big,
         we throw the packet away */
      if (data_size > session->max_buffer_len)
        data_size = 0;
    }
    else 
      data_size = 0;


    /* If we have a huge packet already and the next one cannot be fit into
       this packet, ship out this packet first and then handle the next 
       packet  */

    if (session->curr_buffer_len+header_size+data_size > session->max_buffer_len
        || session->num_packets == MAX_PACKETS) {
      xmShip(session, index, XSERVER);
      session->num_packets = 0;
      session->curr_buffer_len = 0;
    }

    /* Store the offset */
    
    session->offsets[session->num_packets++] = session->curr_buffer_len;

    memcpy(session->buffer+session->curr_buffer_len, header, header_size);
    session->curr_buffer_len += header_size;
      

    while(data_size > 0) {
      unsigned int num_data;
      int numTimes;
      
      /* Read the data whose size is mentioned in the header */
                           
      numTimes = 0;
      /* Wait for 10 ms and test repeatedly */
      while(!xmWaitForData(socket, 10) && numTimes <= 1000) 
        numTimes++;

      if (numTimes > 1000)
      	return 0;

      if ((num_data = read(socket, session->buffer+session->curr_buffer_len, 
                           data_size)) <= 0)
        return 0;

      data_size -= num_data;
      session->curr_buffer_len += num_data;
    }

    /* Check if this is a reply to an earlier request for which we are 
       waiting */
    if (session->await != NULL && *header == XSERVER_REPLY)
      xmCheckAwaitingReply(session, GETCARD16(header+SEQ_NUM_OFFSET), index);

		if ((index == MASTER_XSERVER) && 
				(*header == Ev_ConfigureNotify || *header == Ev_ConfigureRequest)) {
			session->ResizeReqBuf = (char *)malloc(SERVER_HEADER_SIZE);
			assert(session->ResizeReqBuf != NULL);
			memcpy(session->ResizeReqBuf, header, SERVER_HEADER_SIZE);
		}

    if (!xmWaitForData(socket, 0))
      more_data = FALSE;
  }

  xmShip(session, index, XSERVER);
  return session->curr_buffer_len;
}

int 
xmReadClient(register XMSession *session)
{
  unsigned short int cli_data_size; /* Client length size is 2 bytes */
  unsigned int data_size;
  char header[CLIENT_HEADER_SIZE];
  int header_size;
  int socket;
  CARD32 seq;
  Boolean more_data;

  seq = session->xclient->sequence_num;
  session->curr_buffer_len = 0;
  session->num_packets = 0; 
  more_data = TRUE;

  socket = session->xclient->socketd;

  while(more_data) {

    /* This loop is entered only if there is some data available on this socket.
       If the size of the data is zero, there is something wrong with the 
       connection. */

    /* First read the header */

    if ((header_size = read(socket, header, CLIENT_HEADER_SIZE)) <= 0)
      return 0;

    /* The data came from an X client.  The length is encoded in bytes 3 and 4.
       The length is expressed in units of 4 bytes. */

    cli_data_size = GETCARD16(header + CLI_SIZE_OFFSET);
    data_size = (unsigned int) (cli_data_size << 2);

    /* X client "Request packet" length is length of header plus length 
       of data.  */
    data_size -= header_size;

    /* If we have a huge packet already and the next one cannot be fit 
       into this packet, ship out this packet first and then handle the 
       next packet  */

    if (session->curr_buffer_len+header_size+data_size > session->max_buffer_len
        || session->num_packets == MAX_PACKETS) {
      xmShip(session, 0, XCLIENT);
      session->num_packets = 0;
      session->curr_buffer_len = 0;
    }

    /* Stow away the offset in session->buffer at which the header of this 
       packet begins */
                   
    session->offsets[session->num_packets++] = session->curr_buffer_len; 

    memcpy(session->buffer+session->curr_buffer_len, header, header_size);
    session->curr_buffer_len += header_size;
      
    while(data_size > 0) {
      int num_data;
      int numTimes;

      /* Read the data whose size is mentioned in the header */
                           
      numTimes = 0;
      /* Wait for 10 ms and test repeatedly */
      while(!xmWaitForData(socket, 10) && numTimes <= 1000) 
      	numTimes++;

      if (numTimes > 1000)
      	return 0;

      if ((num_data = read(socket, session->buffer+session->curr_buffer_len, 
                           data_size)) <= 0) 
        return 0;

      data_size -= num_data;
      session->curr_buffer_len += num_data;
    }

    /* Sequence number needs to be handled here.
    session->xclient->sequence_num++; */
    seq++;

    /* For some requests we have to grab information from replies coming from 
       the X servers.  If this is one such packet, set the sequence number we 
       are waiting for */
    if (*header == X_InternAtom)
      xmPutAwaitTable(session, *header, seq);

    if (!xmWaitForData(socket, 0))
      more_data = FALSE;
  }

  xmShip(session, 0, XCLIENT);
  return session->curr_buffer_len;
}


static void 
xmShip(register XMSession *session, int index, int serv_or_cli)
{
  EventList *exposeEventList;

  xmLOG("\n ------------------------------------------------------\n");
  xmLOG("Number of fake requests = %d\n", session->numFakeReqs);

  if (serv_or_cli & XSERVER) {
    xmLOG("%d bytes from SERVER %d to CLIENT\n", session->curr_buffer_len, 
                                                   index);
    xmAnalyze(session, XSERVER, index); 

    if (index != MASTER_XSERVER) {
      /* We transmit only expose events from the slave servers */
      exposeEventList = xmMapResIDsAtoms(session, MASTER_XSERVER, 
																				 index, XSERVER, NULL, 0);
      if (exposeEventList->buffer != NULL) {
        write(session->xclient->socketd, exposeEventList->buffer, 
              exposeEventList->length);
        free(exposeEventList->buffer);
        
      }
      free(exposeEventList);
    }
    else /* index == MASTER_XSERVER */
      write(session->xclient->socketd, session->buffer, 
            session->curr_buffer_len);

    if (xmlog)
      writeall(session->buffer, session->curr_buffer_len, XSERVER, 0); 
  }
  else {
    xmLOG("%d bytes from CLIENT\n", session->curr_buffer_len);

    xmAnalyze(session, XCLIENT, index); 

    /* Send data to all the slave servers and also map resource ids */
    xmMultiCast(session, ALL_SERVERS);
  }
}

/* Write to all servers */
void 
xmMultiCast(register XMSession *session, int type)
{
  int i;
  int prev = FIRST_SERVER;  /* Index of last server written to */
	Atom *atomsIndexList;
	int atomsListSize;

	xmGetAtomsList(session, &atomsIndexList, &atomsListSize);

  for(i = 0; i < MAX_CONNECTIONS; i++) 
    if (session->xservers[i] != NULL) {
      /* We need not change the resource ids and atoms for the master server.  
				 For every one else, we need the right resource ids and atoms.  */
         
      if (i != MASTER_XSERVER && 
					(type == ALL_SERVERS || type == SLAVE_SERVERS_ONLY))
        xmMapResIDsAtoms(session, i, prev, XCLIENT, 
												 atomsIndexList, atomsListSize); 
      
			if (i == MASTER_XSERVER && type == SLAVE_SERVERS_ONLY)
				continue;

      write(session->xservers[i]->socketd, session->buffer, 
            session->curr_buffer_len);

      if (xmlog)
        writeall(session->buffer, session->curr_buffer_len, XCLIENT, i);
      prev = i;
    }

		free(atomsIndexList);
}

void 
writeall(char *buffer, int numdata, int serv_or_cli, int index)
{
  int i;
  int j;
  CARD16 seq;
  unsigned int size;

  if (serv_or_cli & XSERVER) {
     i = 0;
     do {
        seq = GETCARD16(buffer+i+2);
        xmLOG("Sequence Number = %u\n", seq);
        if (buffer[i] == 1)
          size = (GETCARD32(buffer+i+4) << 2) + 32;
        else
          size = 32;
        if (size > 262144)
          size = 32;
        xmLOG("Offset: %d\n", i);
        for(j = 0; j < size; j++) {
          xmLOG(" %d", (unsigned char) buffer[i+j]);
          if (isalnum(buffer[i+j]))
            xmLOG("%c", (unsigned char) buffer[i+j]);
        }
        numdata -= size;
        i += size;
        xmLOG("\n");
     } while(numdata > 0);
  }
  else {
    xmLOG("From Client to Server %d;  Number of Bytes = %d\n", index, numdata);
    for(j = 0; j < numdata; j++) {
      xmLOG(" %d", (unsigned char) buffer[j]);
      if (isalnum(buffer[j]))
        xmLOG("%c", (unsigned char) buffer[j]);
    }
    xmLOG("\n");
  }
}


/* Waits for "nsecs" milli-seconds on the socket specifed.  It returns 
   TRUE if there is some data or returns FALSE */
static Boolean 
xmWaitForData(int socket, int msecs)
{
  fd_set read_from;
  struct timeval tv;

  FD_ZERO(&read_from);
  FD_SET(socket, &read_from);

  /* Check if there is more data to be read on this socket */
  tv.tv_sec = msecs/1000;
  tv.tv_usec = (msecs%1000)*1000;

  if (select(socket+1, &read_from, (fd_set *) NULL, (fd_set *) NULL, &tv) < 0)
    xmErrorQuitSYS("select error");
    
  return FD_ISSET(socket, &read_from);
}


static void
xmPutAwaitTable(register XMSession *session, CARD8 req_type, CARD32 seq_num)
{
  register int index = session->num_await;

  /* Table might not be created yet, or table size may have to be increased */
  if (session->num_await == session->await_table_size) {
    session->await = (AwaitReply *) realloc(session->await, 
         sizeof(AwaitReply)*(session->num_await+AWAIT_REPLY_TABLE_SIZE));
    assert(session->await != NULL);
    session->await_table_size += AWAIT_REPLY_TABLE_SIZE;
  }

  /* Fill in information about the packet we are waiting for */
  session->await[index].seq_num = seq_num;
  session->await[index].request_type = req_type;
  session->await[index].num_replied = 0;
  session->num_await++;
  xmDEBUG("AWAIT = %d\n",session->num_await);
}

static void 
xmCheckAwaitingReply(register XMSession *session, register CARD16 seq_num, 
                     int server_index)
{
  register AwaitReply *await = session->await;
  register unsigned int i; /* For iteration */
  register char *offset;
  CARD32 testSeqNum;
  
  /* Where does the current packet start? */
  offset = session->buffer + session->offsets[session->num_packets-1];

  /* We wait for multiple packets only once in a while (maybe once during 
     the start of a session).  There will be only a few (less than 10) 
     requests to search for.  Once a client sends requests, the next 
     packet from the server will usually be the reply.  So, linear search 
     is just fine */

  for(i = 0;i < session->await_table_size;i++) {

  	testSeqNum = await[i].seq_num;

  	if (server_index != MASTER_XSERVER)
      testSeqNum += (CARD32) session->numFakeReqs;

    /* Server sends in only the last 4 hex digits of the sequence number */
    if ((testSeqNum & 0xFFFF) == seq_num) {
      await[i].num_replied++;

			if (await[i].request_type == X_InternAtom) { /* InternAtom */
        session->cache_atoms[server_index] = (Atom) GETCARD32(offset+8);
        xmDEBUG("PUTTING ATOM %d\n", (Atom) GETCARD32(offset+8));
        if (await[i].num_replied == session->num_xservers)
          xmPutAtoms(session);
      }
    
      if (await[i].num_replied == session->num_xservers)
        if (--session->num_await == 0) {
          xmDEBUG("FREEING\n");
          free(session->await);
          session->await = NULL;
          session->await_table_size = 0;
        }
    }
	}
}
