#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <assert.h>
#include <X11/Xproto.h>
#include <X11/X.h>
#include <X11/Xatom.h>

#include "xmdefs.h"
#include "xmevents.h"
#include "xmerror.h"

#define MAX_REQUEST_NUM  128
#define MAX_REQUESTS     MAX_REQUEST_NUM+1
#define MAX_EVENT_NUM    34
#define MAX_EVENTS       MAX_EVENT_NUM+1
#define MAX_ERROR_NUM    17
#define MAX_ERRORS       MAX_ERROR_NUM+1

enum { DIFFERING_BYTE_ORDER, SERVER_REFUSED_CONNECTION };

static void xmMapChangeCreateGC(char *, CARD32, CARD32, CARD32, int);
static void xmMapChangeCreateWindow(char *, CARD32, CARD32, CARD32, int);
static void xmMapConfigureWindow(char *, CARD32, CARD32, CARD32, int);
static CARD32 xmMapRID(CARD32, CARD32, CARD32, CARD32, int);
static void xmGrabServerParams(XServer *, char *, int);
static void xmGetMaskCharacterstics(register CARD32, register int *, 
                                    register int *);
static Atom xmMapAtom(register XServer *, register int);
static void xmMapRotateProperties(XServer *, char *);
static void xmReplyClient(int, int);

/* This holds information about the requests that have resource ids and atoms 
   that need to be mapped and their offsets in the request packet */
   
typedef struct {
  char *request_name;        /* Request */
  unsigned char num_rids;    /* Number of Resource IDs that need mapping */
  unsigned char num_atoms;   /* Number of Atoms that need mapping */
  unsigned char id[3];       /* Resource ID offsets */
  unsigned char atom[3];     /* Atom offsets */
} RID_Atom_Offsets;

static RID_Atom_Offsets rid_atom_offsets[MAX_REQUESTS] = {
 { "Request Code = 0", 0, 0 }, 
 { "X_CreateWindow",  2, 0, { 4, 8, 0 } }, 
 { "X_ChangeWindowAttributes",  1, 0, { 4, 0, 0 } }, 
 { "X_GetWindowAttributes",  1, 0, { 4, 0, 0 } }, 
 { "X_DestroyWindow",  1, 0, { 4, 0, 0 } }, 
 { "X_DestroySubwindows",  1, 0, { 4, 0, 0 } }, 
 { "X_ChangeSaveSet",  1, 0, { 4, 0, 0 } }, 
 { "X_ReparentWindow",  2, 0, { 4, 8, 0 } }, 
 { "X_MapWindow",  1, 0, { 4, 0, 0 } }, 
 { "X_MapSubwindows",  1, 0, { 4, 0, 0 } }, 
 { "X_UnmapWindow",  1, 0, { 4, 0, 0 } }, 
 { "X_UnmapSubwindows",  1, 0, { 4, 0, 0 } }, 
 { "X_ConfigureWindow",  1, 0, { 4, 0, 0 } }, 
 { "X_CirculateWindow",  1, 0, { 4, 0, 0 } }, 
 { "X_GetGeometry", 1, 0, { 4, 0, 0 } }, 
 { "X_QueryTree",  1, 0, { 4, 0, 0 } }, 
 { "X_InternAtom",  0, 0 }, 
 { "X_GetAtomName",  0, 0, { 0, 0, 0 }, { 4, 0, 0 } }, 
 { "X_ChangeProperty",  1, 2, { 4, 0, 0 }, { 8, 12, 0 } }, 
 { "X_DeleteProperty",  1, 1, { 4, 0, 0 }, { 8, 0, 0 } }, 
 { "X_GetProperty",  1, 2, { 4, 0, 0 }, { 8, 12, 0 } }, 
 { "X_ListProperties",  1, 0, { 4, 0, 0 } }, 
 { "X_SetSelectionOwner",  1, 1, { 4, 0, 0 }, { 8, 0, 0 } }, 
 { "X_GetSelectionOwner",  0, 1, { 0, 0, 0 }, { 4, 0, 0 } }, 
 { "X_ConvertSelection",  1, 3, { 4, 0, 0 }, { 8, 12, 16 } }, 
 { "X_SendEvent",  1, 0, { 4, 0, 0 } }, 
 { "X_GrabPointer",  3, 0, { 4, 12, 16 } }, 
 { "X_UngrabPointer",  0 }, 
 { "X_GrabButton",  3, 0, { 4, 12, 16 } }, 
 { "X_UngrabButton",  1, 0, { 4, 0, 0 } }, 
 { "X_ChangeActivePointerGrab",  1, 0, { 4, 0, 0 } }, 
 { "X_GrabKeyboard",  1, 0, { 4, 0, 0 } }, 
 { "X_UngrabKeyboard",  0, 0 }, 
 { "X_GrabKey",  1, 0, { 4, 0, 0 } }, 
 { "X_UngrabKey",  1, 0, { 4, 0, 0 } }, 
 { "X_AllowEvents",  0, 0 }, 
 { "X_GrabServer",  0, 0 }, 
 { "X_UngrabServer",  0, 0 }, 
 { "X_QueryPointer",  1, 0, { 4, 0, 0 } }, 
 { "X_GetMotionEvents",  1, 0, { 4, 0, 0 } }, 
 { "X_TranslateCoords",  2, 0, { 4, 8, 0 } }, 
 { "X_WarpPointer",  2, 0, { 4, 8, 0 } }, 
 { "X_SetInputFocus",  1, 0, { 4, 0, 0 } }, 
 { "X_GetInputFocus",  0, 0 }, 
 { "X_QueryKeymap",  0, 0 }, 
 { "X_OpenFont",  1, 0, { 4, 0, 0 } }, 
 { "X_CloseFont",  1, 0, { 4, 0, 0 } }, 
 { "X_QueryFont",  1, 0, { 4, 0, 0 } }, 
 { "X_QueryTextExtents",  1, 0, { 4, 0, 0 } }, 
 { "X_ListFonts",  0, 0 }, 
 { "X_ListFontsWithInfo",  0, 0 }, 
 { "X_SetFontPath",  0, 0 }, 
 { "X_GetFontPath",  0, 0 }, 
 { "X_CreatePixmap",  2, 0, { 4, 8, 0 } }, 
 { "X_FreePixmap",  1, 0, { 4, 0, 0 } }, 
 { "X_CreateGC",  1, 0, { 4, 0, 0 } }, 
 { "X_ChangeGC",  1, 0, { 4, 0, 0 } }, 
 { "X_CopyGC",  2, 0, { 4, 8, 0 } }, 
 { "X_SetDashes",  1, 0, { 4, 0, 0 } }, 
 { "X_SetClipRectangles",  1, 0, { 4, 0, 0 } }, 
 { "X_FreeGC",  1, 0, { 4, 0, 0 } }, 
 { "X_ClearArea",  1, 0, { 4, 0, 0 } }, 
 { "X_CopyArea",  3, 0, { 4, 8, 12 } }, 
 { "X_CopyPlane", 3, 0, { 4, 8, 12 } }, 
 { "X_PolyPoint",  2, 0, { 4, 8, 0 } }, 
 { "X_PolyLine",  2, 0, { 4, 8, 0 } }, 
 { "X_PolySegment",  2, 0, { 4, 8, 0 } }, 
 { "X_PolyRectangle",  2, 0, { 4, 8, 0 } }, 
 { "X_PolyArc",  2, 0, { 4, 8, 0 } }, 
 { "X_FillPoly",  2, 0, { 4, 8, 0 } }, 
 { "X_PolyFillRectangle",  2, 0, { 4, 8, 0 } }, 
 { "X_PolyFillArc",  2, 0, { 4, 8, 0 } }, 
 { "X_PutImage",  2, 0, { 4, 8, 0 } }, 
 { "X_GetImage",  1, 0, { 4, 0, 0 } }, 
 { "X_PolyText8",  2, 0, { 4, 8, 0 } }, 
 { "X_PolyText16",  2, 0, { 4, 8, 0 } }, 
 { "X_ImageText8",  2, 0, { 4, 8, 0 } }, 
 { "X_ImageText16",  2, 0, { 4, 8, 0 } }, 
 { "X_CreateColormap", 2, 0, { 4, 8, 0 } }, 
 { "X_FreeColormap",  1, 0, { 4, 0, 0 } }, 
 { "X_CopyColormapAndFree",  2, 0, { 4, 8, 0 } }, 
 { "X_InstallColormap",  1, 0, { 4, 0, 0 } }, 
 { "X_UninstallColormap",  1, 0, { 4, 0, 0 } }, 
 { "X_ListInstalledColormaps",  1, 0, { 4, 0, 0 } }, 
 { "X_AllocColor",  1, 0, { 4, 0, 0 } }, 
 { "X_AllocNamedColor",  1, 0, { 4, 0, 0 } }, 
 { "X_AllocColorCells",  1, 0, { 4, 0, 0 } }, 
 { "X_AllocColorPlanes",  1, 0, { 4, 0, 0 } }, 
 { "X_FreeColors",  1, 0, { 4, 0, 0 } }, 
 { "X_StoreColors",  1, 0, { 4, 0, 0 } }, 
 { "X_StoreNamedColor",  1, 0, { 4, 0, 0 } }, 
 { "X_QueryColors",  1, 0, { 4, 0, 0 } }, 
 { "X_LookupColor",  1, 0, { 4, 0, 0 } }, 
 { "X_CreateCursor",  3, 0, { 4, 8, 12 } }, 
 { "X_CreateGlyphCursor",  3, 0, { 4, 8, 12 } }, 
 { "X_FreeCursor",  1, 0, { 4, 0, 0 } }, 
 { "X_RecolorCursor",  1, 0, { 4, 0, 0 } }, 
 { "X_QueryBestSize",  1, 0, { 4, 0, 0 } }, 
 { "X_QueryExtension",  0, 0 }, 
 { "X_ListExtensions",  0, 0 }, 
 { "X_ChangeKeyboardMapping",  0, 0 }, 
 { "X_GetKeyboardMapping",  0, 0 }, 
 { "X_ChangeKeyboardControl",  0, 0 }, 
 { "X_GetKeyboardControl",  0, 0 }, 
 { "X_Bell",  0, 0 }, 
 { "X_ChangePointerControl",  0, 0 }, 
 { "X_GetPointerControl",  0, 0 }, 
 { "X_SetScreenSaver",  0, 0 }, 
 { "X_GetScreenSaver",  0, 0 }, 
 { "X_ChangeHosts",  0, 0 }, 
 { "X_ListHosts",  0, 0 }, 
 { "X_SetAccessControl",  0, 0 }, 
 { "X_SetCloseDownMode",  0, 0 }, 
 { "X_KillClient",  1, 0, { 4, 0, 0 } }, 
 { "X_RotateProperties",  1, 0, { 4, 0, 0 } },  /* has a list of atoms to map */
 { "X_ForceScreenSaver",  0, 0 }, 
 { "X_SetPointerMapping",  0, 0 }, 
 { "X_GetPointerMapping",  0, 0 }, 
 { "X_SetModifierMapping",  0, 0 }, 
 { "X_GetModifierMapping",  0, 0 }, 
 { "Request Code = 120",  0 }, 
 { "Request Code = 121",  0 }, 
 { "Request Code = 122",  0 }, 
 { "Request Code = 123",  0 }, 
 { "Request Code = 124",  0 }, 
 { "Request Code = 125",  0 }, 
 { "Request Code = 126",  0 }, 
 { "X_NoOperation",  0, 0 },
 /* Extended request 128 has resource ids at offsets 8 and 16 */
 { "Extended Code = 128", 2, 0, { 8, 16 } }
};

static char *event_names[MAX_EVENTS] = {
 "Error",
 "Reply",
 "KeyPress", 
 "KeyRelease", 
 "ButtonPress", 
 "ButtonRelease", 
 "MotionNotify", 
 "EnterNotify", 
 "LeaveNotify", 
 "FocusIn", 
 "FocusOut", 
 "KeymapNotify", 
 "Expose", 
 "GraphicsExposure", 
 "NoExpose", 
 "VisibilityNotify", 
 "CreateNotify", 
 "DestroyNotify", 
 "UnmapNotify", 
 "MapNotify", 
 "MapRequest", 
 "ReparentNotify", 
 "ConfigureNotify", 
 "ConfigureRequest", 
 "GravityNotify", 
 "ResizeRequest", 
 "CirculateNotify", 
 "CirculateRequest", 
 "PropertyNotify", 
 "SelectionClear", 
 "SelectionRequest", 
 "SelectionNotify", 
 "ColormapNotify", 
 "ClientMessage", 
 "MappingNotify"
};

static char *error_names[MAX_ERRORS] = {
"Invalid error code",
"Invalid Request",
"Invalid Value",
"Undefined Window",
"Undefined Pixmap",
"Undefined Atom",
"Undefined Cursor",
"Undefined Font",
"Drawing into an Input Window",
"Undefined Drawable",
"Access Error",
"Alloc Error",
"Undefined Colormap",
"Undefined GCContext",
"Invalid ID Choice",
"Name does not exist",
"Invalid Length",
"Implementation Error"
};

static void 
xmReplyClient(int socket, int whatReason)
{
	#define SWAP(a, b, temp) { temp = *(a); *(a) = *(b); *(b) = temp; }
  char *buffer;
  register int replysize;
  xConnSetupPrefix reply;
	char *reason;
  char *byteOrder = "Byte order of X client differs from that of multiplexer!
      This feature not yet supported.";
  char *notAuthorized = "Server refused connection request";
  char temp;  /* For swapping to ensure correct byte order */

	reason = (whatReason == DIFFERING_BYTE_ORDER)? byteOrder:notAuthorized;

  reply.success = 0; /* Connection request failed */
  reply.lengthReason = strlen(reason);
  reply.majorVersion = X_PROTOCOL;
	SWAP((char *) &reply.majorVersion, ((char *) &reply.majorVersion) + 1, temp);

  reply.minorVersion = X_PROTOCOL_REVISION;
  reply.length = ALIGN4(reply.lengthReason) >> 2; 

	if (whatReason == DIFFERING_BYTE_ORDER)
  	SWAP((char *) &reply.length, ((char *) &reply.length) + 1, temp);

  replysize = reply.lengthReason + sizeof(xConnSetupPrefix);
  replysize = ALIGN4(replysize); 

  buffer = (char *)malloc(replysize);
  assert(buffer != NULL);

  memcpy((char *) buffer, (char *) &reply, sizeof(xConnSetupPrefix));
  memcpy(buffer+sizeof(xConnSetupPrefix), reason, reply.lengthReason);

  write(socket, buffer, replysize);

  free(buffer); 
}

int
xmCompleteConnection(XMSession *session)
{
  register int socket;
  xConnSetupPrefix *conn_prefix;
  fd_set readfrom;
  struct timeval tv;
  int num_xservers;
  int num_replied;  /* Number of X servers that have responded */
  int num_ready;
  int i;

  /* Since an X client has already connected, read from the socket
     and send it to all the X servers */
  if ((session->curr_buffer_len = read(session->xclient->socketd, 
                                       session->buffer, BUFSIZE)) <= 0) {
    xmerror = CLIENTSOCKETREADERROR;
    xmErrorQuitSYS("xmCompleteConnection() - error on client socket");
  }

  /* Is the byte order of the X client the same as that of the
     multiplexer? */
  if (session->buffer[0] != xmByteOrder) {
		xmReplyClient(session->xclient->socketd, DIFFERING_BYTE_ORDER);
		return ERROR;
	}

  /* Send connection initiation information to all the X servers */
  xmMultiCast(session, VERBATIM);

  /* Yet to hear from any of the X servers */
  num_replied = 0;
  num_xservers = session->num_xservers;

  FD_ZERO(&readfrom);

  while(num_replied < num_xservers) {
    int maxSocketDesc = xmFDInit(session, &readfrom, XSERVER);

    /* Wait at least 30 seconds to hear from the X servers */
    tv.tv_sec = 30;
    tv.tv_usec = 0;

    if ((num_ready = select(maxSocketDesc+1, &readfrom, (fd_set *) NULL, 
														(fd_set *) NULL, (struct timeval *) &tv)) < 0) {
      xmerror = SELECTERROR;
      xmErrorQuitSYS("xmCompleteConnection() - select error\n");
    }
    
    for(i = 0;i < MAX_CONNECTIONS; i++) {
      if (session->xservers[i] == NULL)
        continue;

      socket = session->xservers[i]->socketd;

      /* Does the server with this socket id have data? */
      if (FD_ISSET(socket, &readfrom)) {
        num_replied++;
        if ((session->curr_buffer_len = read(socket, session->buffer, 
                                             BUFSIZE)) <= 0) {
          xmerror = SERVERSOCKETREADERROR;
          xmErrorNoQuitSYS("Error reading socket\n");
          if (i == MASTER_XSERVER)
            return ERROR;
          else {
            xmCloseServerConnection(session, i);
            continue;
          }
        }
        conn_prefix = (xConnSetupPrefix *) session->buffer;

        /* Check to see if connection to the server was successful.  If yes, 
           we need to send data back to the X client only if this X server
           is the master X server.  */

        if (conn_prefix->success == xFalse) {
					printf("Connection refused\n");
          xmCloseServerConnection(session, i);
          if (i == MASTER_XSERVER) {
            xmErrorNoQuit("Connection to master server refused\n");
		        xmReplyClient(session->xclient->socketd, SERVER_REFUSED_CONNECTION);
            return ERROR;
          }
        }
        else {
					printf("Connection successful\n");
          /* Connection successful */
          xmGrabServerParams(session->xservers[i], session->buffer, 
                             session->curr_buffer_len);

          /* Write data to the client only if the data comes from the master 
             X server */
          if (i == MASTER_XSERVER) 
            write(session->xclient->socketd, session->buffer, 
                  session->curr_buffer_len);
         }
      }
      else if (num_ready == 0) { 
								/* Waited for at least 30 seconds with no response */
        xmCloseServerConnection(session, i);
        xmErrorNoQuit("Connection to %s timed out\n", 
                         session->xservers[i]->xserver_name);
        if (i == MASTER_XSERVER)
          return ERROR;
      }

    } /* for */
  } /* while */

	return 0;
}

static void 
xmGrabServerParams(XServer *xserver, char *buffer, int bufsize)
{
  int num_pixmap_formats;
  int num_roots;
  int vendor_string_size;
  char *tmp_buffer;
  int num_trailing_zeroes, num_ones;  /* in Resource ID mask */
  int i, j; /* For iteration */

  tmp_buffer = (char *)malloc(bufsize);
  assert(tmp_buffer != NULL);

  /* Copy buffer to a temporary place.  We will stick pointers to this
     buffer in the structure */
  memcpy(tmp_buffer, buffer, bufsize);

  /* Connection prefix information */

  xserver->serv_info = (ServerParams *) malloc(sizeof(ServerParams));
  xserver->serv_info->prefix = (xConnSetupPrefix *) tmp_buffer;
  tmp_buffer += sizeof(xConnSetupPrefix);
  bufsize -= sizeof(xConnSetupPrefix);

  /* Connection setup information */

  xserver->serv_info->setup = (xConnSetup *) tmp_buffer;
  xserver->max_req_len = 
                 (unsigned int) xserver->serv_info->setup->maxRequestSize << 2;

  xmDEBUG("Mask = %8X  Base = %8X\n", ((xConnSetup *) tmp_buffer)->ridMask,  
                                      ((xConnSetup *) tmp_buffer)->ridBase);
  xmGetMaskCharacterstics(((xConnSetup *) tmp_buffer)->ridMask, 
                          &num_trailing_zeroes, &num_ones);

  xserver->shift = num_trailing_zeroes;

  /* Make a note of the number of PIXMAP formats, roots, and the number of 
     bytes in the vendor string.  We are not interested in the vendor string 
     though */

  num_pixmap_formats = xserver->serv_info->setup->numFormats;
  num_roots = xserver->serv_info->setup->numRoots;
  vendor_string_size = xserver->serv_info->setup->nbytesVendor;

  /* There will be padding depending upon the number of bytes in the vendor 
     string.  Step over it. */

  tmp_buffer += sizeof(xConnSetup) + ALIGN4(vendor_string_size);
  bufsize -= sizeof(xConnSetup) + ALIGN4(vendor_string_size);

  /* List of PIXMAP formats */

  xserver->serv_info->pixmap_formats = (xPixmapFormat *) tmp_buffer;
  tmp_buffer += sizeof(xPixmapFormat)*num_pixmap_formats;
  bufsize += sizeof(xPixmapFormat)*num_pixmap_formats;

  /* There can be more than one root.  Allocate enough space for an array
     of pointers depending on the number of roots.  */

  xserver->serv_info->list_xwindow_root = 
								(ListxWindowRoot *)malloc(sizeof(ListxWindowRoot)*num_roots);
  assert(xserver->serv_info->list_xwindow_root != NULL);

  for(i = 0; i < num_roots; i++) {
    /* Have a pointer to the array element we are interested in.  
       Makes code more easy to read */
    ListxWindowRoot *tmp_wnd = &xserver->serv_info->list_xwindow_root[i];
    int num_depths;

    /* Pointer to a ROOT */
    tmp_wnd->window_root = (xWindowRoot *) tmp_buffer;

    /* Make a note of the number of depths allowed in this root */
    num_depths = tmp_wnd->window_root->nDepths;
    tmp_buffer += sizeof(xWindowRoot);
    bufsize -= sizeof(xWindowRoot);
    
    /* We can have a list of depths.  Have enough space to store pointers to 
       these lists */
    tmp_wnd->list_xdepth = (ListxDepth *)malloc(sizeof(ListxDepth)*num_depths);

    for(j = 0; j < num_depths; j++) {
      /* For easier reading */
      ListxDepth *tmp_depth = &tmp_wnd->list_xdepth[j];
      int num_visuals;

      /* Note the address of this Depth description */
      tmp_depth->depth = (xDepth *) tmp_buffer;
      tmp_buffer += sizeof(xDepth);

      /* Number of visuals */
      num_visuals = tmp_depth->depth->nVisuals;

      /* Pointer to this list of visuals */
      tmp_depth->list_visual_type = (xVisualType *)tmp_buffer;
      tmp_buffer += sizeof(xVisualType)*num_visuals;
    }
  }

}

static void 
xmGetMaskCharacterstics(register CARD32 ridMask, register int *trailing_zeroes, 
                        register int *num_ones)
{
  *trailing_zeroes = 0;
  *num_ones = 0;
  
  while((ridMask & 0x01) == 0) {
    (*trailing_zeroes)++;
    ridMask >>= 1;
  }
    
  while(ridMask & 0x01) {
    (*num_ones)++;
    ridMask >>= 1;
  }
}

void 
xmAnalyze(register XMSession *session, int serv_or_cli, int serv_id)
{
  int i;
  register unsigned int offset;
  register unsigned char index;

  xmDEBUG("NUMBER OF PACKETS = %d\n", session->num_packets);

  for(i = 0; i < session->num_packets; i++) {
    offset = session->offsets[i];
    index = (unsigned char) session->buffer[offset];
    if (serv_or_cli & XCLIENT) {

      /* This must be done in the ReadClient function */
      session->xclient->sequence_num++;

      xmDEBUG("== Sequence Number = %d\n", session->xclient->sequence_num);
      if (index > 128)
        xmDEBUG("Request: Extended index %u\n", index);
      else {
        xmLOG("Offset = %d\n", offset);
        xmDEBUG("Request: %s \n", rid_atom_offsets[index].request_name);
      }
    }
    else { /* X server */
      if (session->buffer[offset] == XSERVER_ERROR) {

        if ((unsigned char) session->buffer[offset+1] < MAX_ERRORS) {
          xmDEBUG("ERROR from Server %d: index = %d %s ID = %X Seq = %d\n", 
                   serv_id, (int)(unsigned char) session->buffer[offset+1], 
                   error_names[(int)(unsigned char) session->buffer[offset+1]], 
                   GETCARD32(&session->buffer[offset+4]), 
                   GETCARD16(&session->buffer[offset+2]));
          xmDEBUG("Minor opcode = %d Major opcode = %d\n", 
                          (int) GETCARD16(&session->buffer[offset+8]),
                          (int) session->buffer[offset+10]);
        }
        else {
          xmDEBUG("ERROR from Server %d: Invalid error code = %d\n", serv_id, 
                   (int) (unsigned char) session->buffer[offset+1]);
          printf("ERROR from Server %d: Invalid error code = %d\n", serv_id, 
                   (int) (unsigned char) session->buffer[offset+1]);
        }
      }
      else if (index >= MAX_EVENTS) 
        xmDEBUG("Event: %d Unknown Event\n", index);
			else
        xmDEBUG("Event: %d  %s\n", index, event_names[index]);
    }
  }
}

void xmMakeConfWinPkt(register XMSession *session)
{
  char *tmpbuf;
	char *eventBuf;
	int size;
	int offset;

	tmpbuf = session->buffer;
	eventBuf = session->ResizeReqBuf;
	assert(eventBuf != NULL);

  tmpbuf[0] = X_ConfigureWindow;

	/* Copy resource id of WINDOW */
  *((CARD32 *) (tmpbuf+4)) = GETCARD32(eventBuf+4);

	/* Determine BITMASK */
  *((CARD16 *) (tmpbuf+8)) = (CARD16) (0x4 | 0x8 | 0x10);

	size = 6;
	offset = 12;

	/* Is the x coordinate non-zero? */
	if (GETCARD16(eventBuf+16)) {
	  *((CARD32 *) (tmpbuf+offset)) = (CARD32) GETCARD16(eventBuf+16);
  	*((CARD16 *) (tmpbuf+8)) |= 0x1;
		size++;
		offset += 4;
  }

	/* Is the y coordinate non-zero? */
	if (GETCARD16(eventBuf+18)) {
	  *((CARD32 *) (tmpbuf+offset)) = (CARD32) GETCARD16(eventBuf+18);
  	*((CARD16 *) (tmpbuf+8)) |= 0x2;
		size++;
		offset += 4;
  }

  *((CARD32 *) (tmpbuf+offset)) = (CARD32) GETCARD16(eventBuf+20);
  *((CARD32 *) (tmpbuf+offset+4)) = (CARD32) GETCARD16(eventBuf+22);
  *((CARD32 *) (tmpbuf+offset+8)) = (CARD32) GETCARD16(eventBuf+24);

	/* size */
  *((CARD16 *) (tmpbuf+2)) = (CARD16) size;

	session->curr_buffer_len = (size << 2);
	session->num_packets = 1;
	session->offsets[0] = 0;
}


EventList *
xmMapResIDsAtoms(register XMSession *session, int curr_serv_id, 
                 int prev_serv_id, int serv_or_cli, 
                 Atom *atomsIndexList, int atomsListSize)
{
  int i;  /* For iteration */
  register unsigned char index;
  CARD32 prev_res_mask, prev_res_base;
  CARD32 curr_res_base;
  unsigned char shift_prev;
  short int shift;
  char *tmp_buf; /* Temporary pointer into the buffer */
  EventList *eventList = NULL; /* To return a list of expose events */
  int atomsIndex = 0;

  if (prev_serv_id == FIRST_SERVER)
    prev_serv_id = MASTER_XSERVER;

  prev_res_mask = session->xservers[prev_serv_id]->serv_info->setup->ridMask;
  prev_res_base = session->xservers[prev_serv_id]->serv_info->setup->ridBase;
  curr_res_base = session->xservers[curr_serv_id]->serv_info->setup->ridBase;

  shift_prev = session->xservers[prev_serv_id]->shift;
  shift = shift_prev - session->xservers[curr_serv_id]->shift;
  
  if (serv_or_cli & XCLIENT) {

    for(i = 0; i < session->num_packets; i++) {
      int j; /* For iteration */
      char *offset;

      tmp_buf = session->buffer + session->offsets[i];
      index = (unsigned char) *tmp_buf;

      if (index > MAX_REQUEST_NUM)
        continue;

      switch(index) {

        case X_CreateGC:
        case X_ChangeGC:
          xmMapChangeCreateGC(tmp_buf, prev_res_base, prev_res_mask, 
                              curr_res_base, shift);
          continue;

        case X_ChangeWindowAttributes:
        case X_CreateWindow:
          xmMapChangeCreateWindow(tmp_buf, prev_res_base, prev_res_mask, 
                                  curr_res_base, shift);
          continue;

        case X_ConfigureWindow:
          xmMapConfigureWindow(tmp_buf, prev_res_base, prev_res_mask, 
                               curr_res_base, shift);
          continue;
      
        case X_RotateProperties:
          xmMapRotateProperties(session->xservers[curr_serv_id], tmp_buf);
          continue;
      }

      for(j = 0; j < rid_atom_offsets[index].num_rids; j++) {
        offset = tmp_buf+rid_atom_offsets[index].id[j];
        *((CARD32 *) offset) = xmMapRID(GETCARD32(offset), prev_res_base, 
                                        prev_res_mask, curr_res_base, shift);
      }

      for(j = 0; j < rid_atom_offsets[index].num_atoms; j++) {
        offset = tmp_buf+rid_atom_offsets[index].atom[j];
        assert(atomsIndex <= atomsListSize);
        *((Atom *) offset) = xmMapAtom(session->xservers[curr_serv_id], 
                                       atomsIndexList[atomsIndex++]);
      }
    }
  }
  else {  /* From a slave server */
    eventList = (EventList *)malloc(sizeof(EventList));
    assert(eventList != NULL);
    eventList->buffer = NULL;
    eventList->length = 0;

    for(i = 0; i < session->num_packets; i++) {
      tmp_buf = session->buffer + session->offsets[i];
      index = (unsigned char) *tmp_buf;

      if (index <= XSERVER_REPLY || index > MAX_EVENT_NUM)
        continue;
      
      if (index == Ev_Expose || index == Ev_GraphicsExposure) {
        eventList->buffer = (char *)realloc(eventList->buffer, 
                                    eventList->length+SERVER_HEADER_SIZE);
        assert(eventList->buffer != NULL);
        *((CARD32 *) (tmp_buf+4)) = xmMapRID(GETCARD32(tmp_buf+4), 
                                             prev_res_base, prev_res_mask,
                                             curr_res_base, shift);
        *((CARD16 *) (tmp_buf+SEQ_NUM_OFFSET)) = 
                      (CARD16) (session->seq_num_last_processed);
        memcpy(eventList->buffer+eventList->length, tmp_buf,
               SERVER_HEADER_SIZE);
        eventList->length += SERVER_HEADER_SIZE;
      }
    }
  }

  return eventList;
}

static void 
xmMapChangeCreateGC(char *request, CARD32 prev_res_base, CARD32 prev_res_mask, 
                    CARD32 curr_res_base, int shift)
{
  unsigned short int num_values;
  CARD32 mask;
  CARD32 bitmask;
  int i; /* For iteration*/

  /* Number of values in the list is at offset 2 of the request packet.  This 
     is 3 (if ChangeGC)  or 4 (if CreateGC) plus the number of values.  We take 
     care of this later.  */
  num_values = *((unsigned short int *) (request+2));

  /* Fix GCCONTEXT. It is present for both requests. */
  *((CARD32 *) (request+4)) = xmMapRID(GETCARD32(request+4), prev_res_base, 
                                       prev_res_mask, curr_res_base, shift);

  /* CreateGC request also has DRAWABLE to be fixed */
  if (*request == X_CreateGC) {
    *((CARD32 *) (request+8)) = xmMapRID(GETCARD32(request+8), prev_res_base, 
                                         prev_res_mask, curr_res_base, shift);
    num_values -= 4;
    request += 12;
  }
  else {
    num_values -= 3;
    request += 8;
  }

  bitmask = GETCARD32(request);
  request += 4;
  
  /* Return if we have no PIXMAPs or FONT to fix */
  if ((bitmask & (0x400 | 0x800 | 0x4000 | 0x80000)) == 0)
    return;

  /* We cycle through the list of values and fix PIXMAPs */

  mask = 0x1;
  for(i = 0; i < 32 && num_values; i++) {
    /* Do we have a value? */
    if (bitmask & mask) {
      /* Yes.  Fix it if it is a PIXMAP */
      if (mask & (0x400 | 0x800 | 0x4000 | 0x80000)) 
        *((CARD32 *) request) = xmMapRID(GETCARD32(request), prev_res_base, 
                                         prev_res_mask, curr_res_base, shift);

      /* Step over this */
      request += 4;
      num_values--;
    }
    mask <<= 1;
  }
}


static void 
xmMapChangeCreateWindow(char *request, CARD32 prev_res_base, 
                        CARD32 prev_res_mask, CARD32 curr_res_base, int shift)
{
  unsigned short int num_values;
  CARD32 mask;
  CARD32 bitmask;
  int i; /* For iteration*/

  /* First fix WINDOW id at offset 4 */
  *((CARD32 *) (request+4)) = xmMapRID(GETCARD32(request+4), prev_res_base,
                                       prev_res_mask, curr_res_base, shift);

  /* Get BITMASK */
  if (*request == X_CreateWindow) {
    /* CreateWindow has another id at offset 8 to be fixed */
    *((CARD32 *) (request+8)) = xmMapRID(GETCARD32(request+8), prev_res_base, 
                                         prev_res_mask, curr_res_base, shift);

    num_values = *((unsigned short int *) (request+2)) - 8;
    bitmask = GETCARD32(request+28);
    request += 32;
  }
  else { /* X_ChangeWindowAttributes */
    num_values = *((unsigned short int *) (request+2)) - 3;
    bitmask = GETCARD32(request+8);
    request += 12;
  }

  /* Return if we have no PIXMAPs, COLORMAPs, or CURSORs to fix */
  if ((bitmask & (0x1 | 0x4 | 0x2000 | 0x4000)) == 0)
    return;

  /* Step over to beginning of values list */

  mask = 0x1;
  for(i = 0; i < 32 && num_values; i++) {
    /* Do we have a value? */
    if (bitmask & mask) {
      /* Yes.  Fix it if it is a resource id */
      if (mask & (0x1 | 0x4 | 0x2000 | 0x4000))
        *((CARD32 *) request) = xmMapRID(GETCARD32(request), prev_res_base, 
                                         prev_res_mask, curr_res_base, shift);

      /* Step over this */
      request += 4;
      num_values--;
    }
    mask <<= 1;
  }
}

static void 
xmMapConfigureWindow(char *request, CARD32 prev_res_base, CARD32 prev_res_mask, 
                     CARD32 curr_res_base, int shift)
{
  unsigned short int num_values;
  CARD32 mask;
  CARD16 bitmask;
  int i; /* For iteration*/

  /* Number of values in the list */
  num_values = *((unsigned short int *) (request+2)) - 3;

  /* First fix WINDOW ids at offset 4 */
  *((CARD32 *) (request+4)) = xmMapRID(GETCARD32(request+4), prev_res_base, 
                                       prev_res_mask, curr_res_base, shift);

  /* Get BITMASK at offset 8 */
  bitmask = GETCARD16(request+8);

  /* Return if we have no WINDOW to fix */
  if ((bitmask & 0x20) == 0)
    return;

  /* Step over to beginning of values list */
  request += 12;

  mask = 0x01;
  for(i = 0; i < 16 && num_values; i++) {
    /* Do we have a value? */
    if (bitmask & mask) {
      /* Yes.  Fix it if it is a WINDOW */
      if (mask & 0x20) 
        *((CARD32 *) request) = xmMapRID(GETCARD32(request), prev_res_base, 
                                         prev_res_mask, curr_res_base, shift);

      /* Step over this */
      request += 4;
      num_values--;
    }
    mask <<= 1;
  }
}


static CARD32 
xmMapRID(CARD32 res_id, CARD32 prev_res_base, CARD32 prev_res_mask, 
         CARD32 curr_res_base, int shift)
{
  if ((res_id & ~prev_res_mask) == prev_res_base) {
    res_id &= prev_res_mask;
    res_id = ((shift < 0)? (res_id << shift):(res_id >> shift))|curr_res_base;
  }

  return res_id;
}

static void
xmMapRotateProperties(XServer *server, char *tmp_buf)
{
}


void
xmPutAtoms(register XMSession *session)
{
  int i; /* For iteration */
  register XServer *server;
  register Atom atom;
  register Atom index; /* index at which atom is stored in the atoms table
                           of the master X server */

  /* We never cache pre-defined atoms.  So if we are installing atoms
     in the table, they will be atoms which are not pre-defined */

  atom = session->cache_atoms[MASTER_XSERVER];

  if (atom <= XA_LAST_PREDEFINED)
    return;

  if (atom > 512) {
    xmLOG("Unusually large Atom id %u\n", atom);
    xmErrorNoQuit("Unusually large Atom id %u\n", (unsigned int) atom);
    return;
  }

  server = session->xservers[MASTER_XSERVER];
  index = atom - XA_LAST_PREDEFINED;

  
  /* If the atoms table is not big enough, increase the size of the table
     until this atom can be stored */

  while(index > server->size_atoms) {
    server->atoms = realloc(server->atoms, 
									sizeof(Atom)*(server->size_atoms+ATOMS_TABLE_SIZE));
    assert(server->atoms != NULL);
    server->size_atoms += ATOMS_TABLE_SIZE;
  }
  
  /* Put this atom in the server's table */
  server->atoms[index] = atom;

  for(i = 0;i < MAX_CONNECTIONS;i++) 
    if (session->xservers[i] != NULL && i != MASTER_XSERVER) {
      server = session->xservers[i];

      /* If the table is not big enough to hold this index,
                           increase the size of the table */
      while(index > server->size_atoms) {
        server->atoms = realloc(server->atoms, 
                        sizeof(Atom)*(server->size_atoms + ATOMS_TABLE_SIZE));
        assert(server->atoms != NULL);
        server->size_atoms += ATOMS_TABLE_SIZE;
      }
      server->atoms[index] = session->cache_atoms[i];
    }
}

static Atom
xmMapAtom(register XServer *server, register int index)
{
	Atom tmp;

  if (index < 0) 
    tmp = index + XA_LAST_PREDEFINED;
  else 
    tmp = server->atoms[index];

	printf("Mapped to %d\n", tmp);
	return tmp;
}

void
xmGetAtomsList(XMSession *session, Atom **atomsIndexList, int *atomsListSize)
{
  #define ATOMS_LIST_BLOCK_SIZE 64
  char *tmp_buf;
  int currentListSize = 0;
  unsigned char index;
  int i; /* For iteration */

  *atomsIndexList = NULL;
  *atomsListSize = 0;

  for(i = 0; i < session->num_packets; i++) {
    int j; /* For iteration */
    char *offset;

    tmp_buf = session->buffer + session->offsets[i];
    index = (unsigned char) *tmp_buf;

    if (index > MAX_REQUEST_NUM)
      continue;

    for(j = 0; j < rid_atom_offsets[index].num_atoms; j++) {
      offset = tmp_buf + rid_atom_offsets[index].atom[j];
      if (currentListSize <= *atomsListSize) {
				printf("Reallocing\n");
        *atomsIndexList = (Atom *)realloc(*atomsIndexList, 
                     sizeof(Atom)*(currentListSize + ATOMS_LIST_BLOCK_SIZE));
        currentListSize += ATOMS_LIST_BLOCK_SIZE;
      }
      (*atomsIndexList)[*atomsListSize] = GETCARD32(offset) 
                                                         - XA_LAST_PREDEFINED;
      (*atomsListSize)++;
    }
  }
}

