#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <assert.h>
#include <ctype.h>

#ifdef SunOS
#include <sys/types.h> 
#include <sys/ddi.h>
#endif

#include "xmdefs.h"
#include "xmerror.h"

int xmlog;         /* Do we need to produce a log file */
int xmdebug;       /* To see if debug "-d" flag was specified */
int xmerror;       /* For transmitting error values */
int xserver_port;  /* For X server port on the machine on which xm runs */
XServer *host_xserver; /* Description of X server running on xm'x host */
char *buffer;   /* Buffer to hold packets */
char  xmByteOrder;  /* Little Endian or Big Endian byte order? */

FILE *fp;

static char **xmParseCommandLine(int, char **);
static Boolean xmValidNum(char *);
static void xmDetermineByteOrder();

/* SIGINT, SIGHUP signal handlers */
void 
siginthup_handler()
{
}


void 
main(int argc, char **argv)
{
  char **xservers_list;

  if (argc < 2) {
    fprintf(stderr, "Usage: xm [-l] [-d] X server(s) list\n");
    exit(1);
  }

  /* Install handlers for signals SIGPIPE, SIGINT, SIGHUP.  SIGPIPE signal is 
     given to the program if a socket can no longer be written to.  We handle 
     this elsewhere.  So, ignore this signal. */
  signal(SIGPIPE, SIG_IGN);
  signal(SIGINT,  siginthup_handler);
  signal(SIGHUP,  siginthup_handler);

  /* Initialize globals */
  xmlog = 0;
  xmdebug = 0;
  xserver_port = 0;
  xmerror = NOERROR;
  host_xserver = NULL;

  /* Connect to X server running on xm's host.  We need this connection 
     because we need to authenticate X clients that want to connect to us 
  xmConnectToHostXServer(); */

	xmDetermineByteOrder();

  /* Parse command line and get a list of X servers specified */
  xservers_list = xmParseCommandLine(argc, argv);

  /* Open log file */
	if (xmlog)
    fp = fopen("log","w");

  /* No clients have connected to us yet.  We have to open a socket on which 
     clients that want to connect will "sound" us. And listen on that socket 
     until a client sounds us */
  xmMultiplexer(xservers_list);

  fclose(fp);

}

/* Parses command line and returns a list of "potential" servers specified
   on the command line. */
static char **
xmParseCommandLine(int argcount, char **arglist)
{
  char **xservers_list; 
  char **temp;
  int i; /* For iteration */

  /* Ignore program name */
  ++arglist;

  /* All but one could be an X server */
  temp = xservers_list = (char **)malloc((argcount)*sizeof(char *));
  assert(temp != NULL);

  for(i = 0; i < argcount; i++)
    xservers_list[i] = (char *) NULL;

  /* Parse options */
  while(*arglist) {
    if (**arglist == '-') {
      /* X server port number */
      if (strcmp(*arglist, "-p") == 0) {
        if (xmValidNum(*++arglist))
          xserver_port = atoi(*arglist);
        else 
          xmErrorNoQuit("No valid port number specified. Assuming port = 0\n");
      }
      /* Print debug information */
      else if (strcmp(*arglist, "-d") == 0) 
        xmdebug = 1;
      /* Maintain log of data transferred */
      else if (strcmp(*arglist, "-l") == 0) 
        xmlog = 1;
      else
        xmErrorQuit("Illegal option %s\n", *arglist);
      arglist++;
    }
    else
      *xservers_list++ = *arglist++;
  }

  return temp;
}
          

/* Parses parameter to the X multiplexer to find out the name of the host 
   and the port number to connect to.  Returns TRUE on finding a valid 
   host:port combination otherwise returns FALSE.  */

Boolean 
xmGetServerPort(char *string, char *machname, int *port)
{
  char strport[10];
  int i;

  if (string == NULL)
    return FALSE;

  /* Find out the machine name */
  for(; *string && *string != ':' ; machname++, string++)
    *machname = *string;
  *machname = '\0'; /* Terminate machine name */

  /* Find out the port number specified for a host.  Parameters will be the 
     form host:port.display.  We are not concerned about the display */
  if (*string)  {
    for(i = 0, string++; (strport[i] = *string) != '.' && *string; 
        string++, i++)
      	;

    strport[i] = '\0';
    *port = atoi(strport);
    return TRUE;  /* Found a valid host name, port number pair */
  }
  xmerror = NOPORTNUMBER;
  xmErrorNoQuit("No port number\n");
  return FALSE; /* Port number was not specified */
}

static Boolean 
xmValidNum(char *str)
{
  if (str == NULL)
    return FALSE;

  while(*str)
    if (!isdigit(*str++))
      return FALSE;

  return TRUE;
}

static void
xmDetermineByteOrder()
{
	unsigned int num = 0x1;

	if (*((char *) &num) == 0x1)
		xmByteOrder = 'l';  /* X code for Little Endian byte order.
		                       Least significant byte first */
	else
		xmByteOrder = 'B';  /* X code for Big Endian byte order.
		                       Most significant byte first */
}

void 
xmExit(XMSession *session)
{  
  fclose(fp);
  xmCloseAll(session);
  free(session->buffer);
	exit(1);
}

