import pat.Pattern;
import pat.Regex;
import pat.patInt;
import pat.Pthings;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.text.SimpleDateFormat;

/* tokenTest.java by Steven R. Brandt

   This software comes without express or implied warranty.
   No claim is made about the suitability of this software for
   any purpose and neither I nor SteveSoft shall be liable for
   damages suffered by the user of this software.
   */

// Regex allows the user to extend its
// pattern matching abilities and syntax.
//
// To add a new pattern, extend class pattern and
// fill in the abstract functions matchInternal(int pos,Pthings pt)
// and toString().
//
// match(int,Pthings) is the string that we
// are attempting to match, pos is the position within
// the string we are attempting to match at.  If
// successful, the routine should exit with a call to
// nextMatch(pos+n,pt) where n is the number of characters
// that this pattern has matched.
//
// Pthings -- "Pattern things" contains the string we are operating
// on, in a variable called src, as well as ignoreCase, and
// dontMatchInQuotes.

// Our example matches a date of the form #-#-#, but only if that
// could really be a date (Feb 30 is not valid, etc.)
class DatePattern extends pat.Pattern {
	static Regex date = new Regex("(\\d+)-(\\d+)-(\\d+)");

	// This next section is more of an optimization than
	// anything else, it allows us to fail more quickly
	// on patterns that are too short.  Everything still
	// works without them.
	static patInt minDateLen = date.countMinChars();
	static patInt maxDateLen = date.countMaxChars();
	public patInt minChars() { return minDateLen; }
	public patInt maxChars() { return maxDateLen; }

	public int matchInternal(int pos,Pthings pt) {
		// we can't match with fewer than datelen characters
		if(pos+minChars().intValue() >= pt.src.length()) return -1;
		// make sure we don't have a number in the preceeding
		// character, 21-22-1991 is not right.
		if(pos > 0) {
			char d = pt.src.charAt(pos-1);
			if(d >= '0' && d <= '9')
				return -1;
		}
		if(date.matchAt(pt.src,pos)) try {
			int mo = (new Integer(date.substring(0))).intValue();
			int da = (new Integer(date.substring(1))).intValue();
			int yr = (new Integer(date.substring(2))).intValue();
			if(yr > 1900) yr -= 1900;
			GregorianCalendar dt=new GregorianCalendar();
			dt.setValidationMode(true);
			dt.set(yr,mo-1,da-1);
			dt.complete(); // throws an exception if date is invalid
			return nextMatch(date.matchFrom()+date.charsMatched(),pt);
		} catch(Throwable t) { }
		return -1;
	}
	// use names beginning with "(?" to make new unique patterns.
	public String toString() {
		return "(?$date)"+nextString();
	}
};

// Extending Regex makes a new Regex compiler
// with extended syntax.  The method compile1
// is the only method you need supply.
//
// StrPos
//	inc() moves ahead one character
//	eos   boolean variable, true on end-of-string
//	match(char) matches a single character, but
//		not if it has been preceeded by a \
//    match(String s) matches each non-escaped char in
//		s, but only if not escaped with "\".  It also
//		advances past String s.
//
// Regex
//	add(pattern) adds a pattern to the list
//	compile1(StrPos) the routine you must over-ride to
//		compile your pattern.

class RegexWithDate extends Regex {
	RegexWithDate(String p) { super(p); }
	// We don't need to advance sp past
	// the current character.  sp.inc() will be called
	// automatically by compile.  If we had wanted to
	// match a 2 character pattern then we should call
	// sp.inc() once.
	public void compile1(StrPos sp,Rthings rt) throws RegSyntax {
		if(sp.match("(?$date)")) // this sequence in the compile
					   // String makes it.
			add(new DatePattern());
		else
			super.compile1(sp,rt);
	}
};

// Now to use the new DatePatternattern in an example.
// If it is successful, "It worked!" should print on the screen.
public class deriv {
	public static void main(String[] notUsed) {
		RegexWithDate r = new RegexWithDate("(?$date)");
		RegRes v1,v2;
		r.search("An invalid date: 2-30-1991."); // should fail,
			// there is no 30th of Feb
		v1=r.result();
		r.search("A valid date: 1-3-1998."); // should succeed.
		v2=r.result();
		if(v2.didMatch() && !v1.didMatch())
			System.out.println("It worked!");
		else System.out.println("v1="+v1+"\nv2="+v2);
	}
};
