// URLQueue.java
// A.L. Borchers, 1997 November
// University of Kentucky Department of Computer Science
// Queue for Scout engine which is synched to a set of rules

package Scout;

import java.io.Serializable;

import java.net.URL;

import java.util.Vector;

public class URLQueue implements Serializable {
  
  // Vector on which URLs are queued
  private Vector v;
  
  // The scout object we work for
  transient Scout scout;
  
  // Number of rules sharing this queue
  int rulesWorking;
  
  // Flag for manual shutdowns
  protected boolean done= false;
  
  URLQueue(Scout s) {
    scout= s;
    rulesWorking= 0;
    v= new Vector();
  }
  
  protected synchronized URL removeFront() {
    URL out= null;
    // While there's nothing on the queue, wait
    while (v.size() == 0 && scout.rulesLive() && !done) {
      try { 
		/*
        scout.logger.log("URLQueue.removeFront - calling wait on thread with " +
          rulesWorking + " rules working");
		*/
        wait();
      }
      catch (InterruptedException e) {
        // can be interrupted by an up, down or append
        // if it was a down that brought us to 0 rules working, break
		/*
        scout.logger.log("URLQueue.removeFront - Wait interrupted with " +
          rulesWorking + " rulesWorking");
		*/
      }
      if (rulesWorking == 0 || v.size() > 0) {
        break;
      }
    }
    if (done) {
      return null;
    }
    // Otherwise a URL arrived or all rules completed
    if (v.size() > 0) {
      out= (URL)v.elementAt(0);
      v.removeElementAt(0);
    }
    else {
      // no urls so mark object as done
      done= true;
    }
    // 
    scout.logger.log("URLQueue.removeFront - returning " + 
      ((out == null) ? "null" : out.toString()));
    return out;
  }
  
  public synchronized void append(URL in) {
    v.addElement(in);
    notifyAll();
  }
  
  protected int size() {
    return v.size();
  }
  
  protected void setScout(Scout s) {
    scout= s;
  }
  
  // Rules call this before processDoc to signal intention to use the queue
  protected synchronized void up() {
    rulesWorking++;
    // scout.logger.log("URLQueue.up - " + rulesWorking + " rules working");
    notifyAll();
  }
  
  // Each Rule calls the following method on completion of a document
  // whether it has added to the queue or not
  protected synchronized void down() {
    rulesWorking--;
    // scout.logger.log("URLQueue.down - " + rulesWorking + " rules working");
    notifyAll();
  }


}

