// Rule.java
// A.L. Borchers, 1997 November
// University of Kentucky Department of Computer Science

package Scout;

import pat.Regex;

import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;

public class Rule extends Thread {
	
	// Scout object we are working for
	protected Scout scout= null;
	
	// Hash of rule name=value pairs extracted from template
	protected RuleHash attr= null;
	
	// Vector in which to store extracted data
	protected Vector results= null;
	
	// Document object acquired from DocBuffer 
	protected Document doc= null;
	
	// Sequence numbers of string gathered from scout
	protected int sequenceNumber;
	
	// Constructor
	// Subclasses should always call this, but may add additional
	// initializations in their own constructors
	// Throws RuleFormatException if a VAR// argument is found that cannot
	// be resolved in Scout
	public Rule(Scout scout, RuleHash h)
		throws RuleFormatException {
		this.scout= scout;
		attr= h;
		setName(attr.get("name"));
		// interpolate any VAR// arguments
		Enumeration e= attr.keys();
		while (e.hasMoreElements()) {
			String key= (String)e.nextElement();
			String val= attr.get(key);
			Regex varMatch= new Regex("VAR/([A-z0-9]*)/");
			while (varMatch.search(val)) {
				String varKey= varMatch.substring(0);
				String varVal= scout.config.get("RUNTIME",varKey);
				if (varVal == null) {
					throw new RuleFormatException("No runtime value for " + 
						varKey + " defined");
				}
				val= varMatch.left() + varVal + varMatch.right();
			}
			// put the new interpolated version back in the table
			attr.put(key,val);
		}
		sequenceNumber= -1;
	}
	
	// Process the buffer output from scoutData until it
	// returns null (i.e. scout closed the buffer)
	public synchronized void run() {
		scout.logger.log(getName() + ".run - starting");
		// keep retrieveing documents until a null is returned
		while ((doc= scout.buffer.get(this,sequenceNumber)) != null) {
			// create a new results vector for each document
			results= new Vector();
			sequenceNumber++;
			if (attr.containsKey("validate") && !doc.isValid()) {
				scout.logger.log(getName() + ".run - ignoring invalid document " + 
					sequenceNumber);
			}
			else {
				scout.urls.up();
				scout.logger.log(getName() + ".run - acquired document " + 
					doc.getURL().toString() + " [" + sequenceNumber + "]");
				long startTime= (new Date()).getTime();
				processDoc();
				long processTime= ((new Date()).getTime() - startTime)/1000;
				scout.logger.log(getName() + ".run - finished document " + 
					doc.getURL().toString() + " [" + sequenceNumber + "] in " +
					(processTime/60) + " minutes " + (processTime%60) + 
					" seconds");
				scout.urls.down();
			}
			// report results even if it is empty, this allows other rules which may depend
			// on output of this one to wait on a report
			scout.ruleResults.put(this,sequenceNumber,results);
		}
		scout.logger.log(getName() + ".run - finished");
	}
	
	// Subclasses should override this method to implement extraction
	// Include a call to this method to get the diagnostic msg
	public synchronized void processDoc() {
		scout.logger.log(getName() + ".processDoc - processing document " + 
			doc.getURL().toString() + " [" + sequenceNumber + "]");
	}
	
	// Interpolate VAL/./ items int the given string
	protected String interpolateDataValues(String s) {
		Regex valMatch= new Regex("VAL/([A-z0-9]*)/");
		while (valMatch.search(s)) {
			// get the output of ruleName and combine into a disjunctive pattern
			String ruleName= valMatch.substring(0);
			String ruleValues= "";
			scout.logger.log(getName() + ".interpolateDataValues - requesting output from " + 
				ruleName + "." + sequenceNumber);
			Vector ruleOutput= scout.ruleResults.get(this,ruleName,sequenceNumber);
			if (ruleOutput.size() > 0) {
				for (int i= 0; i < ruleOutput.size()-1; i++) {
					ruleValues+= (((String)ruleOutput.elementAt(i)) + "|");
				}
				ruleValues= "(" + ruleValues + 
					(String)ruleOutput.elementAt(ruleOutput.size()-1) + ")";
			}
			// if we require output from this rule and there is none return, 
			// else just leave "" in place of the interp
			String require= attr.get("require");
			if (require != null && require.indexOf(ruleName) >= 0 && ruleValues.equals("")) {
				scout.logger.log(getName() + ".interpolateDataValues - No output of rule " + 
					ruleName + " available");
				s= null;
				break;
			}
			s= valMatch.left() + ruleValues + valMatch.right();
		}
		return s;
	}
	
	// Return a string identifying this rule and it's configuration items
	public String dumpConfig() {
		return getName() + " - " + attr.toString();
	}
	
}
