// Document.java
// A.L. Borchers, 1997 November
// University of Kentucky Department of Computer Science


package Scout;

import HTTPClient.NVPair;

import java.io.Serializable;

import java.net.URL;

import java.util.Vector;


class Document implements Serializable {

  private boolean isValid;
  private boolean isHTML;
  private URL url;
  private NVPair[] headers;
  private byte[] bytes;
  private Vector tags;

  Document() {
  }

  // Set the URL of this document to url
  protected void setURL(URL url) {
    this.url= url;
  }

  // Set valid flag for this document
  protected void setValid(boolean isValid) {
    this.isValid= isValid;
  }

  // Set flag for whether this is parsed HTML
  protected void setHTML(boolean isHTML) {
    this.isHTML= isHTML;
  }

  // Set the headers array for this document
  protected void setHeaders(NVPair[] headers) {
    this.headers= headers;
  }

  // Add a header to the document, replacing if a
  // matching header exists
  protected void addHeader(NVPair newHeader) {
    int i= 0;
    while (i < headers.length && !headers[i].getName().equalsIgnoreCase(newHeader.getName())) {
      i++;
    }
    if (i < headers.length) {
      // replace old header
      headers[i]= newHeader;
    }
    else {
      // realloc and append
      NVPair[] temp= headers;
      headers= new NVPair[temp.length + 1];
      System.arraycopy(temp,0,headers,0,temp.length);
      headers[temp.length]= newHeader;
    }
  }

  // Set the byte content for this document
  protected void setBytes(byte[] bytes) {
    this.bytes= bytes;
  }

  // Set the content of this document from a text string
  protected void setText(String text) {
    bytes= text.getBytes();
  }

  // Set the tag vector for this document
  protected void setTags(Vector tags) {
    this.tags= tags;
  }

  // Return true if document is marked as valid
  public boolean isValid() {
    return isValid;
  }

  // Return true if document is marked as HTML
  public boolean isHTML() {
    return isHTML;
  }

  // Return the URL of this document
  public URL getURL() {
    return url;
  }
  
  // Return the headers array of this document
  public NVPair[] getHeaders() {
    return headers;
  }

  // Get a header value by name
  public String getHeader(String headerKey)
    throws NoSuchHeaderException {
      if (headers == null) {
	throw new NoSuchHeaderException();
      }
      int i= 0;
      while (i < headers.length && !headers[i].getName().equalsIgnoreCase(headerKey)) {
	i++;
      }
      if (i == headers.length) {
	throw new NoSuchHeaderException();
      }
      return headers[i].getValue();
  }


  // Return the content type header value
  public String getContentType()
    throws NoSuchHeaderException {
      return getHeader("Content-type");
  }

  // Get the actual content length (NOT the header)
  public int getContentLength() 
    throws DocumentAccessException {
      if (bytes == null) {
	throw new DocumentAccessException("Field bytes uninitialized");
      }
      return bytes.length;
  }


  // Return true if this is a type text/* document as indicated
  // by the Content-type header
  public boolean isText()
    throws NoSuchHeaderException {
      return getContentType().toLowerCase().startsWith("text/");
  }

  // Return a copy of the document content as a byte array
  public byte[] getBytes() 
  throws DocumentAccessException {
      if (bytes == null) {
	throw new DocumentAccessException("Field bytes uninitialized");
      }
      byte[] out= new byte[bytes.length];
      System.arraycopy(bytes,0,out,0,bytes.length);
      return out;
  }

  // Return the content of this document as a text String
  // NOTE: returns the string even if the document is not of text type
  // DocumentAccessException is thrown only if the content is null
  public String getText() 
    throws DocumentAccessException {
      if (bytes == null) {
	throw new DocumentAccessException("Field bytes uninitialized");
      }
      return new String(bytes);
  }

  // Return the tag vector of this document
  public Vector getTags() {
    return tags;
  }
  

}
