// DocBuffer.java
// A.L. Borchers, 1997 November
// University of Kentucky Department of Computer Science

package Scout;

// Class maintains the components of a Web document,URL, tags and 
// text that is to be written to by the scout and read from by numRules 
// rules

import java.io.InputStream;
import java.io.IOException;

class DocBuffer {
  
  // The Scout we are working for
  protected Scout scout;
  
  // Document to be retrieved by the Rules
  private Document doc= null;
  
  // Total number of rules out there
  private int numRules;
  
  // Number of rules waiting on the current document
  private int rulesWaiting;
  
  // Currently available string number
  // Used to keep fast rules from getting
  // the same data twice
  private int sequenceNumber;
  
  // Flag to break wait loops if forced to exit
  protected boolean done= false;
  
  // Construct a new DocBuffer object
  protected DocBuffer(Scout scout, int numRules) {
    this.scout= scout;
    this.numRules= numRules;
    // Set rulesWaiting to 0 initially so scout will fill
    // the buffer
    rulesWaiting= 0;
    sequenceNumber= -1;
  }
  
  // assign the buffer document
  protected synchronized void fill(Document doc) {
    // If there are still rules waiting on this buffer
    // force the producer to wait
    while (rulesWaiting > 0 && !done) {
      // scout.logger.log("DocBuffer.fill - calling wait on scout");
      try {
        wait();
      }
      catch (InterruptedException e) {
      }
    }
    // When the last rule has finished, we can set the new doc and increment the 
    // sequence number so the rules can get back in
    rulesWaiting= numRules; // assume so many will attempt access
    this.doc= done ? null : doc;
    if (!done && doc.isValid()) {
      try {
        String contentType= doc.getContentType();
        int contentLength= doc.getContentLength();
        scout.logger.log("DocBuffer.fill - Buffered " +
          ( doc.isHTML() ? 
          (contentLength + " bytes of text and " + 
          doc.getTags().size() + " tags")
          : 
        (contentLength + " bytes of Content-type " + 
          contentType)));
        // Dang! That almost turned into Lisp?!
      }
      catch (DocumentAccessException dae) {
        scout.logger.log("DocBuffer.fill - Document buffered as valid but " + 
          "exception thrown: " + dae.toString());
      }
    }
    else 
      if (!done) {
        scout.logger.log("DocBuffer.fill - Buffered invalid document at " +
          "sequence number " + (sequenceNumber+1));
      }
      else {
        scout.logger.log("DocBuffer.fill - Manual shutdown");
      }
      sequenceNumber++;
      notifyAll();
  }
  
  // get the next document referenced greater than lastSequenceNumber
  // forcing the consumer to wait until such a string is available
  protected synchronized Document get(Rule rule, int lastSequenceNumber) {
    while (sequenceNumber <= lastSequenceNumber && !done) {
      // Note: majority of waits are called in 0 rules waiting state
      // (i.e. while scout is filling the buffer...)
      try {
        /*
			scout.logger.log("DocBuffer.get - Calling wait on " + rule.getName() + " with " +
          rulesWaiting + " rules waiting");
		  */
        wait();
      }
      catch (InterruptedException e) {
      }
    }
    if (done) {
      scout.logger.log("DocBuffer.get - Manual shutdown with " + rulesWaiting + 
        " rules waiting");
    }
    Document out= done ? null : doc;
    // 
    rulesWaiting= (rulesWaiting == 0) ? rulesWaiting : rulesWaiting - 1;
    if (rulesWaiting == 0) {
      notifyAll();
    }
    return out;
  }
  
  // close this buffer down when the last rule is finished
  protected synchronized void close() {
    // If there are still rules working on this buffer
    // force the producer to wait   
    while (rulesWaiting > 0) {
      try {
		// scout.logger.log("DocBuffer.close() - stalling scout");
        wait();
      }
      catch (InterruptedException e) {
      }
    }
    scout.logger.log("DocBuffer.close() - closing");
    // we still need to increment sequence number
    // to keep rules from waiting forever
    sequenceNumber++;
    doc= null;
    notifyAll();
  }
  
}
