/*

CacheManager.java
A.L. Borchers, 1997 November
University of Kentucky Department of Computer Science

Class for managing a cache of URLs

*/

package Scout;

import ADT.*;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.FileReader;
import java.io.FileWriter;

import java.io.FileNotFoundException;
import java.io.IOException;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

class CacheManager {

  // set true to make full cache an error
  public static final boolean noReduce= false;

  // default name of cache directory
  public static final String defaultCacheDirectory= "Cache";
  // default name of cache index file
  public static final String defaultCacheIndexFile= "cache.idx";
  // default number of files to cache
  public static final int defaultMaxCacheFiles= 128;
  // default fraction for cache reduction
  public static final float defaultCacheReduceFactor= 0.5f;

  // name of cache directory
  private String cacheDirectory= defaultCacheDirectory;
  // name of cache index file
  private String cacheIndexFile= defaultCacheIndexFile;
  // maximum number of files to store before purging
  private int maxCacheFiles= defaultMaxCacheFiles;
  // fraction for cache reduction
  private float cacheReduceFactor= defaultCacheReduceFactor;

  // Queue containing the cache order
  private Queue cacheOrder= null;
  //  Table mapping URLs to local files
  private Hashtable cacheTable= null;

  // the scout we work for
  private Scout scout= null;

  CacheManager(Scout scout)
    throws CacheException {
      this.scout= scout;
      // get configuartion info from the scout
      String temp= null;
      temp= scout.config.get("SCOUT","CacheDir");
      cacheDirectory= temp == null ? defaultCacheDirectory : temp;
      temp= scout.config.get("SCOUT","CacheIndex");
      cacheIndexFile= temp == null ? cacheIndexFile : temp;
      temp= scout.config.get("SCOUT","MaxCacheFiles");
      maxCacheFiles= temp == null ? maxCacheFiles : Integer.parseInt(temp);
      temp= scout.config.get("SCOUT","CacheReduceFactor");
      cacheReduceFactor= temp == null ? cacheReduceFactor : Float.valueOf(temp).floatValue();
      // verify cache directory is ready and init the queue and table 
      verifyCacheDirectory();
      loadCache();
  }

  protected void cacheDocument(String url, Document doc) 
    throws IOException, CacheException {
      if (noReduce && maxCacheFiles > 0 &&  cacheOrder.size() >= maxCacheFiles) {
        throw new CacheException("Cache full!");
      }
      String fileName= makeUniqueCacheName(url);
      ObjectOutputStream out= new ObjectOutputStream(new FileOutputStream(cacheDirectory + "/" + fileName));
      out.writeObject(doc);
      out.close();
      add(url,fileName);
  }

  // report the cache size
  protected int size() {
    return cacheOrder.size();
  }

  // return true if a document is cached
  protected boolean isCached(String url) {
    Enumeration e= cacheOrder.elements();
    while (e.hasMoreElements()) {
      String compare= (String)e.nextElement();
      if (compare.equals(url)) {
        return true;
      }
    }
    return false;
  }

  // return a stream for reading the cache file indicated by a url
  protected Document getDocument(String url) 
    throws CacheException {
      Document out= null;
      try {
	      ObjectInputStream in= null;
	      try {
	        String fileName= (String)cacheTable.get(url);
	        if (fileName == null)
	          throw new CacheException("Failed to find document for URL " + url + ". Cache inconsistent?");
	        in= new ObjectInputStream(new FileInputStream(cacheDirectory + "/" + fileName));
	      }
	      catch (FileNotFoundException e) {
	        throw new CacheException("Failed to find cache file which should exist");
	      }
	      out= (Document)in.readObject();
	      in.close();
      }
      catch (Exception e) {
	      throw new CacheException("Error loading document " + url + ": " + e);
      }
      return out;
  }

  // save the cache
  protected void save()
    throws IOException {
      scout.logger.log("CacheManager.save - Saving cache information");
      BufferedWriter writer= new BufferedWriter(new FileWriter(cacheDirectory + "/" + cacheIndexFile));
      for (int i= 0; i < cacheOrder.size(); i++) {
      	String url= (String)cacheOrder.remove();
	      String file= (String)cacheTable.get(url);
	      writer.write(url + " " + file + "\n");
	      cacheOrder.append(url);
      }
      writer.close();
  }

  // add an item to the cache queue and table
  private void add(String url, String fileName) {
    if (maxCacheFiles > 0 && cacheOrder.size() >= maxCacheFiles) reduce();
    cacheOrder.append(url);
    cacheTable.put(url,fileName);
  }

  // Load the cache order file if it exists, else create a new queue and table
  private void loadCache() 
    throws CacheException {
      try {
	      File cacheIndex= new File(cacheDirectory + "/" + cacheIndexFile);
	      cacheOrder= new Queue();
	      cacheTable= new Hashtable();
	      if (cacheIndex.exists()) {
	        String line= null;
	        BufferedReader reader= new BufferedReader(new FileReader(cacheDirectory + "/" + cacheIndexFile));
	        while ((line= reader.readLine()) != null) {
	          line= line.trim();
	          if (!line.equals("")) {
	            int split= line.indexOf(" ");
	            if (split < 0)
		            throw new Exception("Line split symbol not found");
	            String url= line.substring(0,split);
	            String file= line.substring(split+1);
	            cacheOrder.append(url);
	            cacheTable.put(url,file);
	          }
	        }
	        reader.close();
	      }
      }
      catch (Exception e) {
	      throw new CacheException("Failed to load cache index - " + e.toString());
      }
  }

  // Verify that the directory for cache exists in the required condition
  private void verifyCacheDirectory() 
    throws CacheException {
      String cacheDirName= scout.config.get("SCOUT","CacheDir");
      if (cacheDirName == null) {
	      throw new CacheException("No cache directory specified in Scout configuration");
      }
      File cacheDir= new File(cacheDirName);
      // if directory exists and is writable, alls well
      if (cacheDir.exists() && cacheDir.isDirectory() && cacheDir.canWrite()) {
	      return;
      }
      // Otherwise, figure out what's wrong
      // Check writable
      if (cacheDir.exists() && cacheDir.isDirectory()) {
	      throw new CacheException("Cannot write to the specified cache directory");
      }
      // Check existence
      if (cacheDir.exists()) {
        throw new CacheException("Specified cache directory exists as a file");
      }
      // Otherwise it doesn't exist so create it
      cacheDir.mkdirs();
  }

  // generate a guaranteed unique file name for a file in the cache directory
  private String makeUniqueCacheName(String url) {
    // generate the simple way
    String fileName= "c" + url.hashCode() + ".scd";
    File testFile= new File(cacheDirectory + "/" + fileName);
    // while the name generates collisions, hash on url extended by ~
    while (testFile.exists()) {
      url= url + "~";
      fileName= "c" + url.hashCode() + ".scd";
      testFile= new File(cacheDirectory + "/" + fileName);
    }
    return fileName;
  }

  // reduce the cache size to accomodate new entries
  private void reduce() {
    int removeCount= (int)(cacheOrder.size()*cacheReduceFactor);
    for (int i= 0; i < removeCount; i++) {
      cacheTable.remove(cacheOrder.remove());
    }
  }

}
