// Tag.java
// (c) 1997 A.L. Borchers
// Class for representing SGML tags in a dictionary style form

package SGMLKit;

import java.io.IOException;
import java.io.Reader;
import java.io.Serializable;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;


// NullAttributeValue object is used when attributes do not have a value so there's something
// to store in the hash (null would be an error and any arbitrary string could conflict with
// an actual argument for some unforeseen tag). Other classes referring to the tag hash items,
// e.g. RuleHash, should test whether the value associated with a key is instanceof String 
// before using it, and adjust their behavior accordingly (e.g. by internally denoting the 
// attribute with an inferred or default type, we generally treat them as flag "true" values.)

// NOTE: javac will be very upset if any other class attempts to refer to NullAttributeValue, 
// since it is only defined in this file. This is quite deliberate. Since NullAttributeValue
// has no members or methods, there is no need for other classes to know anything about it...

class NullAttributeValue implements Serializable {
	// 
}

// A Tag is a hashtable of attributes and their values with an identifier (SGML GI) string 
// and an integer index into the text file where the tag is inlined.

public class Tag extends Hashtable {
	
	// Various break character arrays used with the PeekabooReader that parses the tag
	public static final char[] atomBreaks= {' ','>'};
	public static final char[] nameBreaks= {' ','=','>'};
	public static final char[] quoteChars= {'"','\''};
	
	// Flags to the PeekabooReader
	public static final boolean expandWhitespace= true;
	public static final boolean includeDelimiter= true;
	
	// Value to store if an attribute doesn't have an explicit assignment
	private NullAttributeValue nullAttributeValue= new NullAttributeValue();
	
	// Element generic identifier
	private String identifier= null;		
	
	// Position to insert tag in text stream
	private int lCharPosition= -1;		
	
	// Reader to parse tag
	private transient PeekabooReader reader= null;
	
	// -----------------------------------------------------------------------------
	// Tag
	// -----------------------------------------------------------------------------
	// Construct a tag object given a reader set at the tag open char
	// -----------------------------------------------------------------------------
	public Tag(PeekabooReader r, int lCharPosition) 
		throws InvalidSGMLException, IOException {
		try {
			reader= r;
			// verify entry state
			int next= reader.peek();
			if ((char)next != '<') {
				throw new InvalidSGMLException("Tag.Tag - Expected tag open, found " + (char)next);
			}
			// discard the tag open
			reader.read();
			// Position of tag insert in text
			this.lCharPosition= lCharPosition;
			String attributeName= null;
			Object attributeValue= null;
			// read the tag identifier
			identifier= readNextNameToken().toLowerCase();
			while (!tagConsumed(true)) {
				// read next attribute name
				attributeName= readNextNameToken().toLowerCase();
				attributeValue= nullAttributeValue;
				// see if there's a value to read
				if (!tagConsumed(false) && reader.nextCharIs('=')) {
					reader.read(); // consume the '='
					attributeValue= readNextValueToken();
				}
				put(attributeName,attributeValue);
			}
		}
		catch (Exception e) {
			e.printStackTrace();
			System.err.println("\nTag so far: " + ((identifier != null) ? toString() : "???"));
			throw new InvalidSGMLException(e.toString());
		}
	}
	
	// -----------------------------------------------------------------------------
	// tagConsumed
	// -----------------------------------------------------------------------------
	// return true if next non-whitespace is end of tag (>) consuming the > if 
	// consumeTag flag is set true
	// -----------------------------------------------------------------------------
	private boolean tagConsumed(boolean consumeTag) 
		throws IOException {
		reader.skipWhitespace();
		if (reader.nextCharIs('>')) {
			if (consumeTag) {
				reader.read();
			}
			return true;
		}
		return false;
	}
	
	// -----------------------------------------------------------------------------
	// readNextNameToken
	// -----------------------------------------------------------------------------
	// return next attribute name (atomic) token
	// -----------------------------------------------------------------------------
	private String readNextNameToken() 
		throws IOException {
		reader.skipWhitespace();
		String out= reader.readToAny(nameBreaks,!includeDelimiter,expandWhitespace);
		return out;
	}
	
	// -----------------------------------------------------------------------------
	// readNextValueToken
	// -----------------------------------------------------------------------------
	// return next attribute value (quoted or atomic) token 
	// -----------------------------------------------------------------------------
	private String readNextValueToken()
		throws IOException {
		String out= null;
		reader.skipWhitespace();
		// if quoted handle by reading open quote and reading to close
		if (reader.nextCharIsAny(quoteChars)) {
			char quoteChar= (char)reader.read();
			out= quoteChar + reader.readTo(quoteChar,includeDelimiter);
		}
		else {
			// else treat as an atomic token
			out= readNextNameToken();
		}
		return out;
	}
	
	// -----------------------------------------------------------------------------
	// toString
	// -----------------------------------------------------------------------------
	// Return an HTML string representation of the tag with <> delimiters
	// -----------------------------------------------------------------------------
	public String toString() {
		String sOut= null;
		// using startsWith instead of equals is a bit of a kludge, but 
		// since some folks extend the comment with extra dashes...
		if (identifier.startsWith("!--")) {
			sOut= new String(identifier);
		}
		else {
			sOut= identifier;
			Enumeration sKeys= keys();
			// while attributes remain, append them
			while (sKeys.hasMoreElements()) {
				String attributeName= (String)sKeys.nextElement();
				sOut= sOut + " " + attributeName;
				// if the attribute value is a NullAttribute leave alone,
				// otherwise append the string value
				Object attributeValue= get(attributeName);
				if (!(attributeValue instanceof NullAttributeValue)) {
					sOut= sOut + "=" + (String)attributeValue;
				}
			}
		}
		// return with the appropriate brackets
		return "<" + sOut + ">";		
	}
	
	// -----------------------------------------------------------------------------
	// getPosition
	// -----------------------------------------------------------------------------
	// Get the character position of this tag's inlining in it's parent document
	// -----------------------------------------------------------------------------
	public int getPosition() {
		return lCharPosition;
	}
	
	// -----------------------------------------------------------------------------
	// getIdentifier
	// -----------------------------------------------------------------------------
	// Get the generic identifier for this tag
	// -----------------------------------------------------------------------------
	public String getIdentifier() {
		return new String(identifier);
	}
	
	// -----------------------------------------------------------------------------
	// get
	// -----------------------------------------------------------------------------
	// Override for Hashtable.get returns a string for attribute values
	// -----------------------------------------------------------------------------
	public String get(String attributeName) {
		Object o= super.get(attributeName);
		if (o instanceof NullAttributeValue) {
			return "true";
		}
		else {
			return (String)o;
		}
	}
	
	
	// -----------------------------------------------------------------------------
	// getUnquoted
	// -----------------------------------------------------------------------------
	// Override for Hashtable.get returns a string for attribute values
	// -----------------------------------------------------------------------------
	public String getUnquoted(String attributeName) {
		Object o= super.get(attributeName);
		if (o == null) {
			return null;
		}
		if (o instanceof NullAttributeValue) {
			return "true";
		}
		else {
			String result= (String)o;
			if ((result.startsWith("\"") && result.endsWith("\"")) ||
				(result.startsWith("'") && result.endsWith("'"))) {
				result= result.substring(1);
				result= result.substring(0,result.length()-1);
			}
			return result;
		}
	}
	
	
	// -----------------------------------------------------------------------------
	// locateTag
	// -----------------------------------------------------------------------------
	// return the first index of a tag in the vector with identifier = tagID or
	// -1 if fails 
	// -----------------------------------------------------------------------------
	public static int locateTag(Vector tags, String tagID, int i) {
		while (i < tags.size() && !((Tag)tags.elementAt(i)).getIdentifier().equals(tagID)) {
			i++;
		}
		return (i == tags.size()) ? -1 : i;
	}
	
	// -----------------------------------------------------------------------------
	// extractTaggedRegion
	// -----------------------------------------------------------------------------
	// return the first occurence after startTag of a region in text tagged by 
	// openTag and closeTag. Return null if the location of the tags fails.
	// -----------------------------------------------------------------------------
	public static String extractTaggedRegion(Vector tags, String text, 
		String openTag, String closeTag,
		int startTag) {
		String result= null;
		int open, close;
		open= locateTag(tags,openTag,startTag);
		if (open >= 0) {
			close= locateTag(tags,closeTag,open+1);
			if (close >= open+1) {
				result= extractTaggedRegion(tags,text,open,close);
			}
		}
		return result;
	}
	
	public static String extractTaggedRegion(Vector tags, String text, 
					   int open, int close) {
		return text.substring(((Tag)tags.elementAt(open)).getPosition(),
			((Tag)tags.elementAt(close)).getPosition());
	}
	
	
}









