// PeekabooReader.java
// (c) 1997 A.L. Borchers
// Extension of PushbackReader with some additional lookahead features

package SGMLKit;

import java.io.PushbackReader;
import java.io.Reader;
import java.io.IOException;

class PeekabooReader extends PushbackReader {

  public PeekabooReader(Reader r) {
    super(r);
  }

  // Peek at next character on reader
  public int peek() 
    throws IOException {
    int next= read();
    if (next != -1) {
      unread(next);
    }
    return next;
  }

  // Return true if reader is consumed
  public boolean consumed()
    throws IOException {
      int next= peek();
      return next == -1;
  }

  // Return true if next char is equal to argument
  public boolean nextCharIs(char c) 
    throws IOException {
      int next= peek();
      if (next != -1) {
	return (char)peek() == c;
      }
      return false;
  }

  // Return true if next char is any whitespace char
  public boolean nextCharIsWhitespace() 
    throws IOException {
      int next= peek();
      if (next != -1) {
	return Character.isWhitespace((char)next);
      }
      return false;
  }

  // Return true if next char is any char in delim array
  public boolean nextCharIsAny(char[] delim) 
    throws IOException {
      if (consumed()) {
	throw new IOException("Expected delimiter char, found end of read");
      }
      for (int i= 0; i < delim.length; i++) {
	if (nextCharIs(delim[i])) {
	  return true;
	}
      }
      return false;
  }

  // Return true if next char is any char in delim array, testing with 
  // nextCharIsWhitespace if ' ' is found in delim
  public boolean nextCharIsAny(char[] delim, boolean expandWhitespace) 
    throws IOException {
      if (consumed()) {
	throw new IOException("Expected delimiter char, found end of read");
      }
      for (int i= 0; i < delim.length; i++) {
	if (nextCharIs(delim[i]) || (delim[i] == ' ' && expandWhitespace && nextCharIsWhitespace())) {
	  return true;
	}
      }
      return false;
  }

  // Skip to the next non-whitespace on the reader
  public void skipWhitespace() 
    throws IOException {
      while (!consumed() && nextCharIsWhitespace()) {
	read();
      }
  }

  // Read to the next occurence of delim, including or excluding delim as 
  // indicated by the includeDelimiter flag
  public String readTo(char delim, boolean includeDelimiter)
    throws IOException {
      StringBuffer out= new StringBuffer();
      while (!consumed() && !nextCharIs(delim)) {
	int next= read();
	// honor escapes
	if ((char)next == '\\') {
	  out.append((char)next);
	  if (consumed()) {
	    throw new IOException("Invalid escape sequence");
	  }
	  next= read();
	}
	out.append((char)next);
      }
      if (consumed()) {
	throw new IOException("Expected " + delim + ", found end of read");
      }
      if (includeDelimiter) {
	out.append((char)read());
      }
      return out.toString();
  }


  // Read to the next occurence of delim, including or excluding delim and 
  // expanding whitespace as indicated by the flags
  public String readTo(char delim, boolean includeDelimiter, boolean expandWhitespace) 
    throws IOException {
      StringBuffer out= new StringBuffer();
      while (!consumed() && (delim == ' ' && expandWhitespace) ? !nextCharIsWhitespace() : !nextCharIs(delim)) {
	int next= read();
	// honor escapes
	if ((char)next == '\\') {
	  out.append((char)next);
	  if (consumed()) {
	    throw new IOException("Invalid escape sequence");
	  }
	  next= read();
	}
	out.append((char)next);
      }
      if (consumed()) {
	throw new IOException("Expected " + delim + ", found end of read");
      }
      if (includeDelimiter) {
	out.append((char)read());
      }
      return out.toString();
  }

  // Read to the next occurence of any delim, including or excluding delim and 
  // expanding whitespace as indicated by the flags
  public String readToAny(char[] delim, boolean includeDelimiter, boolean expandWhitespace) 
    throws IOException {
      StringBuffer out= new StringBuffer();
      while (!consumed()) {
	if (nextCharIsAny(delim,expandWhitespace)) {
	  break;
	}
	int next= read();
	// honor escapes
	if ((char)next == '\\') {
	  out.append((char)next);
	  if (consumed()) {
	    throw new IOException("Invalid escape sequence");
	  }
	  next= read();
	}
	out.append((char)next);
      }
      if (consumed()) {
	throw new IOException("Expected delimiter char, found end of read");
      }
      if (includeDelimiter) {
	out.append((char)read());
      }
      return out.toString();
  }


}
