/*
* JavaDocObject.java
* A.L. Borchers, 1998
*
* Data item describing a JavaDoc element for inclusion in a tree
*
*/

package JavaDoc;

import java.io.Serializable;

import java.net.URL;

import java.util.Hashtable;
import java.util.Vector;

import SGMLKit.Tag;

public class JavaDocObject implements Serializable {
	
	// Flags int is a bit array of type, modifier, and other 
	// flag data that is descriptive of all Java classes, 
	// interfaces, and instances.
	protected int flags= 0;
	
	// The type region of the flags int should only have one 
	// bit on at any time, while the modifier and general flags 
	// region may have several, so we divide them explicitly 
	// using masks
	
	// Mask for data type region
	public static final int _TYPE_MASK=    0x0000FFFF;
	
	// Mask for modifier region
	public static final int _MDFR_MASK=    0x00FF0000;
	
	// Mask for misc. flags region
	public static final int _MISC_MASK=    0xFF000000;
	
	// General mask for modifier and misc masks
	public static final int _HIGH_MASK=    0xFFFF0000;
	
	// Data type constants
	
	// The lack of any type info in the _TYPE_MASK region indicates
	// we don't know what the object is
	public static final int _TYPE_UNKNOWN= 0;
	
	// Primitive types
	public static final int _TYPE_VOID=         0x00000001;
	public static final int _TYPE_BYTE=         0x00000002;
	public static final int _TYPE_CHAR=         0x00000004;
	public static final int _TYPE_SHORT=        0x00000008;
	public static final int _TYPE_INT=          0x00000010;
	public static final int _TYPE_LONG=         0x00000020;
	public static final int _TYPE_FLOAT=        0x00000040;
	public static final int _TYPE_DOUBLE=       0x00000080;
	public static final int _TYPE_BOOLEAN=      0x00000100;
	
	// Complex types - the first two are most significant
	public static final int _TYPE_INTERFACE=    0x00000200; 
	public static final int _TYPE_CLASS=        0x00000400; 
	public static final int _TYPE_PACKAGE=      0x00000800;
	public static final int _TYPE_EXCEPTION=    0x00001000;
	public static final int _TYPE_ERROR=        0x00002000;
	public static final int _TYPE_FIELD=        0x00004000; // (instance)
	public static final int _TYPE_METHOD=       0x00008000;
	public static final int _TYPE_ARRAY=		0x02000000;
	
	// Access and other modifiers
	public static final int _MDFR_FRIENDLY=     0x00010000;
	public static final int _MDFR_PUBLIC=       0x00020000;
	public static final int _MDFR_PRIVATE=      0x00040000;
	public static final int _MDFR_PROTECTED=    0x00080000;
	public static final int _MDFR_STATIC=       0x00100000;
	public static final int _MDFR_FINAL=        0x00200000;
	public static final int _MDFR_TRANSIENT=    0x00400000;
	public static final int _MDFR_SYNCHRONIZED= 0x00800000;
	
	// Other flags
	
	public static final int _FLAG_DEPRECATED=   0x01000000;

	
	// given a url string, return the last name preceded by a / and 
	// followed by .name.html

	public static final String getPackageNameFromObjectURL(Tag t) {
		String result= null;
		String urlStr= t.getUnquoted("href");
		if (urlStr != null) {
			result= getPackageNameFromObjectURL(urlStr);
		}
		return result;
	}

	public static final String getPackageNameFromObjectURL(String urlStr) {
		String result= null;
		int split= -1;
		// isolate after "/"
		result= urlStr.substring(urlStr.lastIndexOf("/")+1);
		// isolate before ".html"
		split= result.lastIndexOf(".html");
		result= split > -1 ? result.substring(0,result.lastIndexOf(".html")) : result;
		// isolate before "." (before object name)
		split= result.lastIndexOf(".");
		result= split > -1 ? result.substring(0,split) : result;		
		return result;
	}

	// given a url string, return the last name preceded by a dot and 
	// followed by .html

	public static final String getObjectNameFromObjectURL(Tag t, boolean trimPackageName) {
		String result= null;
		String urlStr= t.getUnquoted("href");
		if (urlStr != null) {
			result= getObjectNameFromObjectURL(urlStr, trimPackageName);
		}
		return result;
	}

	public static final String getObjectNameFromObjectURL(String urlStr, boolean trimPackageName) {
		String result= null;
		int split= -1;
		// isolate before ".html"
		split= urlStr.lastIndexOf(".html");
		result= split > -1 ? urlStr.substring(0,split) : urlStr;
		// if trimPackageName is set, isolate after "." (package name), else 
		// isolate after last "/" (leaves package name in result)
		split= trimPackageName ? result.lastIndexOf(".") : result.lastIndexOf("/");
		result= split > -1 ? result.substring(split+1) : result;
		return result;
	}


	// instance members
	
	protected String javaDocURL= null;	// url of javadoc describing this object (relative from # for members)
	
	protected String objectName= null;	// name of class, field, method
	
	protected String className= null;	// name of class (if class object) or class returned (if method)
	
	protected String parent= null;		// always null for non-class objects

	protected String container= null;	// package of class or class of member
	
	protected Vector interfaces= null;	// always null for non-class objects

	protected Vector methodargs= null;  // always null for non-method objects
	
	protected String description= null;	// descriptive text
	
	// Nullie
	JavaDocObject() {
	}
	
	protected void setType(int type) {
		// Or in the new type info, taking care not to contaminate
		// the high order bits
		flags= flags|(type&_TYPE_MASK);
	}
	
	protected void setType(String typeString) {
		// Note special handling to ensure exclusivity of type 
		// any previous type data is wiped
		setType(getType(typeString));
	}
	
	public int getType() {
		return flags&_TYPE_MASK;
	}
	
	protected void setModifier(int modifier) {
		// Since modifiers are additive, we just or the new one in, 
		// assuring that we don't contaminate the rest of the vector
		flags= flags|(modifier&_MDFR_MASK);
	}
	
	protected void setModifier(String modifier) {
		setModifier(getModifier(modifier));
	}
	
	// Get all modifiers as a mask
	protected int getModifiers() {
		return flags&_MDFR_MASK;
	}
	
	// Boolean test whether a modifier bit is set
	protected boolean getModifier(int modifier) {
		return (flags&modifier&_MDFR_MASK) != 0;
	}
	
	// Two JavaDocObjects are considered equal if they have the same 
	// fully qualified name, flags, and - if the object is a method - 
	// parameter list
	public boolean equals(JavaDocObject compare) {
		boolean result= this.objectName.equals(compare.objectName) && 
						this.flags == compare.flags;
		if (result && (this.flags&_TYPE_METHOD) == _TYPE_METHOD) {
			result= this.methodargs.size() == compare.methodargs.size();
			// compare args
			for (int i= 0; result && i < this.methodargs.size(); i++) {
				result= ((String)this.methodargs.elementAt(i)).equals(
						(String)compare.methodargs.elementAt(i));
			}

		}
		return result;
	}
	
	// toString prints a table of the items and text translation of 
	// modifier and type flags
	public String toString() {
		String out= "\n" +
			"URL:         " + javaDocURL + "\n" + 
			"Name:        " + objectName + "\n" + 
			"Type:        ";

		// Get modifier info from high 16 bits of flag
		if ((flags&_MDFR_FRIENDLY) > 0)				out+= "friendly ";
		if ((flags&_MDFR_PUBLIC) > 0)				out+= "public ";
		if ((flags&_MDFR_PRIVATE) > 0)				out+= "private ";
		if ((flags&_MDFR_PROTECTED) > 0)			out+= "protected ";
		if ((flags&_MDFR_STATIC) > 0)				out+= "static ";
		if ((flags&_MDFR_FINAL) > 0)				out+= "final ";
		if ((flags&_MDFR_TRANSIENT) > 0)			out+= "transient ";
		if ((flags&_MDFR_SYNCHRONIZED) > 0)			out+= "synchronized ";

		// Get type info from low 16 bits of flag
		if ((flags&_TYPE_MASK) == _TYPE_UNKNOWN)	out+= "unknown ";
		if ((flags&_TYPE_VOID) > 0)					out+= "void ";
		if ((flags&_TYPE_BYTE) > 0)					out+= "byte "; 
		if ((flags&_TYPE_CHAR) > 0)					out+= "char "; 
		if ((flags&_TYPE_SHORT) > 0)				out+= "short "; 
		if ((flags&_TYPE_INT) > 0)					out+= "int "; 
		if ((flags&_TYPE_LONG) > 0)					out+= "long "; 
		if ((flags&_TYPE_FLOAT) > 0)				out+= "float "; 
		if ((flags&_TYPE_DOUBLE) > 0)				out+= "double "; 
		if ((flags&_TYPE_BOOLEAN) > 0)				out+= "boolean "; 
		if ((flags&_TYPE_INTERFACE) > 0)			out+= "interface ";
		if ((flags&_TYPE_CLASS) > 0)				out+= "class ";
		if ((flags&_TYPE_PACKAGE) > 0)				out+= "package ";
		if ((flags&_TYPE_EXCEPTION) > 0)			out+= "exception ";
		if ((flags&_TYPE_ERROR) > 0)				out+= "error ";
		if ((flags&_TYPE_FIELD) > 0)				out+= "field ";
		if ((flags&_TYPE_ARRAY) > 0)				out+= "array ";
		if ((flags&_TYPE_METHOD) > 0)				out+= "method ";

		out+= "\n" +
			"Class:       " + className + "\n" + 
			"Parent:      " + parent + "\n" + 
			"Container:   " + container + "\n" +
			"Interfaces:  " + interfaces + "\n" + 
			"Arguments:   " + methodargs + "\n" + 
			"Flags:       " + flags + "\n" + 
			"Description: " + description + "\n\n";

		return out;

	}
	
	
	// Boolean test whether a given string token is a modifier keyword
	public static boolean isModifierToken(String modString) {
		return getModifier(modString) != _TYPE_UNKNOWN; 
	}
	
	// Boolean test whether a given string token is a type keyword
	public static boolean isTypeToken(String typeString) {
		return getType(typeString) != _TYPE_UNKNOWN;
	}
	
	// Map a type token to the appropriate int, returning unknown if
	// the token doesn't map
	public static int getType(String typeString) {
		int type= _TYPE_UNKNOWN;
		// use strict equality on primitive types
		if (typeString.equals("void")) {
			type= _TYPE_VOID;
		}
		else if (typeString.equals("byte")) {
			type= _TYPE_BYTE;
		}
		else if (typeString.equals("char")) {
			type= _TYPE_CHAR;
		}
		else if (typeString.equals("short")) {
			type= _TYPE_SHORT;
		}
		else if (typeString.equals("int")) {
			type= _TYPE_INT;
		}
		else if (typeString.equals("long")) {
			type= _TYPE_LONG;
		}
		else if (typeString.equals("float")) {
			type= _TYPE_FLOAT;
		}
		else if (typeString.equals("double")) {
			type= _TYPE_DOUBLE;
		}
		else if (typeString.equals("boolean")) {
			type= _TYPE_BOOLEAN;
		}
		// use ignore case on complex types
		else if (typeString.equalsIgnoreCase("package")) {
			type= _TYPE_PACKAGE;
		}
		else if (typeString.equalsIgnoreCase("interface")) {
			type= _TYPE_INTERFACE;
		}
		else if (typeString.equalsIgnoreCase("class")) {
			type= _TYPE_CLASS;
		}
		else if (typeString.equalsIgnoreCase("exception")) {
			type= _TYPE_EXCEPTION;
		}
		else if (typeString.equalsIgnoreCase("error")) {
			type= _TYPE_ERROR;
		}
		else if (typeString.equalsIgnoreCase("field")) {
			type= _TYPE_FIELD;
		}
		else if (typeString.equalsIgnoreCase("method")) {
			type= _TYPE_METHOD;
		}
		return type;
	}
	
	// Map a modifier token to the appropriate int, returning unknown 
	// if the token doesn't map
	public static int getModifier(String modString) {
		int modifier= _TYPE_UNKNOWN;
		if (modString.equals("friendly")) {
			modifier= _MDFR_FRIENDLY;
		}
		else if (modString.equalsIgnoreCase("public")) {
			modifier= _MDFR_PUBLIC;
		}
		else if (modString.equals("private")) {
			modifier= _MDFR_PRIVATE;
		}
		else if (modString.equals("protected")) {
			modifier= _MDFR_PROTECTED;
		}
		else if (modString.equals("static")) {
			modifier= _MDFR_STATIC;
		}
		else if (modString.equals("final")) {
			modifier= _MDFR_FINAL;
		}
		else if (modString.equals("transient")) {
			modifier= _MDFR_TRANSIENT;
		}
		else if (modString.equals("synchronized")) {
			modifier= _MDFR_SYNCHRONIZED;
		}
		return modifier;
	}
	
	
}




// junkyard

/*
		if ((flags&0x0000FFFF) == 0) {
			// No type info resolved
			out+= "unknown";
		}
		else {
			// may be exclusive primitive or multivalued complex
			if ((flags&0x000001FF) > 0) {
				switch (flags&0x000001FF) {
				case _TYPE_VOID: out+= "void ";	break;
				case _TYPE_BYTE: out+= "byte"; break;
				case _TYPE_CHAR: out+= "char"; break;
				case _TYPE_SHORT: out+= "short"; break;
				case _TYPE_INT: out+= "int"; break;
				case _TYPE_LONG: out+= "long"; break;
				case _TYPE_FLOAT: out+= "float"; break;
				case _TYPE_DOUBLE: out+= "double"; break;
				case _TYPE_BOOLEAN: out+= "boolean"; break;
				}
			}
			else {
				// complex type may have more than one bit set
				out+= "complex - ";
				int complex= flags&0x0000FE00;
				if ((complex&_TYPE_INTERFACE) > 0)out+= "interface ";
				if ((complex&_TYPE_CLASS) > 0) out+= "class ";
				if ((complex&_TYPE_PACKAGE) > 0) out+= "package ";
				if ((complex&_TYPE_EXCEPTION) > 0) out+= "exception ";
				if ((complex&_TYPE_ERROR) > 0) out+= "error ";
				if ((complex&_TYPE_FIELD) > 0) out+= "field ";
				if ((complex&_TYPE_METHOD) > 0) out+= "method ";
			}
		}
		out+= "\n";
		// Get modifiers
		out+= "Modifiers: ";
		int modifiers= flags&0xFFFF0000;
		if (modifiers > 0) {
			if ((modifiers&_MDFR_FRIENDLY) > 0) out+= "friendly ";
			if ((modifiers&_MDFR_PUBLIC) > 0) out+= "public ";
			if ((modifiers&_MDFR_PRIVATE) > 0) out+= "private ";
			if ((modifiers&_MDFR_PROTECTED) > 0) out+= "protected ";
			if ((modifiers&_MDFR_STATIC) > 0) out+= "static ";
			if ((modifiers&_MDFR_FINAL) > 0) out+= "final ";
			if ((modifiers&_MDFR_TRANSIENT) > 0) out+= "transient ";
			if ((modifiers&_MDFR_SYNCHRONIZED) > 0) out+= "synchronized ";
		}
		else {
			out+= "none";
		}
		*/
