

package JavaDoc;


import java.io.*;
import java.util.*;


public class BuildMethodIndex {

	private static Hashtable resultsTable= null;
	private static Hashtable closureTable= null;
	private static Hashtable methodsTable= null;

	public static void main(String args[]) {
		if (args.length < 3) {
			usage();
		}
		else {
			System.out.println("Loading results table");
			resultsTable= getResultsTable(args[0]);
			System.out.println("Finding closure");
			closureTable= buildClosureTable(args[1]);
			resultsTable= null;
			calculateClosure();
			System.out.println("Building method index");
			buildMethodsHash();
			System.out.println("Saving methods index");
			saveMethodsTable(args[2]);
		}
	}


	private static boolean calculateClosure() {
		boolean result= true;
		int elements= closureTable.size();
		Enumeration keys= closureTable.keys();
		while (keys.hasMoreElements()) {
			String key= (String)keys.nextElement();
			Vector jdoVector= (Vector)closureTable.get(key);
			JavaDocObject primary= (JavaDocObject)jdoVector.elementAt(0);
			result= closeObject(jdoVector,primary.parent);
			if (!result) break;
		}	
		return result;
	}

	private static boolean closeObject(Vector jdoVector, String closeWith) {
		// possibilities
		// class - parentage followable all the way to java.lang.Object, 
		// which has null parent
		// interface - parentage null if not descended from another interface 
		boolean result= true;
		if (closeWith != null) {
			try {
				JavaDocObject primary= (JavaDocObject)jdoVector.elementAt(0);
				System.out.println("Closing " + primary.className + 
					" with " + closeWith);
				if (closeWith != null) {
					Vector parentVector= (Vector)closureTable.get(closeWith);
					if (parentVector != null) {
						JavaDocObject parent= (JavaDocObject)parentVector.elementAt(0);
						if (!parent.equals(primary)) {
							// can't inherit the object, so only need to look at 
							// fields and methods
							for (int i= 1; i < parentVector.size(); i++) {
								JavaDocObject member= (JavaDocObject)parentVector.elementAt(i);
								if (!isInheritable(member)) {
									System.out.println(primary.className + " cannot inherit private member " + 
										member.javaDocURL + " from " + parent.className);
								}
								else if (isOverridden(jdoVector,member)) {
									System.out.println(primary.className + " already overrides " + 
										member.javaDocURL + " from " + parent.className);
								}
								else {
									// add the inherited member
									System.out.println(primary.className + " inheriting " + 
										member.javaDocURL + " from " + parent.className);
									jdoVector.addElement(member);
								}
							}
						}

						// recurse to parents parent
						closeObject(jdoVector,parent.parent);
					}
					else {
						System.err.println("WARNING: " + primary.className + 
							" references parent " + closeWith + " but no entry by " +
							" that name could be found...");
					}
				}
				// else nothing to do!
			}
			catch (Exception e) {
				result= false;
				e.printStackTrace();
			}
		}
		return result;
	}

	// return true if the object can be inherited by a subclass
	// i.e. the private flag isn't set...
	private static boolean isInheritable(JavaDocObject jdo) {
		return (jdo.flags&JavaDocObject._MDFR_PRIVATE) == 0;
	}

	// return true if the object is overridden in the jdoVector
	private static boolean isOverridden(Vector jdoVector, JavaDocObject jdo) {
		boolean result= false;
		for (int i= 0; !result && (i < jdoVector.size()); i++) {
			JavaDocObject compare= (JavaDocObject)jdoVector.elementAt(i);
			if (jdo.objectName.equals(compare.objectName) && 
				((jdo.flags&JavaDocObject._TYPE_METHOD) == JavaDocObject._TYPE_METHOD) && 
				((compare.flags&JavaDocObject._TYPE_METHOD) == JavaDocObject._TYPE_METHOD) && 
				(jdo.methodargs.size() == compare.methodargs.size())) {
				// compare args
				boolean allMatch= true;
				for (int j= 0; allMatch && (j < jdo.methodargs.size()); j++) {
					String jdoArg= (String)jdo.methodargs.elementAt(j);
					String cmpArg= (String)compare.methodargs.elementAt(j);
					allMatch= allMatch && jdoArg.equals(cmpArg);
				}
				result= allMatch;
			}
		}
		return result;
	}

	//
	// getResultsTable - load the results table produced by Scout
	// 
	private static Hashtable getResultsTable(String filename) {
		System.out.println("Getting Scout's Results table - this can take a while...");
		Object		o= null;
		Hashtable	r= null;
	    try {
			File f= new File(filename);
			if (f.exists()) {
				// Restore existing state
				ObjectInputStream in= new ObjectInputStream(new FileInputStream(filename));
				o= in.readObject(); // robots exclusion data
				o= in.readObject(); // url queue
				o= in.readObject(); // visits record
				o= in.readObject();	// results
				in.close();
				r= (Hashtable)o;
			}
			else {
				System.err.println("Error: " + filename + " not found...");
			}
		}
		catch (Exception e) {
			// uh-oh!
			System.err.println("ERROR: " + e.toString());
			e.printStackTrace();
		}
		return r;
	}


	private static Hashtable buildClosureTable(String jdoRuleName) {
		Hashtable h= null;
		// get the results for the named rule
		Vector alldocs= (Vector)resultsTable.get(jdoRuleName);
		if (alldocs != null) {
			h= new Hashtable();
			// build closure table with keys the names of the primary objects in the 
			// jdos for each document
			int elements= alldocs.size();
			Vector jdos;
			JavaDocObject jdo;
			for (int i= 0; i < elements; i++) {
				Vector resultsVector= (Vector)alldocs.elementAt(i);
				if (resultsVector.size() > 0) {
					// get the vector describing the next jdo
					jdos= (Vector)resultsVector.elementAt(0);
					if (jdos.size() > 0) {
						jdo= (JavaDocObject)jdos.elementAt(0);
						// System.out.println("Found " + jdo.className);
						h.put(jdo.className, jdos);
					}
					else {
						System.err.println("WARNING: No object data in results for document " + i);
					}
				}
				else {
					System.err.println("WARNING: No result data for document " + i);
				}
			}
		}
		else {
			System.err.println("ERROR: No results found for rule " + jdoRuleName);
		}
		return h;
	}


	private static boolean buildMethodsHash() {
		methodsTable= new Hashtable();
		boolean result= true;
		Enumeration keys= closureTable.keys();
		for (int i= 0; i < closureTable.size(); i++) {
			String nextKey= (String)keys.nextElement();
			Vector v= (Vector)closureTable.get(nextKey);
			JavaDocObject jdo= (JavaDocObject)v.elementAt(0);
			for (int j= 1; j < v.size(); j++) {
				JavaDocObject mem= (JavaDocObject)v.elementAt(j);
				Vector vv= (Vector)methodsTable.get(mem.objectName);
				if (vv == null) {
					vv= new Vector();
					methodsTable.put(mem.objectName,vv);
				}
				if (!vv.contains(jdo.className)) {
					vv.addElement(jdo.className);
				}
			}
		}
		return result;
	}

	//
	// saveClosureTable - save closureTable to specified file
	//
	private static void saveMethodsTable(String outfile) {
		System.out.println("Saving class hash to " + outfile);
		try {
			ObjectOutputStream oout= new ObjectOutputStream(new FileOutputStream(outfile));
			oout.writeObject(methodsTable);
			oout.close();
		}
		catch (Exception e) {
			System.err.println("ERROR: Error saving to " + outfile);
			e.printStackTrace();
		}
	}

	public static void usage() {
		System.out.println("usage: java BuildMethodIndex infile classdocrule outfile");
		System.exit(0);
	}

}