/*
 * Tree.java
 * A.L. Borchers, 1998
 *
 * A general tree of Child-parent Nodes
 * 
 */

package ADT;

public class Tree {

  public static final int defaultSearch= Search.BFS;

  public static final int PREORDER= 0;
  public static final int POSTORDER= 1;

  public static final int defaultTraversal= PREORDER;

  public static final boolean debug= true;

  Node root= null;

  public Tree() {
    root= new Node(null);
  }

  /* Create a tree rooted at root */
  public Tree(Node root) 
    throws IllegalArgumentException {
      if (root.parent != null) {
        throw new IllegalArgumentException("Node passed to Tree constructor must have null parent");
      }
      this.root= root;
  }

  public Node getRoot() {
    return root;
  }

  public Node findNode(Object data)
    throws IllegalArgumentException {
      return findNode(root,data,defaultSearch);
  }

  public Node findNode(Object data, int mode) 
    throws IllegalArgumentException {
      return findNode(root,data,mode);
  }

  // All searches are via this static method

  public static Node findNode(Node node, Object data, int mode)
    throws IllegalArgumentException {
      switch (mode) {
      case Search.BFS:
	return Search.BFSearch(node,data);
      case Search.DFS:
	return Search.DFSearch(node,data);
      default:
	throw new IllegalArgumentException("Search mode not available");
      }
  }

  public Queue traverse() {
    return traverse(root,defaultTraversal);
  }

  public Queue traverse(int mode)
    throws IllegalArgumentException {
      return traverse(root,mode);
  }

  // Rest of the traversals are static

  public static Queue traverse(Node node) {
    return traverse(node,defaultTraversal);
  }

  public static Queue traverse(Node node, int mode) 
    throws IllegalArgumentException {
      Queue out= new Queue();
      switch (mode) {
      case PREORDER:
	out= traversePreOrder(out,node);
	break;
      case POSTORDER:
	out= traversePostOrder(out,node);
	break;
      default:
	throw new IllegalArgumentException("Traversal mode not available");
      }
      return out;
  }

  private static Queue traversePreOrder(Queue q, Node n) {
    if (n != null) {
      if (debug) {
	System.out.println("Queue.traversePreOrder - Node data: " + n.data + " numChildren: " + n.children.size());
      }
      q.append(n.data);
      for (int i= 0; i < n.children.size(); i++) {
	traversePreOrder(q,n.children.getSibling(i));
      }
    }
    return q;
  }

  private static Queue traversePostOrder(Queue q, Node n) {
    if (n != null) {
      if (debug) {
	System.out.println("Queue.traversePostOrder - Node data: " + n.data + " numChildren: " + n.children.size());
      }
      for (int i= 0; i < n.children.size(); i++) {
	traversePostOrder(q,n.children.getSibling(i));
      }
      q.append(n.data);
    }
    return q;
  }

  public String toString() {
    String out= "";
    int level= 0;
    int childrenToExpand= 0;
    Node curr= root;
    while (curr != null) {
      out+= makePad(level) + curr.toString() + "\n";
      if (childrenToExpand == 0) {
	childrenToExpand= curr.children.size();
	level++;
      }
      if (childrenToExpand > 0) {
	childrenToExpand--;
	curr= curr.parent == null ? curr : curr.parent;
	curr= curr.children.getSibling(childrenToExpand);
      }
      else {
	level--;
	curr= curr.parent;
      }
    }
    return out;
  }

  private String makePad(int length) {
    String pad= ""; 
    for (int i= 0; i < length; i++) {
      pad+= " ";
    } 
    return pad;
  }


}
