package rbs.sim;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;
import rbs.SimulationState;

public class ReactTimeStepSim extends TimeStepSim{
	private static final float COLLISION_ERROR = 0.05f;
	private static final float RESTING_ERROR = 0.05f;
	private static final float COEFFICIANT_OF_RESTITUTION = 1.0f;

	public ReactTimeStepSim(String fileName, int numTimeSteps,
		float timeStepSize, boolean pauseOnCollision, Vector3f gravity,
		boolean isGravityForce){
		super(fileName, numTimeSteps, timeStepSize, pauseOnCollision, gravity,
			isGravityForce);
	}

	private synchronized boolean handleSphereCollision(Sphere objA, 
			Sphere objB){
		logger.fine("SPHERE COLLISION");
		//compute the normal between the geometrical centers of the object 
		Vector3f normal = objA.normalBetween(objB);
		normal.normalize();
		logger.fine("Collision Normal = " + normal);

		//find the collision points
		Vector3f collisionPointA = new Vector3f();
		Vector3f collisionPointB = new Vector3f();
		collisionPointA.set(normal);
		collisionPointB.negate(normal);
		collisionPointA.scale(objA.getRadius());
		collisionPointB.scale(objB.getRadius());
		logger.fine("Collision point on object A" +
			"(relative to geometric center)= " + collisionPointA);
		logger.fine("Collision point on object B" +
			"(relative to geometric center)= " + collisionPointB);
		//find distance collision point is from center of mass
		collisionPointA.add(objA.getCenterOfMass());
		collisionPointB.add(objB.getCenterOfMass());
		logger.fine("Collision point on object A" +
			"(relative to center of mass)= " + collisionPointA);
		logger.fine("Collision point on object B" +
			"(relative to center of mass)= " + collisionPointB);

		//calculate relative velocity
		Vector3f relativeVelocity = new Vector3f(objA.getVelocity());
		relativeVelocity.sub(objB.getVelocity());
		logger.fine("Relative velocity = " + relativeVelocity);

		//calculate the impulse force J
		Vector3f j2 = new Vector3f();
		j2.cross(collisionPointA, normal);
		j2.scale(1.0f / objA.getMassMomentOfInertiaPitch());
		j2.cross(j2, collisionPointA);
		float j2d = normal.dot(j2);

		Vector3f j3 = new Vector3f();
		j3.cross(collisionPointB, normal);
		j3.scale(1.0f / objB.getMassMomentOfInertiaPitch());
		j3.cross(j3, collisionPointB);
		float j3d = normal.dot(j3);

		float j1d = (1.0f / objA.getMass()) + (1.0f / objB.getMass());

		Vector3f impulseForce = new Vector3f(relativeVelocity);
		impulseForce.negate();
		impulseForce.scale(COEFFICIANT_OF_RESTITUTION + 1.0f);
		impulseForce.scale(1.0f / (j1d + j2d + j3d));
		logger.fine("Impulse force = " + impulseForce);
		float impulseMagnitude = impulseForce.length();
		logger.fine("Magnitude of impulse = " + impulseMagnitude);

		//get the coefficiant of friction
		float cof = objA.getMaterial().getKineticCoefficiant(
			objB.getMaterial().getName());
		if(!SimulationState.getInstance().isFrictionEnabled()){
			cof = 0.0f;
		}
		logger.fine("Coefficiant of friction = " + cof);
		Vector3f tangentA = new Vector3f();
		tangentA.cross(normal, relativeVelocity);
		Vector3f tangent = new Vector3f();
		tangent.cross(tangentA, normal);
		tangent.normalize();
		logger.fine("Collision Tangent Vector = " + tangent);
		//compute force due to friction
		Vector3f frictionForce = new Vector3f(tangent);
		frictionForce.scale(impulseMagnitude * cof);

		Vector3f aAcceleration = new Vector3f(normal);
		aAcceleration.scale(impulseMagnitude);
		aAcceleration.add(frictionForce);
		aAcceleration.scale(1.0f / objA.getMass());
		logger.fine("Acceleration of object A = " + aAcceleration);

		Vector3f bAcceleration = new Vector3f(normal);
		bAcceleration.scale(-1.0f * impulseMagnitude);
		bAcceleration.add(frictionForce);
		bAcceleration.scale(1.0f / objB.getMass());
		logger.fine("Acceleration of object B = " + bAcceleration);

		Vector3f aAngularAcceleration = new Vector3f(normal);
		aAngularAcceleration.scale(impulseMagnitude);
		aAngularAcceleration.add(frictionForce);
		aAngularAcceleration.cross(collisionPointA, aAngularAcceleration);
		aAngularAcceleration.scale(1.0f / objA.getMassMomentOfInertiaPitch());
		logger.fine("Angular acceleration of object A = " + 
			aAngularAcceleration);

		Vector3f bAngularAcceleration = new Vector3f(normal);
		bAngularAcceleration.scale(-1.0f * impulseMagnitude);
		bAngularAcceleration.add(frictionForce);
		bAngularAcceleration.cross(collisionPointB, bAngularAcceleration);
		bAngularAcceleration.scale(1.0f / objB.getMassMomentOfInertiaPitch());
		logger.fine("Angular acceleration of object B = " + 
			bAngularAcceleration);
		
		//apply force at collision point
		objA.applyNonImpulseDirectAcceleration(aAcceleration, 
			aAngularAcceleration);
		objB.applyNonImpulseDirectAcceleration(bAcceleration,
			bAngularAcceleration);

		return true;
	}

	private synchronized COLLISION_RESP collideCheckSpheres(Object3D objA, 
		Object3D objB){
		//compute the distance between
		Point3f locA = objA.getLocation();
		Point3f locB = objB.getLocation();
		float dSquared = locA.distanceSquared(locB);
		logger.info("Distance between spheres is sqrt("+dSquared+")");
		float radiusA = ((Sphere)objA).getRadius();
		float radiusB = ((Sphere)objB).getRadius();
		//need to see if are overlapping or just touching
		float dTouching = (float)Math.pow(radiusA + radiusB, 2.0);
		if(dSquared - dTouching >= -COLLISION_ERROR && 
				dSquared - dTouching <= COLLISION_ERROR){
			logger.info("Spheres Touching");
			if(handleSphereCollision((Sphere)objA, (Sphere)objB))
				return COLLISION_RESP.HANDLED;
			else
				return COLLISION_RESP.NONE;
		}
		else if(dSquared < dTouching){ //the objects are overlapping
			logger.info("Collision Detected, Spheres Overlapping");
			return COLLISION_RESP.OVERLAP;
		}
		return COLLISION_RESP.NONE;
	}

	private synchronized boolean handlePlaneSphereCollision(Plane plane,
			Sphere sphere){
		//check the velocity of the sphere along the planes normal
		float rnv = plane.relativeVelocity(sphere, plane.getNormal());
		logger.fine("Relative normal velocity = " + rnv);

		if(Math.abs(rnv) <= RESTING_ERROR){
			logger.fine("Resting Contact between plane and sphere");
			//compute force to offset gravity
			//need to handle friction
			//figure out point on sphere where collision is
			Vector3f collisionPoint = new Vector3f(plane.getNormal());
			collisionPoint.negate();
			collisionPoint.scale(sphere.getRadius());
			logger.fine("Collision point on sphere" +
				"(relative to geometric center)= " + collisionPoint);

			//find relative velocity of collision point
			Vector3f sphereVelocity = new Vector3f();
			sphereVelocity.cross(sphere.getAngularVelocity(), collisionPoint);
			float cofFactor = 1.0f;
			boolean negate = true;
			if(!sphereVelocity.epsilonEquals(new Vector3f(), 0.01f)){
				cofFactor = cofFactor / 20.0f;
				negate = false;
			}
			sphereVelocity.add(sphere.getVelocity());
			logger.fine("Velocity of collision point = " + sphereVelocity);

			//calculate the tangent vector
			Vector3f tangent = new Vector3f();
			tangent.cross(plane.getNormal(), sphereVelocity);
			if(!tangent.epsilonEquals(new Vector3f(), 0.01f)){
				tangent.cross(tangent, plane.getNormal());
				tangent.normalize();

				//calculate friction magnitude
				float cof = sphere.getMaterial().getKineticCoefficiant(
					plane.getMaterial().getName());
				if(!SimulationState.getInstance().isFrictionEnabled()){
					cof = 0.0f;
				}
				else{
					cof = cof * cofFactor;
				}
				logger.fine("Coefficiant of friction = " + cof);
				Vector3f frictionForce = new Vector3f(plane.getNormal());
				float gravityMagnitude = 0.0f;
				if(this.isGravityForce()){
					Vector3f gravity = new Vector3f(this.getGravity());
					gravity.negate();
					gravityMagnitude = plane.getNormal().dot(gravity);
				}
				else{ //gravity is an acceleration
					Vector3f gravity = new Vector3f(this.getGravity());
					gravity.negate();
					gravity.scale(sphere.getMass());
					gravityMagnitude = plane.getNormal().dot(gravity);
				}
				frictionForce.scale(cof * gravityMagnitude);
				float frictionForceMagnitude = frictionForce.length();
				logger.fine("Magnitude of friction force = " + 
					frictionForceMagnitude);

				if(negate)
					tangent.negate();
				logger.fine("Tangent vector of sphere collision = " + tangent);
				sphere.applyNonImpulseForce(tangent, frictionForceMagnitude,
					new Point3f(collisionPoint));
			}

			Vector3f gravity = new Vector3f(this.getGravity());
			gravity.scale(-1.0f);
			if(this.isGravityForce()){
				sphere.applyNormalForce(gravity, new Point3f(),
					plane.getNormal());
			}
			else{
				sphere.applyNormalAcceleration(gravity, new Point3f(),
					plane.getNormal());
			}
			return false;
		}
		else if(rnv < 0.0f){
			logger.info("No Collision Objects moving away from each other");
			return false;
		}
		else{
			//we need to reflect velocity of sphere in direction of normal
			logger.info("Collision between sphere and plane");
			logger.info("PLANE COLLISION");

			//figure out point on sphere where collision is
			Vector3f collisionPoint = new Vector3f(plane.getNormal());
			collisionPoint.negate();
			collisionPoint.scale(sphere.getRadius());
			logger.fine("Collision point on sphere" +
				"(relative to geometric center)= " + collisionPoint);

			//find relative velocity of collision point
			Vector3f sphereVelocity = new Vector3f();
			sphereVelocity.cross(sphere.getAngularVelocity(), collisionPoint);
			float cofFactor = 1.0f;
			if(!sphereVelocity.epsilonEquals(new Vector3f(), 0.01f)){
				cofFactor = cofFactor / 1000.0f;
			}
			sphereVelocity.add(sphere.getVelocity());
			logger.fine("Velocity of collision point = " + sphereVelocity);

			//calculate the tangent vector
			Vector3f tangent = new Vector3f();
			tangent.cross(plane.getNormal(), sphereVelocity);
			if(!tangent.epsilonEquals(new Vector3f(), 0.01f)){
				tangent.cross(tangent, plane.getNormal());
				tangent.normalize();
				logger.fine("Tangent vector of sphere collision = " + tangent);

				//calculate friction magnitude
				float cof = sphere.getMaterial().getKineticCoefficiant(
					plane.getMaterial().getName());
				if(!SimulationState.getInstance().isFrictionEnabled()){
					cof = 0.0f;
				}
				else{
					cof = cof * cofFactor;
				}
				logger.fine("Coefficiant of friction = " + cof);
				Vector3f frictionForce = new Vector3f(plane.getNormal());
				float gravityMagnitude = 0.0f;
				if(this.isGravityForce()){
					Vector3f gravity = new Vector3f(this.getGravity());
					gravity.negate();
					gravityMagnitude = plane.getNormal().dot(gravity);
				}
				else{ //gravity is an acceleration
					Vector3f gravity = new Vector3f(this.getGravity());
					gravity.negate();
					gravity.scale(sphere.getMass());
					gravityMagnitude = plane.getNormal().dot(gravity);
				}
				frictionForce.scale(cof * gravityMagnitude);
				float frictionForceMagnitude = frictionForce.length();

				tangent.negate();
				sphere.applyNonImpulseForce(tangent, frictionForceMagnitude,
					new Point3f(collisionPoint));
			}

			float normVelocity = sphere.normalVelocity(plane.getNormal());
			logger.info("Velocity of sphere in direction of plane = " + 
				normVelocity);
			float magnitude = -2.0f * normVelocity;
			logger.info("Magnitude of impulse force = " + magnitude);
			logger.info("Force applied along = " + plane.getNormal());
			sphere.applyNonImpulseAcceleration(plane.getNormal(), 
				magnitude, new Point3f());

			//we also need to cancel out gravity for the next frame
			Vector3f gravity = new Vector3f(this.getGravity());
			gravity.scale(-1.0f);
			if(this.isGravityForce()){
				sphere.applyOneStepForce(gravity, new Point3f());
			}
			else{
				sphere.applyOneStepAcceleration(gravity, new Point3f());
			}

			return true;
		}
	}

	private synchronized COLLISION_RESP collideCheckPlaneSphere(Plane plane,
			Sphere sphere){
		Point3f planeLoc = plane.getLocation();
		Point3f sphereLoc = sphere.getLocation();
		Vector3f normalVector = new Vector3f();
		normalVector.sub(sphereLoc, planeLoc);
		float radius = sphere.getRadius();

		logger.fine("Plane Normal = " + plane.getNormal());
		//check to see if the sphere is on the plane
		logger.fine("Normal Vector = " + normalVector);
		float abovePlane = normalVector.dot(plane.getNormal());
		logger.fine("Sphere Distance Above Plane = " + abovePlane);
		float heightProj = normalVector.dot(plane.getHeightVector());
		float widthProj = normalVector.dot(plane.getWidthVector());
		logger.fine("Sphere position on plane = (" + 
			widthProj + ", " +  heightProj + ")");

		if(Math.abs(heightProj) <= (plane.getHeight() / 2.0f) &&
				Math.abs(widthProj) <= (plane.getWidth() / 2.0f)){

			if(Math.abs(abovePlane - radius) <= COLLISION_ERROR){	
				logger.info("Plane and Sphere Touching");
				if(handlePlaneSphereCollision(plane, sphere))
					return COLLISION_RESP.HANDLED;
				else
					return COLLISION_RESP.NONE;
			}
			else if(Math.abs(abovePlane) < radius){ //they are overlapping
				logger.info("Sphere and Plane overlapping");
				return COLLISION_RESP.OVERLAP;
			}
			else{
				logger.info("Sphere above or below plane");
				return COLLISION_RESP.NONE;
			}
		}
		else{
			logger.info("Sphere not above plane");
			return COLLISION_RESP.NONE;
		}
	}

	//If we find two objects overlapping return immediatly since there is no
	//	reason to check for other collisions we already know we need to rollback
	//If we find that we handled a collision that is what we will return but
	//	we need to process all of the others
	protected synchronized COLLISION_RESP collisionCheck(){
		Object3D[] objects = this.getObjects();
		COLLISION_RESP finalResult = COLLISION_RESP.NONE;

		for(int objA = 0; objA < this.getNumObjects(); objA++){
			for(int objB = objA + 1; objB < this.getNumObjects(); objB++){
				logger.info("Checking for collision between object " +
					objA + " and object " + objB);
				String typeA = objects[objA].getType();
				String typeB = objects[objB].getType();
				COLLISION_RESP result = COLLISION_RESP.NONE;
				//both spheres
				if(typeA.equals(Object3DFactory.SPHERE_TYPE) &&
						typeB.equals(Object3DFactory.SPHERE_TYPE)){
					result = collideCheckSpheres(objects[objA], objects[objB]);
				}
				//one sphere one plane
				else if(typeA.equals(Object3DFactory.PLANE_TYPE) &&
						typeB.equals(Object3DFactory.SPHERE_TYPE)){
					result = collideCheckPlaneSphere((Plane)objects[objA],
						(Sphere)objects[objB]);
				}
				else if(typeA.equals(Object3DFactory.SPHERE_TYPE) &&
						typeB.equals(Object3DFactory.PLANE_TYPE)){
					result = collideCheckPlaneSphere((Plane)objects[objB],
						(Sphere)objects[objA]);
				}

				if(result == COLLISION_RESP.OVERLAP)
					return result;
				else if(result == COLLISION_RESP.HANDLED)
					finalResult = result;
			}
		}
		return finalResult;
	}
}
