package rbs.sim;

import java.io.BufferedReader;
import java.io.IOException;
import java.lang.CloneNotSupportedException;
import java.util.logging.Logger;
import java.util.ArrayDeque;
import javax.vecmath.Vector3f;
import javax.vecmath.Point3f;

import rbs.render.Renderable;
import rbs.render.RenderableDecorator;

import rbs.SimulationState;

public abstract class Object3D implements Cloneable{
	private static Logger logger =
		Logger.getLogger(Object3D.class.getName());

	private String type;

	//mass properties
	private float mass;
	private Point3f massMomentOfInertia; //Ixx, Iyy, Izz
	private Point3f centerOfMass;  //with respect to center of model

	//translational properties
	private Vector3f velocity;
	private Vector3f acceleration;

	//rotational properties
	private Vector3f angularVelocity;	//big omega
	private Vector3f angularAcceleration; //small alpha

	//the physical representation of the object
	private Renderable model;

	//all of the forces currently acting on the object
	private ArrayDeque<Force> forces;

	//what is this object made out of
	private Material material;

	public Object3D(){
		SimulationState.getInstance().configLocalLogger(logger);

		mass = 0.0f;
		massMomentOfInertia = new Point3f();
		centerOfMass = new Point3f();

		velocity = new Vector3f();
		acceleration = new Vector3f();

		angularVelocity = new Vector3f();
		angularAcceleration = new Vector3f();

		forces = new ArrayDeque<Force>();

		model = null;
		material = null;
	}

	public Object3D(Renderable model){
		this();
		this.model = model;
	}

	public abstract void computeMassMomentOfInertia();

	public float normalTransAcceleration(Vector3f normal){
		return this.getCurrentTransAcceleration().dot(normal);
	}

	public float normalVelocity(Vector3f normal){
		return this.getVelocity().dot(normal);
	}

	public Vector3f normalBetween(Object3D obj){
		Vector3f normal = new Vector3f();
		normal.sub(this.getLocation(), obj.getLocation());
		normal.normalize();
		return normal;
	}

	public float relativeVelocity(Object3D obj){
		return this.relativeVelocity(obj, this.normalBetween(obj));
	}

	public float relativeVelocity(Object3D obj, Vector3f normal){
		float myRelativeVelocity = this.normalVelocity(normal);
		float yourRelativeVelocity = obj.normalVelocity(normal);
		return (myRelativeVelocity - yourRelativeVelocity);
	}

	public void update(float timeDelta){
		logger.fine("Original Acceleration: " + this.getAcceleration());
		logger.fine("Original Angular Acceleration: " + 
			this.getAngularAcceleration());
		logger.fine("Original Velocity: " + this.getVelocity());
		logger.fine("Original Angular Velocity: " + this.getAngularVelocity());
		logger.fine("Original Location: " + this.getLocation());
		logger.fine("Original Orientation: " + this.getOrientation());

		int numForces = forces.size();
		logger.fine("Total of " + numForces + " forces to be applied");
		for(int force = 0; force < numForces; force++){
			logger.fine("Applying force: " + force);
			Force f = forces.poll();
			if(!f.apply(timeDelta, this)){
				forces.addLast(f);
			}
			else{
				logger.fine("Reaping force " + force);
			}
		}

		//compute change in location
		//we need to scale current velocities by timeDelta
		Vector3f vel = new Vector3f(this.getVelocity());
		Vector3f avel = new Vector3f(this.getAngularVelocity());
		vel.scale(timeDelta);
		avel.scale(timeDelta);
		this.getLocation().add(vel);
		this.getOrientation().add(avel);
		logger.fine("New Location: " + this.getLocation());
		logger.fine("New Orientation: " + this.getOrientation());
	}

	protected static Logger getLogger(){
		return logger;
	}

	public Vector3f getCurrentTotalTransForce(){
		Vector3f totalForce = new Vector3f(this.getCurrentTransAcceleration());
		totalForce.scale(this.getMass());
		return totalForce;
	}

	//includes all involved forces
	public Vector3f getCurrentTransAcceleration(){
		Vector3f accel = new Vector3f();
		for(int force = 0; force < forces.size(); force++){
			Force f = forces.poll();
			accel.add(f.currentTransAcceleration());
			forces.addLast(f);
		}
		accel.add(this.getAcceleration());
		return accel;
	}

	//includes all involved forces
	public Vector3f getCurrentRotAcceleration(){
		Vector3f accel = new Vector3f();
		for(int force = 0; force < forces.size(); force++){
			Force f = forces.poll();
			accel.add(f.currentRotAcceleration());
			forces.addLast(f);
		}
		accel.add(this.getAngularAcceleration());
		return accel;
	}

	public Material getMaterial(){
		return this.material;
	}

	public String getType(){
		return this.type;
	}

	public void setType(String type){
		this.type = type;
	}

	//returns the center of mass in world coordinates
	public Point3f getCenterOfMassWorld(){
		Point3f locCOM = new Point3f(this.model.getLocation());
		locCOM.add(this.getCenterOfMass());
		return locCOM;
	}

	//returns the location of the object in world coordinates
	public Point3f getLocation(){
		return this.model.getLocation();
	}

	//set location in world coordinates
	public void setLocation(Point3f loc){
		this.model.setLocation(loc);
	}

	public Point3f getOrientation(){
		return this.model.getOrientation();
	}

	public void setOrientation(Point3f orientation){
		this.model.setOrientation(orientation);
	}

	public Renderable getCoreModel(){
		Renderable core = this.model;
		//check if current renderable is a decorator
		while(core.isDecorated()){
			core = ((RenderableDecorator)core).getCoreRenderable();
		}

		return core; 
	}

	public Renderable getModel(){
		return this.model;
	}

	public void setModel(Renderable model){
		this.model = model;
	}

	public float getMass(){
		return this.mass;
	}

	public void setMass(float mass){
		this.mass = mass;
	}

	//standard impulse forces
	public void applyImpulseForce(Vector3f force, Point3f origin,
			float duration){
		Force f = new Force(force, origin, duration, false);
		this.forces.addFirst(f);
	}
	public void applyImpulseForce(Vector3f forceNormal, float magnitude, 
			Point3f origin, float duration){
		Force f = new Force(forceNormal, magnitude, origin, duration, false);
		this.forces.addFirst(f);
	}
	public void applyImpulseAcceleration(Vector3f force, Point3f origin,
			float duration){
		Force f = new Force(force, origin, duration, true);
		this.forces.addFirst(f);
	}
	public void applyImpulseAcceleration(Vector3f forceNormal, float magnitude, 
			Point3f origin, float duration){
		Force f = new Force(forceNormal, magnitude, origin, duration, true);
		this.forces.addFirst(f);
	}

	//constant forces last for the entire duration of the simulation
	public void applyConstantForce(Vector3f force, Point3f origin){
		Force f = new ConstantForce(force, origin, false);
		this.forces.addFirst(f);
	}
	public void applyConstantForce(Vector3f forceNormal, float magnitude,
			Point3f origin){
		Force f = new ConstantForce(forceNormal, magnitude, origin, false);
		this.forces.addFirst(f);
	}
	public void applyConstantAcceleration(Vector3f force, Point3f origin){
		Force f = new ConstantForce(force, origin, true);
		this.forces.addFirst(f);
	}
	public void applyConstantAcceleration(Vector3f forceNormal, float magnitude,
			Point3f origin){
		Force f = new ConstantForce(forceNormal, magnitude, origin, true);
		this.forces.addFirst(f);
	}

	//non impulse force the time delta is always 1.0
	public void applyNonImpulseForce(Vector3f force, Point3f origin){
		Force f = new NonImpulseForce(force, origin, false);
		this.forces.addFirst(f);
	}
	public void applyNonImpulseForce(Vector3f forceNormal, float magnitude,
			Point3f origin){
		Force f = new NonImpulseForce(forceNormal, magnitude, origin, false);
		this.forces.addFirst(f);
	}
	public void applyNonImpulseAcceleration(Vector3f force, Point3f origin){
		Force f = new NonImpulseForce(force, origin, true);
		this.forces.addFirst(f);
	}
	public void applyNonImpulseAcceleration(Vector3f forceNormal, 
			float magnitude, Point3f origin){
		Force f = new NonImpulseForce(forceNormal, magnitude, origin, true);
		this.forces.addFirst(f);
	}

	public void applyNonImpulseDirectAcceleration(Vector3f transAcceleration,
			Vector3f rotAcceleration){
		Force f = new NonImpulseForce(transAcceleration, rotAcceleration);
		this.forces.addFirst(f);
	}

	//one step force only lasts for one time step
	public void applyOneStepForce(Vector3f force, Point3f origin){
		Force f = new OneStepForce(force, origin, false);
		this.forces.addFirst(f);
	}
	public void applyOneStepForce(Vector3f forceNormal, float magnitude,
			Point3f origin){
		Force f = new OneStepForce(forceNormal, magnitude, origin, false);
		this.forces.addFirst(f);
	}
	public void applyOneStepAcceleration(Vector3f force, Point3f origin){
		Force f = new OneStepForce(force, origin, true);
		this.forces.addFirst(f);
	}
	public void applyOneStepAcceleration(Vector3f forceNormal, 
			float magnitude, Point3f origin){
		Force f = new OneStepForce(forceNormal, magnitude, origin, true);
		this.forces.addFirst(f);
	}

	//normal force
	public void applyNormalForce(Vector3f force, Point3f origin, 
			Vector3f normal){
		Force f = new NormalForce(force, origin, normal, false);
		this.forces.addLast(f);
	}
	public void applyNormalAcceleration(Vector3f force, Point3f origin,
			Vector3f normal){
		Force f = new NormalForce(force, origin, normal, true);
		this.forces.addLast(f);
	}

	public void applyForce(Force force){
		this.forces.add(force);
	}

	public void applyNormalForce(Force force){
		this.forces.addLast(force);
	}

	public void applyForces(ArrayDeque<Force> newForces){
		this.forces = newForces;
	}

	public void applyGravity(Vector3f gravity, boolean isForce){
		if(this.getMass() > 0.0f){
			if(isForce){
				this.applyConstantForce(gravity, new Point3f());
			}
			else {
				this.applyConstantAcceleration(gravity, new Point3f());
			}
		}
	}

	public void setVelocity(Vector3f velocity){
		this.velocity = velocity;
	}

	public Vector3f getVelocity(){
		return this.velocity;
	}

	public float getVelocityX(){
		return this.velocity.x;
	}

	public float getVelocityY(){
		return this.velocity.y;
	}

	public float getVelocityZ(){
		return this.velocity.z;
	}

	public void setVelocityX(float x){
		this.velocity.x = x;
	}

	public void setVelocityY(float y){
		this.velocity.y = y;
	}

	public void setVelocityZ(float z){
		this.velocity.z = z;
	}

	public void setAngularVelocity(Vector3f velocity){
		this.angularVelocity = velocity;
	}

	public Vector3f getAngularVelocity(){
		return this.angularVelocity;
	}

	public float getAngularVelocityX(){
		return this.angularVelocity.x;
	}

	public float getAngularVelocityY(){
		return this.angularVelocity.y;
	}

	public float getAngularVelocityZ(){
		return this.angularVelocity.z;
	}

	public void setAngularVelocityX(float x){
		this.angularVelocity.x = x;
	}

	public void setAngularVelocityY(float y){
		this.angularVelocity.y = y;
	}

	public void setAngularVelocityZ(float z){
		this.angularVelocity.z = z;
	}

	public void setAngularAcceleration(Vector3f acceleration){
		this.angularAcceleration = acceleration;
	}

	public Vector3f getAngularAcceleration(){
		return this.angularAcceleration;
	}

	public float getAngularAccelerationX(){
		return this.angularAcceleration.x;
	}

	public float getAngularAccelerationY(){
		return this.angularAcceleration.y;
	}

	public float getAngularAccelerationZ(){
		return this.angularAcceleration.z;
	}

	public void setAngularAccelerationX(float x){
		this.angularAcceleration.x = x;
	}

	public void setAngularAccelerationY(float y){
		this.angularAcceleration.y = y;
	}

	public void setAngularAccelerationZ(float z){
		this.angularAcceleration.z = z;
	}

	public void setAcceleration(Vector3f acceleration){
		this.acceleration = acceleration;
	}

	public Vector3f getAcceleration(){
		return this.acceleration;
	}

	public float getAccelerationX(){
		return this.acceleration.x;
	}

	public float getAccelerationY(){
		return this.acceleration.y;
	}

	public float getAccelerationZ(){
		return this.acceleration.z;
	}

	public void setAccelerationX(float x){
		this.acceleration.x = x;
	}

	public void setAccelerationY(float y){
		this.acceleration.y = y;
	}

	public void setAccelerationZ(float z){
		this.acceleration.z = z;
	}

	public Point3f getCenterOfMass(){
		return this.centerOfMass;
	}

	public void setCenterOfMass(Point3f com){
		this.centerOfMass = com;
	}

	public void setCenterOfMassX(float x){
		this.centerOfMass.x = x;
	}

	public void setCenterOfMassY(float y){
		this.centerOfMass.y = y;
	}

	public void setCenterOfMassZ(float z){
		this.centerOfMass.z = z;
	}

	public void setMassMomentOfInertia(Point3f mmoi){
		this.massMomentOfInertia = mmoi;
	}

	public Point3f getMassMomentOfInertia(){
		return this.massMomentOfInertia;
	}

	public float getMassMomentOfInertiaPitch(){
		return this.massMomentOfInertia.x;
	}

	public float getMassMomentOfInertiaYaw(){
		return this.massMomentOfInertia.y;
	}

	public float getMassMomentOfInertiaRoll(){
		return this.massMomentOfInertia.z;
	}

	public void setMassMomentOfInertiaPitch(float pitchInertia){
		this.massMomentOfInertia.x = pitchInertia;
	}

	public void setMassMomentOfInertiaYaw(float yawInertia){
		this.massMomentOfInertia.y = yawInertia;
	}

	public void setMassMomentOfInertiaRoll(float rollInertia){
		this.massMomentOfInertia.z = rollInertia;
	}

	public ArrayDeque<Force> getForces(){
		return this.forces;
	}

	public ArrayDeque<Force> copyForces() throws CloneNotSupportedException{
		ArrayDeque<Force> copy = new ArrayDeque<Force>(this.forces.size());
		for(int force = 0; force < this.forces.size(); force++){
			Force f = this.forces.poll();
			copy.addLast((Force)f.clone());
			this.forces.addLast(f);
		}
		return copy;
	}

	@SuppressWarnings("unchecked")
	public Object clone() throws CloneNotSupportedException{
		Object3D copy = (Object3D) super.clone();

		copy.setModel((Renderable)this.getModel().clone());

		copy.setVelocity((Vector3f)this.getVelocity().clone());
		copy.setAngularVelocity((Vector3f)this.getAngularVelocity().clone());

		copy.setAcceleration((Vector3f)this.getAcceleration().clone());
		copy.setAngularAcceleration(
			(Vector3f)this.getAngularAcceleration().clone());

		copy.setCenterOfMass((Point3f)this.getCenterOfMass().clone());
		copy.setMassMomentOfInertia(
			(Point3f)this.getMassMomentOfInertia().clone());

		copy.applyForces(this.copyForces());

		return (Object)copy;
	}

	public void loadFromFile(BufferedReader input) throws IOException{
		this.getModel().loadFromFile(input);
		this.material = 
			MaterialStore.getInstance().getMaterial(input.readLine());
		this.setMass(Float.parseFloat(input.readLine()));

		this.setCenterOfMassX(Float.parseFloat(input.readLine()));
		this.setCenterOfMassY(Float.parseFloat(input.readLine()));
		this.setCenterOfMassZ(Float.parseFloat(input.readLine()));

		//get impulse vector
		Vector3f normal = new Vector3f();
		normal.x = Float.parseFloat(input.readLine());
		normal.y = Float.parseFloat(input.readLine());
		normal.z = Float.parseFloat(input.readLine());
		Point3f origin = new Point3f();
		origin.x += Float.parseFloat(input.readLine());
		origin.y += Float.parseFloat(input.readLine());
		origin.z += Float.parseFloat(input.readLine());
		float magnitude = Float.parseFloat(input.readLine());
		float forceDuration = Float.parseFloat(input.readLine());

		this.computeMassMomentOfInertia();

		//apply the force
		if(forceDuration == 0.0f){  //constant force
			this.applyConstantForce(normal, magnitude, origin);
		}
		else{
			this.applyImpulseForce(normal, magnitude, origin, forceDuration);
		}
	}
}
