/* Class RendererMain
 * Chris Van Horn
 * Singleton object which represents a rendering thread.
 */
package rbs.render;

import java.lang.Runnable;
import java.lang.Thread;
import java.lang.CloneNotSupportedException;
import java.lang.InterruptedException;

import java.util.logging.Logger;

import javax.vecmath.Point3f;

import rbs.SimulationState;
import rbs.reel.FrameReel;
import rbs.reel.StillFrame;

public abstract class RendererThread implements Runnable{
	private static final int MAX_RENDERERS = 4;
	public static Logger logger = Logger.getLogger(
		RendererThread.class.getName());

	private Thread rendererThread;
	private Renderer[] renderers;
	private int numRenderers;
	private FrameReel reel;

	public RendererThread(String threadName){
		renderers = new Renderer[MAX_RENDERERS];
		rendererThread = new Thread(this);
		rendererThread.setName(threadName);
		rendererThread.start();
		SimulationState.getInstance().configLocalLogger(logger);
		logger.info("New renderer thread created");
		logger.config("This renderer thread can have up to " 
			+ MAX_RENDERERS + " attached");
	}

	public synchronized void createWindows(int numRenderers, String[] types){
		this.numRenderers = numRenderers;

		if(this.numRenderers > MAX_RENDERERS)
			this.numRenderers = MAX_RENDERERS;

		for(int renderer = 0; renderer < this.numRenderers; renderer++){
			this.createWindow(renderer, types[renderer]);
		}
	}

	public synchronized void createWindow(int renderer, String type){
		if(rendererIndexCheck(renderer))
			renderers[renderer] = RendererFactory.newRenderer(type);
	}

	public synchronized void initializeAPIs(int x, int y, 
			int width, int height){
		for(int renderer = 0; renderer < this.numRenderers; renderer++){
			this.initializeAPI(renderer, x, y, width, height);
		}
	}

	public synchronized void initializeAPI(int renderer, int x, int y, int width, int height){
		if(rendererIndexCheck(renderer))
			renderers[renderer].initializeAPI(x, y, width, height);
	}

	public synchronized void setCameras(float[] x, float[] y, float[] z,
		float[] lax, float[] lay, float[] laz){

		//package up into Point3fs
		Point3f[] location = new Point3f[this.numRenderers];
		Point3f[] lookat = new Point3f[this.numRenderers];

		for(int renderer = 0; renderer < this.numRenderers; renderer++){
			location[renderer] = new Point3f(x[renderer], 
											 y[renderer],
											 z[renderer]);
			lookat[renderer] = new Point3f(lax[renderer],
										   lay[renderer],
										   laz[renderer]);
		}
		this.setCameras(location, lookat);
	}

	public synchronized void setCameras(Point3f[] location, Point3f[] lookat){
		for(int renderer = 0; renderer < this.numRenderers; renderer++){
			this.setCamera(renderer, location[renderer], lookat[renderer]);
		}
	}

	public synchronized void setCamera(int renderer, Point3f location, 
		Point3f lookAt){
		if(rendererIndexCheck(renderer)){
			renderers[renderer].setCamera(location, lookAt);
			logger.info("Renderer " + renderer + " camera set to location " +
				location.toString() + " looking at " + lookAt.toString());
		}
	}

	public synchronized void setFrustums(float[] frontLeft, float[] frontRight,
		float[] near, float[] far){
		for(int renderer = 0; renderer < this.numRenderers; renderer++){
			this.setFrustum(renderer, frontLeft[renderer], frontRight[renderer],
				near[renderer], far[renderer]);
		}
	}

	public synchronized void setFrustum(int renderer, float frontLeft, 
		float frontRight, float near, float far){
		if(rendererIndexCheck(renderer))
			renderers[renderer].setFrustum(frontLeft, frontRight, near, far);
	}

	public synchronized void openWindows(String[] names){
		for(int renderer = 0; renderer < this.numRenderers; renderer++){
			this.openWindow(renderer, names[renderer]);
		}
	}
	
	public synchronized void openWindow(int renderer, String name){
		if(rendererIndexCheck(renderer))
			renderers[renderer].openWindow(name);
	}

	public synchronized void initRenderers(){

	}

	public synchronized void initRenderer(int renderer){

	}

	public synchronized void closeWindows(){
		for(int renderer = 0; renderer < this.numRenderers; renderer++){
			this.closeWindow(renderer);
		}
	}

	public synchronized void closeWindow(int renderer){
		if(rendererIndexCheck(renderer))
			renderers[renderer].closeWindow();
	}

	public synchronized void destroyWindow(int renderer){
		if(rendererIndexCheck(renderer))
			renderers[renderer].destroyWindow();
	}

	public synchronized void destroyWindows(){
		for(int renderer = 0; renderer < this.numRenderers; renderer++){
			this.destroyWindow(renderer);
		}
	}

	//when rendering the renderer may be null, if try to render before
	//window created
	public synchronized void renderWindow(int renderer, StillFrame frame){
		if(rendererIndexCheck(renderer) && renderers[renderer] != null)
			renderers[renderer].render(frame);
	}

	public synchronized void renderWindows(StillFrame frame){
		for(int renderer = 0; renderer < this.numRenderers; renderer++){
			this.renderWindow(renderer, frame);
		}
	}

	public synchronized void shutdownRenderers(){
		for(int renderer = 0; renderer < this.numRenderers; renderer++){
			this.shutdownRenderer(renderer);
		}
	}

	public synchronized void shutdownRenderer(int renderer){
		if(rendererIndexCheck(renderer)){
			this.closeWindow(renderer);
			this.destroyWindow(renderer);
		}
	}

	public synchronized void setReel(FrameReel reel){
		this.reel = reel;
	}

	public synchronized FrameReel getCurrentReel(){
		return this.reel;
	}


	private boolean rendererIndexCheck(int renderer){
		if(renderer >= 0 && renderer < this.numRenderers){
			return true;
		}
		else{
			logger.warning("Renderer index, " + renderer + ",out of range");
			return false;
		}
	}

	public Object clone() throws CloneNotSupportedException {
		throw new CloneNotSupportedException();
	}
}
