package rbs.reel;

import java.io.Serializable;
import java.io.FileOutputStream;
import java.io.FileInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.lang.NullPointerException;
import java.lang.IndexOutOfBoundsException;
import java.lang.ClassNotFoundException;

import rbs.ui.SerializableReel;

public class BufferedFrameReel 
		implements FrameReel, SerializableReel, Serializable{
	private ArrayList<StillFrame> film;
	private int cursor;

	public BufferedFrameReel(){
		film = new ArrayList<StillFrame>();
		cursor = 0;
	}

	public synchronized boolean pushFrame(StillFrame frame){
		try{
			return film.add(frame);
		} catch(NullPointerException e){
			System.out.println("WARNING: Attempt to add null frame to reel");
			return false;
		}
	}

	public synchronized boolean isBuffered(){
		return true;
	}

	public synchronized void seekBegin(){
		cursor = 0;
	}

	public synchronized void seekEnd(){
		cursor = film.size() - 1;
	}

	//time is relative
	//advance frames until end time greater than currentEndTime + time
	public synchronized void seekForward(float time){
		//convert current end time to milliseconds
		StillFrame currentFrame = this.getCurrentFrame();
		if(currentFrame == null) //no reason in going further
			return;

		float currentEnd = currentFrame.getEndTime();
		float targetTime = currentEnd + time;

		while(targetTime > currentFrame.getEndTime()){
			if(this.getFrame(cursor + 1) == null)
				return;
			cursor++;
			currentFrame = this.getCurrentFrame();
		}
	}

	public synchronized void seekBackward(float time){
		StillFrame currentFrame = this.getCurrentFrame();
		if(currentFrame == null){
			int tempCursor = cursor;
			while(currentFrame == null){
				if(cursor == 0)
					break;
				cursor--;
				currentFrame = this.getCurrentFrame();
			}
		}

		float currentEnd = currentFrame.getEndTime();
		float targetTime = currentEnd - time;

		//start rolling back
		while(targetTime < currentFrame.getEndTime()){
			if(cursor == 0) //we rolled back as far as possible
				return;
			cursor--;
			currentFrame = this.getCurrentFrame();
		}
	}

	public synchronized StillFrame popFrame(){
		StillFrame frame = this.getFrame(cursor);
		if(frame != null)
			cursor++;
		//TODO: if run to end of reel and still more coming consider pausing
		return frame;
	}

	public synchronized int reelLength(){
		return film.size();
	}

	public synchronized boolean insertFrame(int index, StillFrame frame){
		try{
			film.add(index, frame);
			return true;
		} catch(NullPointerException e){
			System.out.println("WARNING: Attempt to add null frame to reel");
			return false;
		} catch(IndexOutOfBoundsException e){
			System.out.print("WARNING: Attempt to add frame");
			System.out.println(" at invalid position on reel");
			return false;
		}
	}

	public synchronized StillFrame getCurrentFrame(){
		return this.getFrame(cursor);
	}

	public synchronized StillFrame getFrame(int index){
		try{
			return film.get(index);
		} catch(IndexOutOfBoundsException e){
			return null;
		}
	}

	public synchronized StillFrame removeFrame(int index){
		try{
			return film.remove(index);
		} catch(IndexOutOfBoundsException e){
			return null;
		}
	}

	public synchronized void save(String filename){
		filename = "canisters/" + filename + ".reel";
		try{
			FileOutputStream fos = new FileOutputStream(filename);
			ObjectOutputStream oos = new ObjectOutputStream(fos);
			oos.writeObject(this);
		} catch(FileNotFoundException e){
			System.out.println("Couldn't write reel");
		} catch(IOException e){
			System.out.println("Couldn't write reel");
		}
	}

	public synchronized void load(String filename){
		filename = "canisters/" + filename + ".reel";
		try{
			FileInputStream fis = new FileInputStream(filename);
			ObjectInputStream ois = new ObjectInputStream(fis);
			BufferedFrameReel fileReel = (BufferedFrameReel)ois.readObject();
			this.setFilm(fileReel.getFilm());
			this.cursor = 0;
		} catch(FileNotFoundException e){
			System.out.println("Couldn't read reel");
		} catch(IOException e){
			System.out.println("Couldn't read reel");
		} catch(ClassNotFoundException e){
			System.out.println("Couldn't read reel");
		}
	}

	private synchronized ArrayList<StillFrame> getFilm(){
		return this.film;
	}

	private synchronized void setFilm(ArrayList<StillFrame> film){
		this.film = film;
	}
}
