package main

import (
	"os"
	"fmt"
	"nsutil"
	"saxutil"
	"strings"
	"strconv"
	"os/exec"
	"bitbucket.org/binet/go-gnuplot/pkg/gnuplot"
)

func pre_name(host, property string) (name string) {
	name = host+property
	name = strings.Replace(name, " ", "", -1)
	name = strings.Replace(name, "\t", "", -1)
	return
}

func main() {

	if len(os.Args) <  6 {
		fmt.Fprintf(os.Stderr,
		"Usage: %s <config file> (-f <filename> | -d <host> <property>) <width> <height>\n",
		os.Args[0])
		os.Exit(0)
	}

	config, err := nsutil.Read_config(os.Args[1])

	if err != nil {
		fmt.Fprintf(os.Stderr, "Error reading configuration file.\n")
	}

	var values []float64
	var times []int
	var host string
	var property string
	var width, height int64

	switch (os.Args[2]) {
		case "-d":
			if len(os.Args) > 4 {
				host = os.Args[3]
				property = os.Args[4]
				rows, res, err := nsutil.Get_data(os.Args[3], os.Args[4], config)
				if err != nil {
					fmt.Fprintf(os.Stderr,
						"Could not get data from database for %s:%s\n",
						os.Args[3], os.Args[4])
					os.Exit(0)
				}

				values, times = saxutil.Arrays_from_rows(rows, res)

			} else {
				fmt.Fprintf(os.Stderr,
					"Option -d requires a host and a property.\n")
				os.Exit(0)
			}
			width, _ = strconv.ParseInt(os.Args[5], 10, 64)
			height, _ = strconv.ParseInt(os.Args[6], 10, 64)

		case "-f":
			if len(os.Args) > 3 {
				host = os.Args[3]
				values, times, _ = saxutil.Arrays_from_file(os.Args[3])
			} else {
				fmt.Fprintf(os.Stderr, "Option -f requires a file name.\n")
				os.Exit(0)
			}
			width, _ = strconv.ParseInt(os.Args[4], 10, 64)
			height, _ = strconv.ParseInt(os.Args[5], 10, 64)

		default:
			fmt.Fprintf(os.Stderr, "Unrecognized option %s\n", os.Args[2])
			os.Exit(0)
	} // switch(os.Args)


	fname := fmt.Sprintf("%s", pre_name(host, property))

	fout, err := os.Create(fname+".dat")
	if err != nil {
		fmt.Fprintf(os.Stderr, "Could not create %s.dat\n", fname)
		os.Exit(1)
	}

	for i := 0; i < len(values); i++ {

		fmt.Fprintf(fout, "%f\t%d\n", values[i], times[i])
	}

	fout.Close()

	plotter, err := gnuplot.NewPlotter("", true, false)
	if err != nil {
		fmt.Fprintf(os.Stderr, "Error getting plotter.\n")
		fmt.Fprintf(os.Stderr, "Exiting...\n")
		os.Exit(0)
	}

	plotter.CheckedCmd("set terminal png small size %d,%d", width, height)
	plotter.CheckedCmd("set size 1,1")
	plotter.CheckedCmd("set xdata time")
	plotter.CheckedCmd("set timefmt \"%%s\"")
	plotter.CheckedCmd("set format x \"%%b %%d %%R\"")

	err = plotter.SetLabels("Time", "Sample value")
	plotter.CheckedCmd("set output \"%s.png\"", fname)
	plotter.CheckedCmd("plot \"%s.dat\" u 2:1 t 'values' w lines lw 2", fname)

	plotter.Close()

	err = exec.Command("sync").Run()

	path, err := exec.LookPath("convert")
	args := []string{"convert"}
	args = append(args, "-scale")
	args = append(args, "100x50")
	args = append(args, fmt.Sprintf("%s.png", fname))
	args = append(args, fname+"-small.png")

	if config.Verbose {
		fmt.Fprintf(os.Stderr, "Args:\n")
		for _, arg := range args {
			fmt.Fprintf(os.Stderr, "\t%s\n", arg)
		}
	}

	var procAttr os.ProcAttr
	procAttr.Files = []*os.File{nil, os.Stdout, os.Stderr}

	process, err := os.StartProcess(path, args, &procAttr)
	if err != nil {
		fmt.Fprintf(os.Stderr, "Start process failed %v\n", err)
	}
	_, err = process.Wait()
	if err != nil {
		fmt.Fprintf(os.Stderr, "Wait failed: %v\n", err)
	}
} // main
