/* fh 		This  module handles the file-handle cache.
 *              FILE HANDLE PACKAGE FOR USER-LEVEL NFS SERVER
 *
 *              Interfaces:
 *                  pseudo_inode
 *                      mostly used internally, but also called from unfsd.c
 *                      when reporting directory contents.
 *                  fh_init
 *                      Initializes the queues and 'flush' timer
 *                  fh_pr
 *                      debugging primitive; converts file handle into a
 *                      printable text string
 *                  fh_create
 *                      establishes initial file handle; called from mount
 *                      daemon
 *                  fh_path
 *                      returns unix path corresponding to fh
 *                  fh_fd
 *                      returns open file descriptor for given file handle;
 *                      provides caching of open files
 *                  fd_idle
 *                      provides mututal exclusion of normal file descriptor
 *                      cache use, and alarm-driven cache flushing
 *                  fh_compose
 *                      construct new file handle from existing file handle
 *                      and directory entry
 *                  fh_psi
 *                      returns pseudo_inode corresponding to file handle
 *                  fh_remove (new, by Don Becker)
 *                      delete the file handle associated with PATH from the
 *                      cache
 *
 * Authors:     Mark A. Shand, May 1988
 *              Donald J. Becker <becker@super.org>
 *              Rick Sladkey <jrs@world.std.com>
 *              Patrick Sweeney <pjs@raster.Kodak.COM>
 *              Orest Zborowski <obz@raster.Kodak.COM>
 *              Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *              Olaf Kirch, <okir@monad.swb.de>
 *
 *              Copyright 1988 Mark A. Shand
 *              This software maybe be used for any purpose provided
 *              the above copyright notice is retained.  It is supplied
 *              as is, with no warranty expressed or implied.
 */

#include "mount.h"
#include "nfs_prot.h"
#include "fh.h"
#include <sys/stat.h>
#include <linux/errno.h>
#include <linux/unistd.h>
#include <dirent.h>


/* hash the psudo inode */

#define hash_psi(psi) (((psi)^((psi)>>8)^((psi)>>16)^((psi)>>24)) & 0xff)

extern int open (const char *pathname, int flags,
					mode_t mode);

extern char * xstrdup (const char *string);

static mutex                    ex_state = inactive;
static mutex                    io_state = inactive;

/* initialise the file-handle cache */

#define HASH_TAB_SIZE           256
static fhcache                  fh_head, fh_tail;
static fhcache *                fh_hashed[HASH_TAB_SIZE];
static fhcache *                fd_lru_head = NULL;
static fhcache *                fd_lru_tail = NULL;
static int                      fh_list_size;
static time_t                   curtime;


static fhcache *                fd_cache[FOPEN_MAX] = { NULL };
static int                      fd_cache_size = 0;


int  re_export = 0;

static int
fh_flush_fds(void);

int fh_create( fhandle3 *fh, char *arg);

static void
fh_unlink_fdcache(fhcache *fhc);

extern int nfsmounted(const char *path, struct  stat *sbp);
psi_t pseudo_inode(ino_t inode, dev_t dev);

static psi_t 
path_psi (char *path, nfsstat3 *status, struct stat *sbp, struct  stat *tsbp);

/*
 * Authors: Mark A Shand et al.
 */

struct {
        enum nfsstat3            error;
        int                     nfs_errno;
} nfs_errtbl[]= {
        { NFS3_OK,               0               },
        { NFS3ERR_IO,            5               },
        { NFS3ERR_PERM,          1               },  
        { NFS3ERR_NOENT,         2               },
        { NFS3ERR_IO,            5               },
        { NFS3ERR_NXIO,          6               },
        { NFS3ERR_ACCES,         13              },
        { NFS3ERR_EXIST,         17              },
        { NFS3ERR_NODEV,         19              },
        { NFS3ERR_NOTDIR,        20              },
        { NFS3ERR_ISDIR ,        21              }, 
        { NFS3ERR_INVAL,         22              },
        { NFS3ERR_FBIG,          27              }, 
        { NFS3ERR_NOSPC,         28              },
        { NFS3ERR_ROFS,          30              },
        { NFS3ERR_NAMETOOLONG,   63              },
        { NFS3ERR_NOTEMPTY,      66              },
        { NFS3ERR_STALE,         70              },
};


/* forward declared local functions */

static void fh_insert_fdcache(fhcache *fhc);

psi_t
fh_psi(nfs_fh3 *fh)
{
        svc_fh *h = (svc_fh *) fh;
        return (h->psi);
}

/* Lookup an NFS error code and return UNIX equivalent. */ 
enum nfsstat3
nfs_errno(void)
{
        int i;
  
        for (i = 0; nfs_errtbl[i].error != -1; i++) {
                if (nfs_errtbl[i].nfs_errno == errno)
                        return (nfs_errtbl[i].error);
        }
        return (NFS3ERR_IO);
}



static void
fh_insert_fdcache(fhcache *fhc)
{
        if (fhc == fd_lru_head)
                return;
        if (fhc->fd_next || fhc->fd_prev)
                fh_unlink_fdcache(fhc);
        if (fd_lru_head)
                fd_lru_head->fd_prev = fhc;
        else
                fd_lru_tail = fhc;
        fhc->fd_next = fd_lru_head;
        fd_lru_head = fhc;
        fd_cache[fhc->fd] = fhc;
        fd_cache_size++;
}

static void
fh_unlink_fdcache(fhcache *fhc)
{
        fhcache *prev = fhc->fd_prev,
                *next = fhc->fd_next;

        fhc->fd_next = fhc->fd_prev = NULL;
        if (next) {
                next->fd_prev = prev;
        } else if (fd_lru_tail == fhc) {
                fd_lru_tail = prev;
        } 
        if (prev) {
                prev->fd_next = next;
        } else if (fd_lru_head == fhc) {
                fd_lru_head = next;
        } 

        fd_cache[fhc->fd] = NULL;
        fd_cache_size--;
}

static char *fh_buildpath(h)
svc_fh *h;
{
    char        pathbuf[PATH_MAX + NAME_MAX + 1], *path;
    long        cookie_stack[HP_LEN + 1];
    char        *slash_stack[HP_LEN];
    struct stat sbuf;
    psi_t               psi;
    int  	i;


  
     // printf("inside buildpath\n");
     // printf("hplen is %d\n", HP_LEN);
     // printf("is h-hash[0] is %d\n", h->hash_path[0]);
     // printf("is h-hash[0] is %d\n", h->hash_path[1]);
     // printf("hpsi is %d\n", h->psi);
    if (h->hash_path[0] >= HP_LEN) {
         // printf("greater than hlpen\n");
        return NULL;
    }  
    if (stat("/", &sbuf) < 0)
        return (NULL);
    psi = pseudo_inode(sbuf.st_ino, sbuf.st_dev);
    if (h->hash_path[0] == 0) {
        if (psi != h->psi)
         return (NULL);
        return xstrdup("/");
    }

     // printf("after psi \n");
   
    if (hash_psi(psi) != h->hash_path[1])
        return (NULL);  
    /* auth_override_uid(ROOT_UID);  */       
    strcpy(pathbuf, "/");
    cookie_stack[2] = 0;
    for (i = 2; i <= h->hash_path[0] + 1; i++) {
        DIR *dir;
        struct dirent *dp;
        
        // printf("inside main build path loop\n");
    backtrack:
        if (stat(pathbuf, &sbuf) >= 0
            && (dir = opendir(pathbuf)) != NULL) {
               // printf("inside opendir\n");
         if (cookie_stack[i] != 0)
                seekdir(dir, cookie_stack[i]);
         while ((dp = readdir(dir))) {
                if (strcmp(dp->d_name, ".") != 0
                    && strcmp(dp->d_name, "..") != 0) {
                        psi = pseudo_inode(dp->d_ino, sbuf.st_dev);
    if (i == h->hash_path[0] + 1) {
                                  // printf("entered i == 2\n");
                                if (psi == h->psi) {
                                        //  printf("inside got\n");
                                        /*GOT IT*/
                                        strcat(pathbuf, dp->d_name);
                                        path = xstrdup(pathbuf);
                                        
                                        closedir(dir);
                                      //   auth_override_uid(auth_u
                                        return (path);
                                }
      } else {
                                if (hash_psi(psi) == h->hash_path[i]) {
                                        // printf("else got\n");
                                        /* PERHAPS WE'VE GOT IT */
                                        cookie_stack[i] = telldir(dir);
                                        cookie_stack[i + 1] = 0;
                                        slash_stack[i] = pathbuf + strlen(pathbuf);
                                        strcpy(slash_stack[i], dp->d_name);
                                        strcat(pathbuf, "/");

                                        closedir(dir);
                                        goto deeper;
                                }
                        }
                }
         }
          /* dp == NULL ;  */
         closedir(dir);
        }
        i--;
        if (i < 2) {
         // auth_override_uid(auth_uid);
         return (NULL);   /* SEARCH EXHAUSTED  */
        } 

        /* Prune path */
        *(slash_stack[i]) = '\0';
        goto backtrack;
    deeper:
        ;
    }
 //   auth_override_uid(auth_uid);
      // printf("reached end\n");
     return (NULL);              /* actually not reached */
}




static fhcache *
fh_lookup(psi_t psi)
{
        register fhcache *fhc;

	// printf("INSIDE fh_lookup\n"); 
        fhc = fh_hashed[psi % HASH_TAB_SIZE];
        while (fhc != NULL && fhc->h.psi != psi)
                fhc = fhc->hash_next;

        return (fhc);
}
static void
fh_inserthead(fhcache *fhc)
{
       register fhcache **hash_slot;

       // printf("inside insert\n");        
       /* Insert at head. */
       fhc->prev = &fh_head;
       fhc->next = fh_head.next;
 
       
       // printf("after insert \n");
        fhc->prev->next = fhc;

        // printf("after insert\n");
         fhc->next->prev = fhc;  

        // printf("after insert\n");
       fh_list_size++;

       // printf("after insert0\n");
       /* Insert into hash tab. */
       hash_slot = &(fh_hashed[fhc->h.psi % HASH_TAB_SIZE]);

        // printf("after insert1\n");
       fhc->hash_next = *hash_slot;

        // printf("after insert2\n");
       *hash_slot = fhc;
        // printf("after insert\n");
}


static void
fh_close(fhcache *fhc)
{
        if (fhc->fd >= 0) {
                fh_unlink_fdcache(fhc); 
                close(fhc->fd);
                fhc->fd = -1;
        }
}

                


static void
fh_delete(fhcache *fhc)
{
        register fhcache **hash_slot;
        
	// printf("inside delete\n");        
        
        /* Remove from current posn */
        fhc->prev->next = fhc->next;
        fhc->next->prev = fhc->prev;
        fh_list_size--;  
	// printf("before hash tab\n");
        /* Remove from hash tab */
        hash_slot = &(fh_hashed[fhc->h.psi % HASH_TAB_SIZE]);
         while (*hash_slot != NULL && *hash_slot != fhc)
                hash_slot = &((*hash_slot)->hash_next);
        if (*hash_slot == NULL)
	      {}
        else
                *hash_slot = fhc->hash_next;
        
        fh_close(fhc);
        /* Free storage. */
        if (fhc->path != NULL) 
                free(fhc->path);
                        
  	// printf("after delete\n"); 
        free(fhc);
}

/* fh_remove */
void
fh_remove(char *path)
{
        psi_t   psi;
        nfsstat3 status;
        fhcache *fhc;

        psi = path_psi(path, &status, NULL, NULL);
        if (psi == 0)
                return;
        ex_state = active;
        fhc = fh_lookup(psi);
        if (fhc != NULL)
                fh_delete(fhc);

        ex_state = inactive;
        return;
}



static void
fh_move_to_front(fhcache *fhc)
{       
        /* Remove from current posn */
	// printf("inside move\n");
        fhc->prev->next = fhc->next;
        fhc->next->prev = fhc->prev;
	
	// printf("after remove\n"); 
        /* Insert at head */
        fhc->prev = &fh_head;
        fhc->next = fh_head.next;
        fhc->prev->next = fhc;
        fhc->next->prev = fhc;   
        // printf("end of move\n");	
}  

fhcache *
fh_find(svc_fh *h, int mode)
{
        register fhcache *fhc, *flush;
        char            buff[1024], *sp;   


         // printf("entering fh_find\n");
        // printf("val0 is %d\n", h->hash_path[0]);
        // printf("val1 is %d\n", h->hash_path[1]);          

        sprintf(buff, "fh_find: psi=%lx... ", (unsigned long) h->psi);
        // printf("after sprintf\n");
        sp = buff + strlen(buff);
        ex_state = active;
        time(&curtime);
        while ((fhc = fh_lookup(h->psi)) != NULL) {
		// printf("after fh_lookup\n");
		// printf("Path us %d\n", fhc->h.hash_path);
		// printf("Hash path is %d\n", h->hash_path);
                if (memcmp(h->hash_path, fhc->h.hash_path, HP_LEN) != 0) {
			// printf("after memcmp\n");
                        if (mode != FHFIND_FCREATE && mode !=
				FHFIND_FEXISTS) {
                                ex_state = inactive;
                                return (NULL);
    					}
			// printf("before delete\n");
                        fh_delete(fhc);
                        break;
                            }
		// printf("before move_to_front\n");
                if (fhc != fh_head.next)
                        fh_move_to_front(fhc);
                    fhc->last_used = curtime;
                    ex_state = inactive;
                    return (fhc);
               }
         // printf("before findcached\n");               
        if (mode == FHFIND_FCACHED) {
                ex_state = inactive;
                return NULL;
        }
                
        for (flush = fh_tail.prev; fh_list_size > FH_CACHE_LIMIT; flush =
fhc) {   /* Don't flush current head. */
                if (flush == &fh_head)
                        break;
                fhc = flush->prev;
                fh_delete(flush);
        }
        fhc = (fhcache *) malloc(sizeof *fhc);
        // printf("before FHFINDCREATE\n");
        if (mode == FHFIND_FCREATE) {
                /* File will be created */
                // printf("mode is fhcreate\n"); 
                fhc->path = NULL;
 } else {
                /* File must exist. Attempt to construct from hash_path */
                char *path;
                
                // printf("********ENTERING file EXISTS\n");                 
                if ((path = fh_buildpath(h)) == NULL) {
                        free(fhc);
                        ex_state = inactive;
                        return NULL;
                }
                // printf("after buildpath\n");
                fhc->path = path;
        }
        fhc->flags = 0;
        if (fhc->path) {
                struct stat     stb;

                if (stat(fhc->path, &stb) >= 0) {
 if (re_export && nfsmounted(fhc->path, &stb))
                                fhc->flags |= FHC_NFSMOUNTED;
                }
        }
        // printf("after fhc->path check\n");
        fhc->fd = -1;
        fhc->last_used = curtime;
        fhc->h = *h;
        fhc->last_clnt = NULL;
        fhc->last_mount = NULL;
        fhc->last_uid = (uid_t)-1;
        fhc->fd_next = fhc->fd_prev = NULL;
        fh_inserthead(fhc); 
        ex_state = inactive;
        // printf("after fhcreation\n");

        return (fhc);
}


/* unix path corresponding to fh */

char *
fh_path(nfs_fh3 *fh, nfsstat3 *status)
{
        fhcache *h;

        if ((h = fh_find((svc_fh *) fh, FHFIND_FEXISTS)) == NULL) {
                *status = NFS3ERR_STALE;
                return (NULL);
        }
        *status = NFS3_OK;
        return (h->path);
}


psi_t
pseudo_inode(ino_t inode, dev_t dev)
{  
        psi_t           dmajor, dminor;
 
        /*
         * Assuming major and minor numbers are small integers,
         * gravitate bits of dmajor & dminor device number to
         * high-order bits of word, to avoid clash with real inode num.   
         */
        /* reverse (byte-wise) */
        dmajor = ((dev & 0xf0f) << 4) | ((dev & 0xf0f0) >> 4);
        dmajor = ((dmajor & 0x3333) << 2) | ((dmajor & 0xcccc) >> 2);
        dmajor = ((dmajor & 0x5555) << 1) | ((dmajor & 0xaaaa) >> 1);
         
        /* spread low-16 -> 32 with 0's in even posn */
        dmajor = ((dmajor & 0xff00) << 8) | (dmajor & 0xff);
        dmajor = ((dmajor & 0xf000f0) << 4) | (dmajor & 0xf000f);
        dmajor = ((dmajor & 0xc0c0c0c) << 2) | (dmajor & 0x3030303);
        dmajor = ((dmajor & 0x22222222) << 1) | (dmajor & 0x11111111);
        dminor = (dmajor & 0x5555) << 15;
        dmajor = dmajor & 0x55550000;
        
        return ((dmajor | dminor) ^ inode);
}
        


static psi_t 
path_psi (char *path, nfsstat3 *status, struct stat *sbp, struct  stat *tsbp)
{
 
 struct stat sbuf; 
	
 // printf("PPSI path is %s\n", path);        
 if (sbp == NULL)
                sbp = &sbuf;

 if (lstat(path, sbp) < 0) {
		// printf("PPSI lstat fails\n");
		perror("lstat fails");
                *status = nfs_errno();
                return (0);
  } 
  // printf("PPSI after lstat\n");
  if (tsbp)
                *tsbp = *sbp;   
        if (S_ISDIR(sbp->st_mode) && strcmp(path, "/") != 0) {
                /* Special case for directories--test for mount point. */
                struct stat ddbuf;
                char *sindx;
                char *fname;
                char squirrel;
              /* Find start of last component of path. */
                if ((sindx = strrchr(path, '/')) == path) {
                        sindx++;
                        fname = sindx;
                } else
                        fname = sindx + 1;
                
                /* Remove last element of path. */
                squirrel = *sindx;
                *sindx = '\0';

                if (lstat(path, &ddbuf) < 0) {
                        *sindx = squirrel;
                        *status = nfs_errno();
                        return (0);   
                } 
                /* Sindx now points to directory entry name. */
                if (ddbuf.st_dev != sbp->st_dev) {
                        /* Directory is a mount point. */
                        DIR *dirp;
                        struct dirent *dp;
                        errno = 0;
                        if ((dirp = opendir(path)) == NULL) {
                                *sindx = squirrel;      /* restore path */
                                if (errno == EACCES)
                                        goto unreadable;
                                if (errno != 0)
                                        *status = nfs_errno();
                                else
 					{
					// printf("PPSI noent case\n"); 
                                        *status = NFS3ERR_NOENT;
					}
                        } else {
                                *sindx = squirrel;      /* restore path */
                                *status = NFS3_OK;
                                do {
                                        if ((dp = readdir(dirp)) == NULL)
                                                {
                                                *status = NFS3ERR_NOENT;
                                                closedir(dirp);
                                                return (0);
                                        }
                                } while (strcmp(fname, dp->d_name) != 0);
                                sbp->st_dev = ddbuf.st_dev;
                                sbp->st_ino = dp->d_ino;
                                closedir(dirp);  
                        }
                } else
                        *sindx = squirrel;      /* restore path */
        unreadable:
                ;
        }
        return (pseudo_inode(sbp->st_ino, sbp->st_dev));

}


int fh_create( fhandle3 *fh, char *path) {
   svc_fh 	key;
   fhcache 	*h; 
   psi_t 	psi;
   nfsstat3 	status;  
   char 	*s;
 
     
   memset((void *)&key, 0, sizeof(key));
   status = NFS3_OK;

   if((psi = path_psi("/", &status, NULL, NULL)) == 0)
	return ((int) status);
   // printf("psi is %d\n", psi); 
    
    s = path;
    // printf("FHCREATE init val: %d\n", key.hash_path[0]);
    
    // printf("hash len val is %d\n", HP_LEN);
    while ((s = strchr(s + 1, '/')) != NULL) { 
                if (++(key.hash_path[0]) >= HP_LEN)
                        return ((int) NFS3ERR_NAMETOOLONG);
                key.hash_path[key.hash_path[0]] = hash_psi(psi);
                /* printf("hashed values %d\n", 
			key.hash_path[key.hash_path[0]]); */
                *s = '\0';
                if ((psi = path_psi(path, &status, NULL, NULL)) == 0)
                        return ((int) status);
          *s = '/';
    }
      if (*(strrchr(path, '/') + 1) != '\0') {
                if (++(key.hash_path[0]) >= HP_LEN)
                        return ((int) NFS3ERR_NAMETOOLONG);
                key.hash_path[key.hash_path[0]] = hash_psi(psi);
                /* printf("hashed values %d\n", key.hash_path[key.hash_path[0]]); */
                if ((psi = path_psi(path, &status, NULL, NULL)) == 0)
                        return ((int) status);
    }
        key.psi = psi;
        // printf("%d\n",key.psi);
        // printf("before fh_find\n");
        h = fh_find(&key, FHFIND_FCREATE);
        // printf("after fh_find\n");
                
        /* assert(h != NULL); */
        if (h->path == NULL) {
                h->fd = -1;
            h->path = xstrdup(path);  
                h->flags = 0;
     }
	fh->fhandle3_len = sizeof(key);   
				/* incorporate the variable file handle */
        memcpy(fh->fhandle3_val, &key, sizeof(key));
        return ((int) status);
}


nfsstat3 
fh_compose (diropargs3 *dopa, nfs_fh3 *new_fh, struct stat **sbpp, 
	  		int fd, int omode, int public) {
	svc_fh  *key;
	fhcache *dirh, *h;
	char *sindx;
	int is_dd;
	nfsstat3 ret;
	struct stat tsbuf;
	char pathbuf[PATH_MAX + NAME_MAX + 1], *fname;

	if ((dirh = fh_find((svc_fh *)dopa->dir.data.data_val, FHFIND_FEXISTS)) == NULL)
 		return NFS3ERR_STALE;	
	
	// printf("FHC after fh_find\n");

	// not public file handle.
	if (strchr(dopa->name, '/') != NULL)
		return NFS3ERR_ACCES;
	fname = dopa->name;

        // printf("fname is %s\n", fname);
        /* 
	construct path
	*/
	
	if (strcmp(fname, ".") == 0 || fname[0] == '\0') { 
		new_fh->data.data_len = dopa->dir.data.data_len;
	 	new_fh->data.data_val = dopa->dir.data.data_val;
		*sbpp = NULL;	
		return (NFS3_OK);
	}
        
         if (strcmp(fname, "..") == 0) {
                is_dd = 1;
                sindx = strrchr(dirh->path, '/');
                if (sindx == dirh->path)
                        strcpy(pathbuf, "/");
                else {
                        int len = sindx - dirh->path;
                        strncpy(pathbuf, dirh->path, len);
                        pathbuf[len] = '\0';
                }
                  
	 } else { /* not .. */
                int len = strlen(dirh->path);
                is_dd = 0;

		// printf("FHC else case\n");
                if (len && dirh->path[len - 1] == '/')
                        len--;
                strncpy(pathbuf, dirh->path, len);
                pathbuf[len] = '/';
                strcpy(pathbuf + (len + 1), fname);
        }
	
        new_fh->data.data_len = dopa->dir.data.data_len;
        new_fh->data.data_val = dopa->dir.data.data_val;
        key = (svc_fh *) new_fh->data.data_val;
        if ((key->psi = path_psi(pathbuf, &ret, *sbpp, &tsbuf)) == 0)
                return (ret);

	// printf("FHC after psi\n");
        if (++(key->hash_path[0]) >= HP_LEN)
                        return NFS3ERR_NAMETOOLONG;
                key->hash_path[key->hash_path[0]] = hash_psi(dirh->h.psi);
       
        h = fh_find(key, FHFIND_FCREATE);
	
        if (h->path == 0) {
             h->path = xstrdup(pathbuf);
             h->flags = 0;
  	 }        
        if (fd >= 0) {
                if (h->fd >= 0)
                        fh_close(h);
                h->fd = fd;
                fh_insert_fdcache(h);
        }
        if (omode >= 0)
                h->omode = omode;

        return (NFS3_OK);	
}


int
path_open(char *path, int omode, int perm)
{
 int fd; 
 int oerrno, ok;
 struct stat buf;

 fh_flush_fds();
 
 // printf("inside path_open \n");
 // printf("path is %s\n", path);
 
 /* If the file exists, make sure it is a regular file. Opening
  * device files might hang the server. There's still a tiny window
  * here, but it's not very likely someone's able to exploit
  * this.
  */
 if(( ok = lstat(path, &buf) >= 0) && !S_ISREG(buf.st_mode)) { 
	// printf("not reg case\n");
 	errno = EISDIR;
	return -1; 
        } 
        // printf("PO: mode is %d\n", omode);
	// printf("PO: perm is %d\n", perm);
	// printf("PO: path is %s\n", path);  
        fd = open(path, omode, perm);
	// printf("fd is %d\n", fd);
        // printf("errno is %d\n", errno); 
        // printf("after lstat ok is %d\n", ok);

        oerrno = errno;
 
        /* The file must exist at this point */
 
        if (!ok && lstat(path, &buf) < 0) { 
        errno = oerrno;
        return -1;
        } 
        
 return(fd);
}

int
fh_fd(fhcache *h, nfsstat3 *status, int omode)
{
        int     retry;
	
	errno = 0;


        // printf("inside fh_fd \n");
        // printf("path is %s\n", h->path);
	if(!h->path) {
		*status = NFS3ERR_STALE;
		return(-1);
		}
	 
        for (retry = 0; retry < 2; retry++) {
         if ((h->fd = path_open(h->path, omode, 0)) >= 0) {
		 // printf("inside if cond\n");
                 io_state = active;
                 h->omode = omode;
                 fh_insert_fdcache(h);
                 return (h->fd);
         }
	 // printf("else case\n");

         *status = nfs_errno();
         if (errno != ENOENT)
         return -1;
          
         /* maybe the cached path is stale */

         free(h->path);
	 h->path = NULL;
	 if(!retry && !fh_buildpath(&h->h))
	 break;
         }
	 return -1;
	
}

void 
fd_inactive (int fd)
{
   io_state = inactive;	
}


static int
fh_flush_fds(void)
{

   if (io_state == active) {

   return(-1);
   }

   while (fd_cache_size >= FD_CACHE_LIMIT)
	fh_close(fd_lru_tail);
   return (0);
}


/*
 * fh_flush
 

void
fh_flush(int force)
{
        register fhcache *h;
 
        if (ex_state == inactive) {
            int cache_size = 0;
 
         ex_state = active;
         time(&curtime);
          Single execution thread 
        
          works in empty case because: fh_tail.next = &fh_tail    
         h = fh_head.next;
         while (h != &fh_tail) {
             if (cache_size > FH_CACHE_LIMIT
                 || curtime > h->last_used + DISCARD_INTERVAL
                 || force) {
                     h = h->next;
                     fh_delete(h->prev);

		 } else {
                     if (h->fd >= 0 &&
                     curtime > h->last_used +
				CLOSE_INTERVAL) 
                     fh_close(h);
                     cache_size++;
                     h = h->next;
                 }
           }

 	if (fh_list_size != cache_size) {
		
	 }
      	fh_list_size = cache_size;
     	ex_state = inactive;
        }
}  */

/* RETSIGTYPE
flush_cache(int sig)
{
        static volatile int     inprogress = 0;
        
        signal (SIGALRM, flush_cache);
        if (_rpcsvcdirty) {
                alarm(BUSY_RETRY_INTERVAL);
                need_flush = 1;
                return;
        } 
        if (inprogress++)
                return;
  	fh_flush(0);
        if (_rpcpmstart)
                rpc_closedown();
        inprogress = 0;
        need_flush = 0;
        alarm(FLUSH_INTERVAL); 
} */



void
fh_init(void)
{
        static int      initialized = 0;

        if (initialized)
                return;
        initialized = 1;

        fh_head.next = fh_tail.next = &fh_tail;
        fh_head.prev = fh_tail.prev = &fh_head;
      //  signal(SIGALRM, flush_cache);   
      //  alarm(FLUSH_INTERVAL);
        
        umask(0);
}



/****************************************************************************/ 


