/*
 * fh.h		This module handles the file-handle cache.
 *
 * Authors:	Mark A. Shand, May 1988
 *		Don Becker, <becker@super.org>
 *		Rick Sladkey, <jrs@world.std.com>
 *		Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *
 *		Copyright 1988 Mark A. Shand
 *		This software maybe be used for any purpose provided
 *		the above copyright notice is retained.  It is supplied
 *		as is, with no warranty expressed or implied.
 */

#include <linux/types.h>

#ifndef FH_H
#define FH_H

#define FHC_NFSMOUNTED          004


enum {
	FHFIND_FEXISTS = 0,	/* file must exist */
	FHFIND_FCREATE,		/* file will be created */
	FHFIND_FCACHED,		/* fh must be in cache */
};

/*
 * This defines the maximum number of handles nfsd will cache.
 */
#define	FH_CACHE_LIMIT		2000

/*
 * This defines the maximum number of file nfsd may keep open for NFS I/O.
 * It used to be 8...
 */
#define FD_CACHE_LIMIT		(3*FOPEN_MAX/4)

#define FLUSH_INTERVAL		5			/* 5 seconds	*/
#define BUSY_RETRY_INTERVAL	2			/* 2 seconds	*/
#define CLOSE_INTERVAL		5			/* 5 seconds	*/
#define DISCARD_INTERVAL	(60*60)			/* 1 hour	*/

/*
 * Type of a pseudo inode
 */

typedef __u32		psi_t;

/*
 * Hashed search path to this file.
 * path is: hash_path[1] ... hash_path[hash_path[0]]
 *
 * hash_path[hash_path[0]+1] ... hash_path[HP_LEN-1] == 0
 */
#define	HP_LEN		(NFS3_FHSIZE - sizeof(psi_t))
typedef struct {
	psi_t		psi;
	__u8		hash_path[HP_LEN];
} svc_fh;

typedef enum { inactive, active } mutex;

/*
 * Paths constructed in this system always consist of real directories
 * (excepting the last element) i.e. they do not contain symbolic links.
 * This is guaranteed by the way NFS constructs the paths.
 * As a consequence we may assume that
 *	/x/y/z/.. == /x/y
 * and	/x/y/z/. == /x/y/z
 * provided that z != . && z != ..
 * These relations are exploited in fh_compose.
 *
 * Further assumptions:
 *	All cached pathnames consist of a leading /
 *	followed by zero or more / separated names
 *	s.t.
 *		name != .
 *		name != ..
 *		index(name, '/') == 0
 */
extern struct nfs_client *      nfsclient;
extern struct nfs_mount *       nfsmount;

typedef struct nfs_mount {
        struct nfs_mount* next;
        struct nfs_client* client;
        int      length;
        char *    path;
         
        }nfs_mount;

typedef struct nfs_client {
        struct nfs_client *  next;
        struct in_addr  clnt_addr;
        struct in_addr  clnt_mask;
        char *    clnt_name;
        unsigned short  flags;
        nfs_mount *  m;
        }nfs_client; 
typedef struct fhcache {
	struct fhcache *	next;
	struct fhcache *	prev;
	struct fhcache *	hash_next;
	struct fhcache *	fd_next;
	struct fhcache *	fd_prev;
	svc_fh			h;
	int			fd;
	int			omode;
	char *			path;
        time_t                  last_used;
        nfs_client *              last_clnt; 
        nfs_mount *             last_mount;
	uid_t			last_uid;
	int			flags;
} fhcache;  


/* Global FH variables. 
extern int			_rpcpmstart;
extern int			fh_initialized;

Global function prototypes. 
extern nfsstat	nfs_errno(void);
extern psi_t	pseudo_inode(ino_t inode, dev_t dev);  */

#endif FH_H
