/***********   

    Write code for showAugleaves function at line 1730.

************/




/**
*    Name:   Theory.java
*    Author: Lei Shen
*    Date:   Feb. 16, 1999
*    Modified by: Suresh Thesayi
*    Modification date:  Jan 20, 2000 
*/
import java.util.*;
import java.io.*;
import java.lang.reflect.*;

/*
 *   This is the main part of the SDatr program for performming
 *   all kinds of query operations on the theory we got by parser.
 */
class Theory extends Observable{

  /** sent output to all the registered observser
    */
  public void handleOutput(String string){
     System.out.print(string);
     setChanged();
     notifyObservers(string);
  }

  /** sent output to all the registered observser
    */
  public void handleOutputln(String string){
     System.out.println(string);
     setChanged();
     notifyObservers(string+"\n");
  }

  /** Printing the result of query. This removes all escape characters.
    */
  public void print(String string){
     int length = string.length();
     char chArray[] = (char[]) string.toCharArray();
     StringBuffer newString = new StringBuffer();
     for(int i = 0; i < length; i++){
        if(chArray[i] != '\\')
           newString.append(String.valueOf(chArray[i]));
        else
           newString.append(String.valueOf(chArray[++i]));
     }
     System.out.print(newString.toString());
     setChanged();
     notifyObservers(newString.toString()+"\n");
  }

  /**
  *   Decide what kind of debug information should be printed 
  *   0: no debug information at all
  *   1: show steps while parsing the system
  *   2: show steps while trying to resolve a query
  *   3: show steps while trying to have a match
  *
  *   theoryFlag: set to true if theory or leaves command is executed
  */
  private       long lDebugLevel;
  private       boolean bWarnDuplicate  = false;
  private       boolean theoryFlag      = false;
  static public long DEBUG_SHOW_NOTHING = 0;
  static public long DEBUG_SHOW_PATH    = 1;
  static public long DEBUG_SHOW_TRACE   = 2;
  static public long DEBUG_SHOW_MATCH   = 3;

  /*---------------------------------------------------------------------*/
  /**
  *  nameTable      :name table for every atom or node in the theory
  *  orgNameTable   :name table created during compile or recompile operation
  *                  Augment operation will not affect this table. The 
  *                  difference of leaves in nameTable and orgNameTable is
  *                  the newly augmented leaves. This table is required
  *                  to be maintained because of the augleaves command.
  *  nodeTable      :name table for every node in the theory
  *  varsTable      :all variables defined in this theory
  *  allShowVector  :all the show and showif statements defined in the theory
  *  hideTable      :nodes to hide (given by #hide statements)
  *  hideifTable    :nodes to hide (evaluated from #hideif statement)
  *  hideifVector   :store lhs and rhs of equality testing conditions 
  *                  given in #hideif statements
  *  hideifNotVector:store lhs and rhs of inequality testing conditions 
  *                  given in #hideif statements
  */
  private  Hashtable nameTable; 
  private  Hashtable orgNameTable; 
  private  Hashtable nodeTable; 
  private  Hashtable varsTable;
  private  Vector    allShowVector;
  private  Vector    hideifVector;
  private  Vector    hideifNotVector;
  private  Hashtable hideTable;
  private  Hashtable hideifTable;

  /*---------------------------------------------------------------------*/
  /**
  *  During Query
  *  MAXRECURSIVEDEPTH :all query should finish within this number of steps
  *  mapTable          :the corresponding variable-value table
  *  globalNode        :current global node in query
  *  globalPath        :current global path in query
  */
  private  long        MAXRECURSIVEDEPTH;
  private  Hashtable   mapTable;
  private  String      globalNode;
  private  Vector      globalPath;

  /*---------------------------------------------------------------------*/
  /**
  *  identifierTable:  Hash table for all variables/identifiers except leaves
  *                    (Leaves are Nodes that are not inherited)
  *  unUsedidentTable: Hash table for all identifiers appearing only once
  *  termTable      :  Hash table for all terms 
  *  varmapTable    :  Hash Table for all mapping from variable to atom
  */
  private  Hashtable  identifierTable;
  private  Hashtable  unUsedIdentTable;
  private  Hashtable  termTable;
  private  Hashtable  varmapTable;

  /*---------------------------------------------------------------------*/
  /**
  *   Theory()     : constructor
  *   clearTheory(): clear the current theory in system
  *   clearQuery() : clear current query in system
  */
  Theory(){
     nameTable       = new Hashtable();
     nodeTable       = new Hashtable();
     varsTable       = new Hashtable();
     allShowVector   = new Vector();
     hideifVector    = new Vector();
     hideifNotVector = new Vector();
     hideTable       = new Hashtable();
     hideifTable     = new Hashtable();

     identifierTable = new Hashtable();
     unUsedIdentTable= new Hashtable();
     termTable       = new Hashtable();
     varmapTable     = new Hashtable();

     lDebugLevel       = DEBUG_SHOW_NOTHING;
     theoryFlag        = false;
     MAXRECURSIVEDEPTH = 1000;
     mapTable          = new Hashtable();
     globalPath        = null;
     globalNode        = null;
  }

  public void clearTheory(){
     nameTable.clear();
     nodeTable.clear();
     varsTable.clear();
     allShowVector.removeAllElements();
     hideifVector.removeAllElements();
     hideifNotVector.removeAllElements();
     hideTable.clear();
     hideifTable.clear();

     identifierTable.clear();
     unUsedIdentTable.clear();
     termTable.clear();
     varmapTable.clear();

     System.gc();
  }

  public void clearQuery(){
     globalPath = null;
     globalNode = null;
     mapTable.clear();
     System.gc();
  }


  /**
    *  if an identifer has appeared before then
    *  we try to use the same memory for storing it and
    *  also mark it as true. Any identifier stored in the 
    *  identifierTable is first marked with false. 
    *  identifierTable will be later checked to 
    *  see if there are any identifier with false marking. If
    *  there are some than it means that these identifiers 
    *  appear only once in the program and hence may be 
    *  regarded as typo error. This error checking routine
    *  is done in function postProcess.
    */
  public String lookupIdentifier(String strToLookUp){
     String strResult = (String)identifierTable.get(strToLookUp);
     if(strResult == null){
         identifierTable.put(strToLookUp, strToLookUp);
         strResult = strToLookUp;
     }
     return strResult;
  }

  /* check unused identifiers */
  public String checkUnUsedID(String strToLookUp, int line, int column){
     String strResult = (String)identifierTable.get(strToLookUp);
     if(strResult == null){
         identifierTable.put(strToLookUp, strToLookUp);
         unUsedIdentTable.put(strToLookUp, "("+String.valueOf(line)+
                                           ","+String.valueOf(column)+")");
         strResult = strToLookUp;
     }
     else
         unUsedIdentTable.remove(strToLookUp);
     return strResult;
  }

  /**
    * return identifier
    */
  public String getIdentifier(String strToLookUp){
     String strResult = (String)identifierTable.get(strToLookUp);
     return strResult;
  }

  /**
    *  if a term has already appeared before, 
    *  we try to use the same memory for storing it.
    */
  public Term lookupTerm(Term termToLookUp){
     String  strKey = termToLookUp.toString();
     Term    termResult = (Term)termTable.get(strKey);
     if(termResult == null){
         termTable.put(strKey, termToLookUp);
         termResult = termToLookUp;
     }
     return termResult;
  }

  /**
    *  var:     variable name
    *  map:     one possible value
    *  add a quick checking for var-map mapping, store var+#map as key, 
    */
  public void addVarMap(String var, String map){
     varmapTable.put(var+"+#"+map, map);
  }

  public void deleteVarMap(String var, String map) {
     varmapTable.remove(var+"+#"+map);
  }

  /**
    *  strToLoopUp:  var+#map key
    *  return the map if any, otherwise null
    */
  public String lookupVarMap(String strToLookUp){
     return (String)varmapTable.get(strToLookUp); 
  } 

  /**
    *  some thing related to name table
    */
  public Hashtable getNameTable(){
     return nameTable;
  }

  /**
    *   clones nameTable to orgNameTable
    */
  public void copyNameTable(){
     orgNameTable = (Hashtable) nameTable.clone();
  }

  /**
  *   Add a Node Name to nodeTable
  */
  public int addToNodeTable(String strName) {
     int result = 0;
     if(nodeTable.get(strName) != null)
        result = -1;
     else
        nodeTable.put(strName,strName);
     return result;
  }	

  /**
  *   check whether the string given is a valid Node name
  *   return -1:  string exist in name table, but not Node type
  *   return -2:  regular node name must begin with A-Z
  *   return  0:  either in defined as node before or legal regular
  */
  public int checkNodeName(String strName){
     int  result = 0;
     EleInNameTable ele = (EleInNameTable) nameTable.get(strName);

     if(ele != null){
        if(!ele.isNode())
           result = -1;
     }

     else if(strName.charAt(0) < 'A' || strName.charAt(0) > 'Z')
          result = -2;

     else 
          nameTable.put(strName, new EleInNameTable(strName, "NODE"));
     return result;
  }

  /**
  *   whether the string given is a Node name in name table
  */
  private boolean isNodeInNameTable(String strName){
     boolean result = false;
     if(nameTable.containsKey(strName)){
         EleInNameTable ele = (EleInNameTable) nameTable.get(strName);
         result = ele.isNode();
     }
     return result;
  }

  /**
  *   whether the string given is an Atom name in name table
  */
  private boolean isAtomInNameTable(String strName){
     boolean result = false;
     if(nameTable.containsKey(strName)){
        EleInNameTable  ele = (EleInNameTable) nameTable.get(strName);
        result = ele.isAtom();
     }
     return result;
  }
  
  public void addToHideTable(String name){
     hideTable.put(name, name);
  }

  public void addToHideifTable(String name){
     hideifTable.put(name, name);
  }

  public void addToHideifVector(Vector lhsVecAtoms, Vector rhsVecAtoms){
     hideifVector.addElement(lhsVecAtoms);
     hideifVector.addElement(rhsVecAtoms);
  }

  public void addToHideifNotVector(Vector lhsVecAtoms, Vector rhsVecAtoms){
     hideifNotVector.addElement(lhsVecAtoms);
     hideifNotVector.addElement(rhsVecAtoms);
  }

  public void addToAllShowVector(IfThenElse curIf){
     allShowVector.addElement(curIf);
  }

  public Hashtable getVarsTable(){
     return varsTable;
  }

  public void setWarnDuplicate(boolean value){
     bWarnDuplicate = value;
  }

  public void setDebugLevel(long level){
     lDebugLevel = level;
  }

  public long getDebugLevel(){
     return lDebugLevel;
  }


  /**
  *    print out the query 
  */
  private void printQuery4DoQuery(String strTitle, long lDepth, 
                                   String strNode, Vector vecAtoms,  
                                   String localNode, Vector localPath,
                                   String gNode, Vector gPath) {
     if(lDebugLevel == DEBUG_SHOW_TRACE){
         handleOutput(strTitle + lDepth + " : " + strNode);
         if(vecAtoms != null){
           handleOutput(":< ");
           for(int i = 0; i < vecAtoms.size(); i ++){
                handleOutput(vecAtoms.elementAt(i).toString() + " ");
           }
           handleOutput(">\n");
         }
         else
           handleOutput("\n");
         handleOutputln("Local :" + localNode + ";" + localPath);
         handleOutputln("Global:" + gNode + ";" + gPath);
         handleOutput(toStringMapTable(mapTable));
         handleOutputln("");
     }
  }

  private void findAppendPath(boolean bSet,  boolean bNotSuppressed,
                              int iMaxMatchLength, Vector appendPath, 
                              Vector srcAtoms, Vector repSrcAtoms,
                              Vector vecAtoms){
       if(bSet){
          String       str, strOrig, repStrOrig;
          Integer      value;
          Hashtable    table = new Hashtable();
          Hashtable    map = new Hashtable();
          Vector       vecVal;
          for(int i = 0; i < srcAtoms.size(); i ++){
             strOrig = (String)srcAtoms.elementAt(i);
             repStrOrig = (String)repSrcAtoms.elementAt(i);

             /*strOrig may be an element (generally variable) that has been 
               made distinct with '#' symbol (for example; $abc#1). All 
               elements are stored in strOrig and str stores elements after 
               stripping strOrig of the '#name',if present. */
             if(strOrig.indexOf('#') != -1)
                str = strOrig.substring(0,strOrig.indexOf('#'));
             else
                str = strOrig;

             if(varsTable.containsKey(str)) //to handle variables
                if(mapTable.containsKey(strOrig))
                   str = (String) ((Vector) 
                               mapTable.get(strOrig)).firstElement(); 
             value = (Integer) table.put(str, new Integer(1));
             if(value != null)
                table.put(str, new Integer(value.intValue()+1));
             
             if(!strOrig.equals(repStrOrig)){
                vecVal = (Vector) map.get(str);
                if(vecVal == null)
                   vecVal = new Vector();
                vecVal.add(repStrOrig);
                map.put(str, vecVal);
             }
          }

          if(bNotSuppressed){
             String oldstr;
             for(int i = 0; i < vecAtoms.size(); i ++){
                str = (String) vecAtoms.elementAt(i);
                vecVal = (Vector) map.remove(str);
                if(vecVal != null){
                   oldstr = str;
                   str = (String)vecVal.remove(0);
                   if(vecVal.size() != 0)
                      map.put(oldstr, vecVal);
                }
                appendPath.addElement(str);
             }
          }
          else{
             for(int i = 0; i < vecAtoms.size(); i ++){
                str = (String) vecAtoms.elementAt(i);
                value = (Integer) table.remove(str);
                if(value == null)
                   appendPath.addElement(str);
                else if(value.intValue() > 1)
                   table.put(str, new Integer(value.intValue()-1));
             }
          }
       }
       else{
           if(bNotSuppressed){
              for(int i = 0; i < iMaxMatchLength; i++){
                 if(!srcAtoms.elementAt(i).equals(repSrcAtoms.elementAt(i)))
                    appendPath.addElement(repSrcAtoms.elementAt(i));
                 else
                    appendPath.addElement(vecAtoms.elementAt(i));
              }
           }

           int  isize = vecAtoms.size();
           for(int i = iMaxMatchLength; i < isize; i ++)
              appendPath.addElement(vecAtoms.elementAt(i));
       }
  }
  
  
  /**
  *   boolean   bGlobal:   indicate whether it is a global query or local query
  *   String    strNode:   the Node Name indicated in the query if any
  *   Vector    vecAtoms:  the Path indicated in the query if any
  *   Vector    vecResult: where the result should stored
  *   long      lDepth:    the recursion called depth
  *   String    localNode: the local variable for Node 
  *   Vector    localPath: the local variable for Path
  *   String    gNode:     the global variable for Node
  *   Vector    gPath:     the global variable for Path
  */
  public int doQuery(
      boolean bGlobal, String strNode, Vector vecAtoms, Vector vecResult, 
      long lDepth, String localNode, Vector localPath, String gNode,   
      Vector gPath) throws ParseException {

       if(lDepth >= MAXRECURSIVEDEPTH){
           handleOutputln("Recursion too deep in evaluating QUERY");
           return 0;
       }

       Hashtable mapTableCopy = (Hashtable) mapTable.clone();
       mapTable.clear();

       printQuery4DoQuery("Before Level: ", lDepth, strNode,
                           vecAtoms, localNode, localPath, gNode, gPath);

       //populate the query
       if(vecAtoms == null)
          if(bGlobal)
             vecAtoms = (Vector) gPath.clone();
          else
             vecAtoms = (Vector) localPath.clone();

       if(strNode == null)
          if(bGlobal) 
             strNode = gNode; 
          else
             strNode = localNode;

       //update local or global node path
       localNode = strNode;
       localPath = (Vector)vecAtoms.clone();

       if(bGlobal){
          gNode = strNode;
          gPath = (Vector) vecAtoms.clone();
       }

       if(lDepth == 0){
           globalNode = strNode;
           globalPath = (Vector)vecAtoms.clone();
           gNode      = globalNode;
           gPath      = (Vector)globalPath.clone();
       }

       printQuery4DoQuery("After Level: ", lDepth, strNode,
                          vecAtoms, localNode, localPath, gNode, gPath);

       EleInNameTable ele = (EleInNameTable) nameTable.get(strNode);
       if(ele == null || !ele.isNode()){
          handleOutputln(toStringQuery(strNode, vecAtoms, vecResult));
          handleOutputln("The node " + strNode + " is not part of the theory.");
          return 0;
       }

       //try to find a match from all rules in current Node
       Rule  curRule;
       int   iCurMatchLength;
       Rule  longestMatch = null;
       int   iMaxMatchLength = -1;
       int   iMaxPossibleMatch = vecAtoms.size();

       Hashtable longestMapTable = (Hashtable)mapTable.clone();
       Vector   longestMatchedRule = new Vector();
       boolean  duplicate = false;
       Vector vecRules = (Vector) ele.getVecRules();
       int ruleIndex = 0;
       for(int i = 0; i < vecRules.size(); i++) {
            curRule = (Rule)vecRules.elementAt(i);
            mapTable.clear();
            iCurMatchLength = FindMatch(curRule.isSet(),
                                        curRule.getVecAtoms(), vecAtoms);
            if(iCurMatchLength >= 0 && iCurMatchLength >= iMaxMatchLength){
                if(iCurMatchLength == iMaxMatchLength){
                   longestMatchedRule.addElement(curRule);
                   duplicate = true;
                }
                else{
                   longestMatchedRule.removeAllElements();
                   longestMatchedRule.addElement(curRule);
                   duplicate = false;
                }

                longestMatch    = curRule;
                ruleIndex = i;
                longestMapTable = (Hashtable)mapTable.clone();
                iMaxMatchLength = iCurMatchLength;

                //if(iMaxMatchLength == iMaxPossibleMatch)
                //   break;
            }
       }
       
       mapTable = (Hashtable)longestMapTable.clone();
       //printout necessary message
       if(longestMatch == null)
           return 0;

       else{
          if(lDebugLevel == DEBUG_SHOW_TRACE){
             handleOutput("Match: " + longestMatch);
             handleOutput(toStringMapTable(mapTable));
          }
          ele.setMatchFlagAt(ruleIndex);
       }

       //warning if conflict in rules
       if(duplicate && bWarnDuplicate){
           handleOutput("More than one maximal match for node " +
                        strNode + " with local query <");
           for(int i = 0; i < vecAtoms.size(); i ++){
               handleOutput(" " + vecAtoms.elementAt(i).toString());
           }
           handleOutputln(" >");
           for(int i = 0; i < longestMatchedRule.size(); i ++){
              handleOutput("  "+longestMatchedRule.elementAt(i).toString());
           }
           handleOutputln("");
       }

       Vector appendedPath = new Vector();
       findAppendPath(longestMatch.isSet(), longestMatch.isNotSuppressed(), 
                      iMaxMatchLength, appendedPath,longestMatch.getVecAtoms(),
                      longestMatch.getRepVecAtoms(), vecAtoms);

       if(longestMatch.getVecTerms() != null){
           Term   curTerm = null;
           String curNode = null;
           Vector vecRightHandSide = longestMatch.getVecTerms();
           int    jsize = vecRightHandSide.size();
           int    evalRes = 0;

           for(int j = 0; j < jsize; j ++){
              curTerm = (Term) vecRightHandSide.elementAt(j);
              curNode = curTerm.getNode();
              evalRes = evalTerm(curTerm, ele, vecResult, lDepth+1,
                           localNode, localPath, gNode, gPath, appendedPath);
              if(evalRes == 0)
                 return 0;
              else if(evalRes == 2)
                 return 2;
            }
       }

       globalNode = gNode;
       globalPath = (Vector)gPath.clone();
       localNode  = gNode;
       localPath  = (Vector)gPath.clone();
       mapTable   = mapTableCopy;
       return 1;
   }

   /**
   *  in case the variable is used in set idea
   *  if there are several different mapping of variables
   *  could yield to a good match, 
   *  then the first good mapping will be picked
   *  for example:
   *  #vars $var1: 1 2. 
   *  rule is:
   *    {var1} == something.
   *  query is <1 2>
   *  ==> $var1 will map to 1
   *  However, please do not use variable in set if possible.
   */
   private int FindMatch(boolean bSet, Vector srcAtoms, Vector destAtoms) 
               throws ParseException{
      int lMatchLength = srcAtoms.size();
      if(lMatchLength == 0)
         return lMatchLength;

      if(bSet){
          Hashtable   table = new Hashtable();
          Enumeration e = destAtoms.elements();
          while(e.hasMoreElements()){
             String str = (String)e.nextElement();
             Integer value = (Integer) table.put(str, new Integer(1));
             if(value != null)
                table.put(str, new Integer(value.intValue()+1));
          }

          /* The lengtheners eg., +1, +2 are used for lengthening the
             rule, if matched. This added length is kept count of in the
             variable addLen. Lengtheners are themselves counted as
             atoms. These considerations must be kept in mind while 
             adding the count in lengtheners. */
          Integer addLen = new Integer(0);
          Vector carrier = new Vector(1); // Just to get passing by reference
                                          // functionality.
          carrier.addElement(addLen);
          boolean flag = subFindMatch(bSet, 0, carrier, 
                 (Vector)srcAtoms.clone(), (Vector) destAtoms.clone(), table);
          addLen = (Integer) carrier.firstElement();
          return (flag ? lMatchLength+addLen.intValue(): -1);
      }
      
      /* loop variable increment for source atoms*/
      int i;
      /* loop variable increment for destination atoms */
      int j=0;

      /* Forcefully added length */
      int addLen=0;

      /* Number of matched elements for a regular expression */
      int l=0;

      /* Loop variable */
      int m=0;

      /* Decision variables */
      boolean done,isStar,isBrak,isPlus,isQuestion;


      for(i = 0; i < lMatchLength; i++){
        String strSrc  = (String)srcAtoms.elementAt(i);
	Vector vec = new Vector();
        int lowLimit=0,highLimit=0;
        String inBrak = new String();
	String tempStr = new String();
	String prePoundString = new String();

	// Match the endofpath pattern
        if(strSrc.equals("$$"))
	   if(j>=destAtoms.size()) 
	      return j+addLen;

        // If the srcAtom is a lengthening pattern like +1, +3 etc
        if(strSrc.charAt(0) == '+'){
           if(strSrc.charAt(1) == '+')
              addLen = 10000;
           else {
              try{
                 addLen = addLen + 
                   Integer.decode(strSrc.substring(1)).intValue();
              }catch(java.lang.NumberFormatException e){ addLen = 10000; }
           }
           
           continue;
        }

	done = true;

        /* Checking for negation */
        boolean negFlag = false;
        if(strSrc.charAt(0) == '!'){
           negFlag = true;
           strSrc = strSrc.substring(1, strSrc.length());
        }
        
	/* Seperating the regular expression a* from strSrc of form a*#1 (Say)
	   and placing in prePoundString. '#' cannot be the first char.
           '#' is checked to see if escaped*/
        int poundIndex = strSrc.lastIndexOf('#');
	if(poundIndex > 0 && strSrc.charAt(poundIndex-1) != '\\')
		prePoundString = strSrc.substring(0,poundIndex);
	else 
		prePoundString = strSrc;

	/* Check what regular expression it is. RegEx op cannot be the only
           char in string. */
        {
          int len = prePoundString.length();
          if(len > 1){
              isStar = prePoundString.endsWith("*");
              isPlus = prePoundString.endsWith("+");
              isQuestion = prePoundString.endsWith("?");
              isBrak = prePoundString.endsWith(")");
          }
          else {
             isStar = false; 
             isPlus = false; 
             isQuestion = false; 
             isBrak = false; 
          }

          /* If the RegEx op is preceded by an escape character, strSrc is
             not a regular expression */
          if(isStar || isPlus || isQuestion || isBrak)
             if(len > 1 && prePoundString.charAt(len-2) == '\\') {
                isStar = false; 
                isPlus = false;
                isQuestion = false;
                isBrak = false;
             }
        }
              

	/* Negations not permitted on regular expressions: message */
	if((isStar || isPlus || isQuestion || isBrak) && negFlag)
	   throw new ParseException(
	       "Negation not permitted for regular expressions: "
	       +"!"+strSrc);

	/* Split the regular expression off the regular expression symbol */
	if(isStar || isPlus || isQuestion || isBrak){
	    if(isBrak)
	      tempStr = prePoundString.substring(0,prePoundString.indexOf('('));
	    else
	      tempStr = prePoundString.substring(0,prePoundString.length()-1);
	}
	else
	    tempStr = prePoundString;


	/* If the element is a general regular expression of the form 
           a{low_limit,hi_limit} then extract the information out of it */
	if (isBrak){
	   inBrak = prePoundString.substring(
		prePoundString.indexOf('(')+1,prePoundString.length()-1);
	   int indexOfComma = inBrak.indexOf(',');
	   if(indexOfComma == -1)
	      return -1;
	   String leftOfComma = (String) 
		        inBrak.substring(0,indexOfComma);
	   String rightOfComma = (String)
			inBrak.substring(indexOfComma+1);
	   if(leftOfComma.length() == 0)
	      lowLimit = 0;
	   else 
	      lowLimit = (int) 
			   Integer.valueOf(leftOfComma).shortValue();
	   if(rightOfComma.length() == 0)
	      highLimit = 10000;
	   else 
	      highLimit = (int) 
			   Integer.valueOf(rightOfComma).shortValue();
	}

	/* Match all the destination atoms that match the source element */
	 while (done){
	   if(j>=destAtoms.size() && !isStar && !isPlus && !isQuestion 
                        && !isBrak) 
	       return -1;
	   if(j>=destAtoms.size() && (isStar || isPlus || isQuestion 
                        || isBrak)){
	       if(isBrak)
		  if(l<lowLimit)
		     return -1;
	       if(i+1 < srcAtoms.size()){
		  for(m=i+1; m<srcAtoms.size(); m++){
		      prePoundString = (String)srcAtoms.elementAt(m);

                      /* Check what regular expression it is. RegEx op cannot 
                         be the only char in string. */
                      {
                          int len = prePoundString.length();
                          if(len >1){
                             isStar = prePoundString.endsWith("*");
                             isPlus = prePoundString.endsWith("+");
                             isQuestion = prePoundString.endsWith("?");
                             isBrak = prePoundString.endsWith(")");
                          }
                          else {
                             isStar = false; 
                             isPlus = false; 
                             isQuestion = false; 
                             isBrak = false; 
                          }


                          /* If the RegEx op is preceded by an escape 
                             character, strSrc is not a regular expression */
                          if(isStar || isPlus || isQuestion || isBrak)
                             if(len>1 && prePoundString.charAt(len-2)=='\\') {
                                isStar = false; 
                                isPlus = false;
                                isQuestion = false;
                                isBrak = false;
                             }
                      }

		      if(isBrak){
			 inBrak = prePoundString.substring(
				         prePoundString.indexOf('(')+1,
					 prePoundString.length()-1);
			 int indexOfComma = inBrak.indexOf(',');
			 if(indexOfComma == -1)
			    return -1;
			 String leftOfComma = (String) 
					inBrak.substring(0,indexOfComma);
			 String rightOfComma = (String)
					inBrak.substring(indexOfComma+1);
			 if(leftOfComma.length() == 0)
			    lowLimit = 0;
			 else
			    lowLimit = (int)Integer
					.valueOf(leftOfComma)
					.shortValue();
			 if(lowLimit != 0)
			    return -1;
		      }
		      else if(isPlus)
			 return -1;
		      else if(!isQuestion && !isStar)
			 return -1;
		  }
		  if(mapTable.containsKey(strSrc))
		     return -1;
		  mapTable.put(strSrc,vec);
		  return j+addLen;
	       }
	       else{
		  if(mapTable.containsKey(strSrc))
		     return -1;
		  mapTable.put(strSrc,vec);
		  return j+addLen;
	       }
           }
	   
           String strDest = (String)destAtoms.elementAt(j);
	   if(!isStar && !isPlus && !isQuestion && !isBrak) 
              done = false;

           if(varsTable.containsKey(tempStr)){ 
              String mapped =  lookupVarMap(tempStr+"+#"+strDest);
              if(mapped == null && isStar ){
		 if(mapTable.containsKey(strSrc))
		    return -1;
                 mapTable.put(strSrc, vec);
		 done = false;
		 continue;
              } 
	      else if(mapped == null && !isStar && !isPlus 
			&& !isQuestion && !isBrak){
                 if(negFlag){
                    Vector oldmapped = (Vector)mapTable.get("!"+strSrc);
                    if(!mapTable.containsKey("!"+strSrc)){
                       vec.addElement(strDest);
                       mapTable.put("!"+strSrc, vec);
                    }
                    else if(!oldmapped.firstElement().equals(strDest))
                       return -1;
                    j++;
                    continue; 
                 }
                 else
                    return -1; 
              }
	      else if(mapped != null && !isStar && !isPlus 
			&& !isQuestion && !isBrak){
                 if(!negFlag){
                     Vector oldmapped = (Vector)mapTable.get(strSrc);
                     if(!mapTable.containsKey(strSrc)){
                        vec.addElement(mapped);
                        mapTable.put(strSrc, vec);
                     }
                     else if(!oldmapped.firstElement().equals(mapped))
                        return -1;
                 }
                 else 
                    return -1;
              }
	      else if(mapped == null && isPlus){
	         if(l==0) return -1; 
	         if(mapTable.containsKey(strSrc))
		    return -1;
                 mapTable.put(strSrc, vec);
	         done=false;
	         continue;
              } 
	      else if(mapped == null && isBrak){
	         if(l<lowLimit)
		    return -1;
	         if(mapTable.containsKey(strSrc))
		    return -1;
                 mapTable.put(strSrc, vec);
	         done=false;
	         continue;
	      }
	      else if(mapped != null && isBrak){
	         if(l>highLimit-1){
		    done=false;
		    continue;
	         }
	      }
	      else if(mapped == null && isQuestion){
	         if(mapTable.containsKey(strSrc))
		    return -1;
                 mapTable.put(strSrc, vec);
	         done=false;
	         continue;
	      }
	      else if(mapped != null && isQuestion){
	         vec.addElement(mapped);
	         if(mapTable.containsKey(strSrc))
		    return -1;
                 mapTable.put(strSrc, vec);
	         done=false;
	         j++;
	         continue;
	      }

	      if(isStar || isPlus || isBrak){
	         vec.addElement(mapped);	
	         j++;
	         l++;
	         continue;
	      }
           }
	   else{
              if(!tempStr.equals(strDest) && isStar ){
		 if(mapTable.containsKey(strSrc))
		    return -1;
                 mapTable.put(strSrc, vec);
		 done = false;
		 continue;
	      }
              else if(!tempStr.equals(strDest) && !isStar && !isPlus 
			&& !isQuestion && !isBrak){
                 if(!negFlag)
      		    return -1; 
	      }
              else if(tempStr.equals(strDest) && !isStar && !isPlus 
			&& !isQuestion && !isBrak){
		 if(negFlag)
		    return -1;
	      }
	      else if(!tempStr.equals(strDest) && isPlus){
		 if(l==0) return -1;
		 if(mapTable.containsKey(strSrc))
		    return -1;
                 mapTable.put(strSrc, vec);
		 done = false;
		 continue;
	      }	
	      else if(!tempStr.equals(strDest) && isBrak){
		 if(l<lowLimit) return -1;
		 if(mapTable.containsKey(strSrc))
		    return -1;
                 mapTable.put(strSrc, vec);
		 done = false;
		 continue;
	      }	
	      else if(tempStr.equals(strDest) && isBrak){
		 if(l>highLimit-1){
		    done=false;
		    continue;
		 }
	      }	
	      else if(!tempStr.equals(strDest) && isQuestion){
		 if(mapTable.containsKey(strSrc))
		    return -1;
                 mapTable.put(strSrc, vec);
		 done = false;
		 continue;
	      }
	      else if(tempStr.equals(strDest) && isQuestion){
		 vec.addElement(strDest);
		 if(mapTable.containsKey(strSrc))
		    return -1;
                 mapTable.put(strSrc, vec);
		 done = false;
		 j++;
		 continue;
	      }

	      if(isStar || isPlus || isBrak){
                 vec.addElement(strDest);
                 j++;
                 l++;
                 continue;
              }
	   }
	   j++;
	   l++;
	 }
	 l=0;		  
      }
      return j+addLen; 
   }

   private void printDebug4FindMatch(int level, String title, 
                                     Vector srcAtoms, Vector destAtoms,
                                     Hashtable destTable){
      handleOutputln(title + level);
      handleOutputln("srcAtoms" + srcAtoms);
      handleOutputln("destAtoms" + destAtoms);
      handleOutputln("destTable" + destTable);
      handleOutputln("mapTable" +  mapTable);
      handleOutputln(""); 
   }

   /**
   *  In order to make recursion, make a copy of those
   *  some local variable should be recovered,
   *  therefore, we need make some copies while necessary  
   */
   private boolean subFindMatch(boolean bSet, int level, Vector carrier,
           Vector srcAtoms, Vector destAtoms, Hashtable destTable) 
           throws ParseException{
      /*
      *  <> or {} can match anything
      */
      if(srcAtoms.size() == 0){
          if(lDebugLevel == DEBUG_SHOW_MATCH)
             printDebug4FindMatch(level, "empty src, return true @level",
                                  srcAtoms, destAtoms, destTable);
          return true;
      }

      if(lDebugLevel == DEBUG_SHOW_MATCH)
         printDebug4FindMatch(level, "in level", 
                              srcAtoms, destAtoms, destTable);

      /*
      *  the algorithm is recursive, however, the structure
      *  in java is passed by pointer, so we need make a copy ourselves
      */
      Vector srcAtomsCopy     = (Vector) srcAtoms.clone();
      Vector destAtomsCopy    = (Vector) destAtoms.clone();
      Hashtable destTableCopy = (Hashtable)destTable.clone();
      Hashtable mapTableCopy  = (Hashtable)mapTable.clone();

      /*
      *  what's the first atoms in left hand side
      */
      String strAtom = (String)srcAtoms.firstElement();
      String strVar = strAtom;
      srcAtoms.removeElementAt(0);

      // If the srcAtom is a lengthening pattern like +1, +3 etc
      if(strAtom.charAt(0) == '+'){
         Integer addLen = (Integer) carrier.firstElement();
         int iadd;
         if(strAtom.charAt(1) == '+') // Infinity
            iadd = 10000;
         else { // +1, +_2, +3, etc
            try{
               iadd = addLen.intValue() 
                           + Integer.decode(strAtom.substring(1)).intValue();
            }catch(java.lang.NumberFormatException e){ iadd = 10000; }
         }
         addLen = new Integer(iadd-1); // The lengthener itself will be counted
                                       // as an atom later. So it should 
                                       // be subtracted from here.
         carrier.setElementAt(addLen, 0);
         return subFindMatch(bSet, level+1, carrier, srcAtoms,
                              destAtoms, destTable);
      }


      boolean  negFlag = false;
      if(strAtom.charAt(0) == '!'){
         negFlag = true;
         strAtom = strAtom.substring(1, strAtom.length());   
      }

      String strAtomOrig = strAtom; 

      /*strAtom may be an element (generally variable) that has been made
      distinct with '#' symbol (for example; $abc#1). All elements are
      stored in strAtomOrig and strAtom stores elements after stripping
      it of the '#name',if present. */
      if(strAtom.indexOf('#') != -1)
         strAtom = strAtom.substring(0,strVar.indexOf('#'));

      /*
      *  if it is a variable, check whether used this variable before
      */
      if(varsTable.containsKey(strAtom)){
         if(mapTable.containsKey(strAtomOrig))
            strVar = (String) ((Vector) 
                        mapTable.get(strAtomOrig)).firstElement();
         else
            strVar = null;
         if(strVar != null)
            strAtom = strVar;
      }

      /** add this for negative in the set idea
        */ 
      if(strVar != null && negFlag){
         if(destTable.containsKey(strAtom)) {
            if(lDebugLevel == DEBUG_SHOW_MATCH)
               printDebug4FindMatch(level, "return false noset at level",
                                    srcAtoms, destAtoms, destTable);
               return false;
         }
         else{
               return subFindMatch(bSet, level+1, carrier, srcAtoms,
                                    destAtoms, destTable);
         }
      }

      /**
      * just a simple atom or bounded variable, 
      * ==> only need to check whether it appears in destTable
      */
      if(strVar != null && !negFlag){
         Integer value = (Integer) destTable.remove(strAtom);
         if(value == null){
            srcAtoms  = srcAtomsCopy;
            destTable = destTableCopy;
            mapTable  = mapTableCopy;
            if(lDebugLevel == DEBUG_SHOW_MATCH)
               printDebug4FindMatch(level, "return false noset at level",
                                     srcAtoms, destAtoms, destTable);
            return false;
         }
         else if(value.intValue() > 1)
                  destTable.put(strAtom, new Integer(value.intValue()-1));
         return subFindMatch(bSet, level+1, carrier, srcAtoms,
                              destAtoms, destTable);
      }

      /**
      *  it is a unbounded variable
      *  try every possible mapping
      */
      else{
         Vector vecVars = (Vector) varsTable.get(strAtom);
         if(vecVars == null)
            throw new ParseException("Undefined variable: " + strAtom);

         for(Enumeration eVars = vecVars.elements(); 
                          eVars.hasMoreElements();){
             Vector    v  = (Vector) srcAtoms.clone();
             Hashtable h1 = (Hashtable)mapTable.clone();
             strVar = (String) eVars.nextElement();

             Hashtable h2 = (Hashtable)destTable.clone();
             Integer value = (Integer) destTable.remove(strVar);
             /**
             *     it good for this mapping so far
             */
             if((value != null && !negFlag) ||
                   (value == null && negFlag)) {
                Vector  vec = new Vector();
                vec.addElement(strVar);
                mapTable.put(strAtomOrig, vec);
                if(value != null && value.intValue() > 1)
                   destTable.put(strVar, 
                                 new Integer(value.intValue()-1));
                if(subFindMatch(bSet, level+1, carrier, srcAtoms, 
                                 destAtoms, destTable)){
                   if(lDebugLevel == DEBUG_SHOW_MATCH)
                      printDebug4FindMatch(
                                level, "return trun @level", 
                                srcAtoms, destAtoms, destTable);
                   return true;
                   }
             }
             srcAtoms =  v;
             mapTable  = h1; 
             destTable = h2;
          }
          /**
          *  all the mapping can not work, return false;
          */
          {
             srcAtoms  = srcAtomsCopy;
             destTable = destTableCopy;
             destAtoms = destAtomsCopy;
             mapTable  = mapTableCopy;
             if(lDebugLevel == DEBUG_SHOW_MATCH)
                printDebug4FindMatch(level, "return false set @level",
                                      srcAtoms, destAtoms, destTable);
             return false;
          }
      }
   }

   /**
   *  evaluate the value of a Term
   *  might called recursively according to the rules
   *  Parameters:
   *    Term           term        : the term to be evaluated
   *    EleInNameTable ele         : current item in name table
   *    Vector         rsltAtoms   : where to put result
   *    long           lDepth      : the depth for recursion
   *    String         localNode   : current local Node Name
   *    Vector         localPath   : current local Path value
   *    String         globalNode  : current global Node Name
   *    Vector         globalPath  : current global Path value
   *    Vector         appendedPath: the path should be appended
   */
   private int evalTerm(
         Term term, EleInNameTable ele, Vector rsltAtoms, long lDepth, 
         String localNode, Vector localPath, String gNode, Vector gPath, 
         Vector appendedPath) throws ParseException{
      String      strNode = term.getNode();
      Enumeration e = null;

      //<terms> or "<terms>"
      if(strNode == null){
         Vector vecAtoms = new Vector();
         if(term.getVecTerms() != null){
            e = term.getVecTerms().elements(); 
            int evalRes = 0;
            while(e.hasMoreElements()) {
               evalRes = evalTerm((Term)e.nextElement(), ele, vecAtoms,
                                        lDepth, localNode, localPath, 
                                        gNode, gPath, appendedPath);
               if(evalRes == 0)
                  return 0;
               else if(evalRes == 2)
                  return 2;
            }
         }
         if(appendedPath != null){
            for(e = appendedPath.elements(); e.hasMoreElements();)
               vecAtoms.addElement(e.nextElement());
         }
         return doQuery(term.isGlobal(), strNode, vecAtoms, rsltAtoms,
                        lDepth, localNode, localPath, gNode, gPath);
      }

      //Node or  Node:<terms> or "Node" or "Node:<terms>"
      else if(isNodeInNameTable(strNode)){
         Vector vecAtoms = null;
         if(term.getVecTerms() != null){
            vecAtoms = new Vector();
            e = term.getVecTerms().elements(); 
            int evalRes = 0;
            while(e.hasMoreElements()){
               evalRes = evalTerm((Term)e.nextElement(), ele, vecAtoms, 
                                           lDepth, localNode, localPath,
                                           gNode, gPath, appendedPath);
               if(evalRes == 0)
                  return 0;
               else if(evalRes == 2)
                  return 2;
            }
            if(appendedPath != null){
               for(e = appendedPath.elements(); e.hasMoreElements();)
                  vecAtoms.addElement(e.nextElement());
            }
         }
         return doQuery(term.isGlobal(), strNode, vecAtoms, rsltAtoms, 
                        lDepth, localNode, localPath, gNode, gPath);
      }

      //this should be a valid node name but it does not exist
      else if(strNode.charAt(0) >= 'A' && strNode.charAt(0) <= 'Z' &&
              !isAtomInNameTable(strNode)){
         handleOutputln(strNode + " :No such node exists in the theory");
         return 0;
      }

      //this is a variable
      else if(varsTable.containsKey(strNode)){
         Vector strVar = (Vector) mapTable.get(strNode);
         if(strVar == null)
            throw new ParseException("Unbounded Variable: " + strNode);
         else
            rsltAtoms.addElement((String)strVar.firstElement());
      }

      // this is a regular expression match variable
      else if(mapTable.containsKey(strNode)){
         Vector strVar = (Vector) mapTable.get(strNode);

         if(strVar != null)
            for(int i=0;i<strVar.size();i++)
               rsltAtoms.addElement(strVar.elementAt(i));
      }

      // if the above IF fails for its kind
      else if((strNode.endsWith("+") || strNode.endsWith("?") 
		|| strNode.endsWith(")") || strNode.endsWith("*") 
		|| strNode.indexOf("#") != -1 || strNode.charAt(0) == '!') 
	  	&& strNode.length()>1)
         throw new ParseException("No Match found in LHS for " + strNode);
	
      // if it is a suppress atom (!) is used for suppressing rules)
      else if(strNode.equals("!"))
         return 2;

      //just an atom
      else
         rsltAtoms.addElement(strNode);

      return 1;
   }

   /**
   *   postCompilation
   *        procedure called after program compilation.
   */   
   public void postCompilation() throws Exception{
      resolveVarsInShow();
      resolveHideif();
   }

   /**
   *  resolveVarsInShow
   *      resolve all variables in show statements. Variables are considered
   *      as sets and thus powerLoop is called on it. For eg., 
   *            #vars $ab: a b.
   *            #vars $cd: c d.
   *            #show <$ab $cd>.  ==> <a c>, <a d>, <b c>, <b d>.
   *                    which is similar to <a b :: c d> in show. This 
   *                    resolution is possible in postCompilation only since
   *                    #vars can occur after #show.
   */
   private void resolveVarsInShow() throws Exception{
      //resolve variables in the #show if-then-else statement
      IfThenElse  ifThenElseQuery = null;
      Enumeration eShowVec = null; 
      Vector curShowVec = null;
      Vector newShowVectors = null;
      for(int i = 0; i < allShowVector.size(); i ++){
         newShowVectors = new Vector();
         ifThenElseQuery  = (IfThenElse) allShowVector.elementAt(i);
	 if(ifThenElseQuery.getShowVector() != null){
	    eShowVec = ifThenElseQuery.getShowVector().elements();
            while(eShowVec.hasMoreElements()){
               curShowVec = (Vector)eShowVec.nextElement();
               Vector vecVecVar = new Vector();
               int size = curShowVec.size();
               for(int j=0; j<size; j++){
                  String str = (String)curShowVec.elementAt(j);
                  Vector vec = (Vector)varsTable.get(str);
                  if(vec == null){
                     vec = new Vector();
                     vec.addElement(str);
                  }
                  vecVecVar.addElement((Vector)vec.clone());
               }
               
               Vector resVec = new Vector();
               powerLoop(vecVecVar, size, new Vector(), 0, resVec); 

               for(int j=0; j<resVec.size(); j++)
                  newShowVectors.addElement((Vector)resVec.elementAt(j));
            }
            ifThenElseQuery.setShowVector(newShowVectors);
         }
	 if(ifThenElseQuery.getLHSCondition() != null){
	    resolveVarsInIfThenElse(ifThenElseQuery);
	 }
      }
   }

   /* Power loop functionality */
   void powerLoop(Vector vecVecAtoms, int size, Vector vecAtoms, int level, 
                   Vector resVec) {
      int iMax = ((Vector)vecVecAtoms.elementAt(level)).size();
      for(int i=0; i<iMax; i++){
         vecAtoms.addElement((String)
                         ((Vector)vecVecAtoms.elementAt(level)).elementAt(i));
         if(level == size-1)
            resVec.addElement((Vector)vecAtoms.clone());
         else
            powerLoop(vecVecAtoms, size, vecAtoms, level+1, resVec);
         vecAtoms.removeElementAt(level);
      }
   }

   private void resolveVarsInIfThenElse(IfThenElse ifThenElseQuery) 
                throws Exception{
      Vector      vecThen   = null;
      Vector      vecElse   = null;
      Vector      vecThenIf = null;
      Vector      vecElseIf = null;
      Vector 	  vecElsif  = null;

      vecThen = (Vector) ifThenElseQuery.getThenVector();
      vecElse = (Vector) ifThenElseQuery.getElseVector();
      vecThenIf = (Vector) ifThenElseQuery.getThenIf();
      vecElseIf = (Vector) ifThenElseQuery.getElseIf();
      vecElsif = (Vector) ifThenElseQuery.getElsif();

      if(vecThen != null){
         Enumeration eThenVec = vecThen.elements();
         Vector newThenVectors = new Vector();
         while(eThenVec.hasMoreElements()){
            Vector curThenVec = (Vector)eThenVec.nextElement();
            Vector vecVecVar = new Vector();
            int size = curThenVec.size();
            for(int j=0; j<size; j++){
               String str = (String)curThenVec.elementAt(j);
               Vector vec = (Vector)varsTable.get(str);
               if(vec == null){
                  vec = new Vector();
                  vec.addElement(str);
               }
               vecVecVar.addElement((Vector)vec.clone());
            }
            
            Vector resVec = new Vector();
            powerLoop(vecVecVar, size, new Vector(), 0, resVec); 

            for(int j=0; j<resVec.size(); j++)
               newThenVectors.addElement((Vector)resVec.elementAt(j));
         }
         ifThenElseQuery.setThenVector(newThenVectors);
      }

      if(vecThenIf != null){
         Enumeration eVecThenIf = vecThenIf.elements();
         for(;eVecThenIf.hasMoreElements();){
            IfThenElse newIfThenElse = 
                          (IfThenElse) eVecThenIf.nextElement();
            resolveVarsInIfThenElse(newIfThenElse);
         }
      }

      if(vecElsif != null){
         Enumeration eVecElsif = vecElsif.elements();
         IfThenElse newIfThenElse = new IfThenElse(); 
         for(;eVecElsif.hasMoreElements();){
             newIfThenElse = (IfThenElse) eVecElsif.nextElement();
             resolveVarsInIfThenElse(newIfThenElse);
         }
      }

      if(vecElse != null){
         Enumeration eElseVec = vecElse.elements();
         Vector newElseVectors = new Vector();
         while(eElseVec.hasMoreElements()){
            Vector curElseVec = (Vector)eElseVec.nextElement();
            Vector vecVecVar = new Vector();
            int size = curElseVec.size();
            for(int j=0; j<size; j++){
               String str = (String)curElseVec.elementAt(j);
               Vector vec = (Vector)varsTable.get(str);
               if(vec == null){
                  vec = new Vector();
                  vec.addElement(str);
               }
               vecVecVar.addElement((Vector)vec.clone());
            }
            
            Vector resVec = new Vector();
            powerLoop(vecVecVar, size, new Vector(), 0, resVec); 

            for(int j=0; j<resVec.size(); j++)
               newElseVectors.addElement((Vector)resVec.elementAt(j));
         }
         ifThenElseQuery.setElseVector(newElseVectors);
      }

      if(vecElseIf != null){
         Enumeration eVecElseIf = vecElseIf.elements();
         for(;eVecElseIf.hasMoreElements();){
             IfThenElse newIfThenElse 
                       = (IfThenElse) eVecElseIf.nextElement();
             resolveVarsInIfThenElse(newIfThenElse);
         }
      }
   }


   /**
   *  resolveHideif
   *      resolve rule specified in #hideif into nodes.
   */
   private void resolveHideif() throws Exception{
      notifyObservers(null);

      Enumeration    eNodes  = nameTable.elements();
      EleInNameTable ele     = null;
      String         strNode = null;
      Vector	     vecIf   = null;
      Vector	     vecIfWhat= null;
      Vector	     vecResult= null;
      Hashtable      hideTableClone = (Hashtable) hideTable.clone();
      Hashtable      newHideTable = (Hashtable) hideTable.clone();
      boolean 	     flag;

	
      //find all nodes that are affected by #hideifNot 
      //#hideifNot contains the path that is not to be hidden
      while(eNodes.hasMoreElements()){
         ele     = (EleInNameTable) eNodes.nextElement();
         strNode = ele.getStrKey();
         if(ele.isNode() && !hideTableClone.containsKey(strNode)) {
	    for(int i = 0;i<hideifNotVector.size();i+=2){
	       vecIf= (Vector) hideifNotVector.elementAt(i);
	       vecIfWhat= (Vector) hideifNotVector.elementAt(i+1);
	       vecResult = new Vector();
	       flag = false;
               if(doQuery(false, strNode, vecIf, vecResult, 
                        0,  null, null, null, null) == 1){
                  if(vecResult.size() == vecIfWhat.size()){
                     flag = true;
                     for(int j = 0; j < vecResult.size(); j ++){
                        if(((String)vecResult.elementAt(j)) != 
                                ((String)vecIfWhat.elementAt(j))){
                           flag = false;
                           break;
                        }
                     }
                  }
		  if(!flag) {
		     newHideTable.put(strNode,strNode);
		     addToHideifTable(strNode);
		  }
	       }
	    }
         }
      }

      //replace the existing hideTable with new hideTable
      hideTableClone= (Hashtable) newHideTable.clone();

      eNodes = nameTable.elements();

      //find all nodes that are affected by #hideif and put in hideTable
      while(eNodes.hasMoreElements()){
         ele     = (EleInNameTable) eNodes.nextElement();
         strNode = ele.getStrKey();
         if(ele.isNode() && !hideTableClone.containsKey(strNode)) {
	    for(int i = 0;i<hideifVector.size();i+=2){
	       vecIf= (Vector) hideifVector.elementAt(i);
	       vecIfWhat= (Vector) hideifVector.elementAt(i+1);
	       vecResult = new Vector();
	       flag = false;
               if(doQuery(false, strNode, vecIf, vecResult, 
                          0,  null, null, null, null) == 1){
                  if(vecResult.size() == vecIfWhat.size()){
                     flag = true;
                     for(int j = 0; j < vecResult.size(); j ++){
                        if(((String)vecResult.elementAt(j)) != 
                                ((String)vecIfWhat.elementAt(j))){
                           flag = false;
                           break;
                        }
                     }
                  }
		  if(flag)
		     addToHideifTable(strNode);
	       }
	    }
         }
      }
   }

   /**
   *   warnRedundantId
   *      Display warning message for identifiers appearing only once
   *      in theory.
   */
   private void warnRedundantId(){
      boolean appendFlag = false;
      String strIdent;
      String position;
      StringBuffer strMsg = new StringBuffer();
      strMsg.append("\nWarning! The following identifiers "+
                        "appear only once in the program");
      strMsg.append("\nCheck for possible spelling mistake or redundancy");
      for(Enumeration e = unUsedIdentTable.keys();e.hasMoreElements();){
         strIdent = (String)e.nextElement();
         position = (String)unUsedIdentTable.get(strIdent);
         //Check to see if identifier is present in nodeTable
         if(!nodeTable.contains(strIdent)){
            strMsg.append("\n   "+position+"\t"+strIdent);
            appendFlag = true;
         }
      }
      if(appendFlag)
         System.out.println(strMsg.toString());
   }

   /*-----------------------------------------------------------------------*/
   /**
   *   Display all the theory 
   */
   public void showTheory() throws Exception{
      //mark the execution of theory command
      theoryFlag = true;

      notifyObservers(null);

      //any default queries?
      int   iAllShowSize = allShowVector.size();
      if(iAllShowSize == 0){
         handleOutputln("No show or showif statement in theory\n");
         return;
      }

      Enumeration    eNodes  = nameTable.elements();
      EleInNameTable ele     = null;
      String         strNode = null;

	
      //show every Node not affected by #hide and #hideif statement
      while(eNodes.hasMoreElements()){
         ele     = (EleInNameTable) eNodes.nextElement();
         strNode = ele.getStrKey();
         if(ele.isNode() && !hideTable.containsKey(strNode) 
                && !hideifTable.containsKey(strNode)) {
            showTheory(strNode);
         }
      }
   }

   public void queryNode(String node){
      try{
         showTheory(node);
      }
      catch(Exception e){
         handleOutput(e.toString());
      }
   }

   public void showTheory(String strNode) throws Exception{
      Enumeration eAllQuery = null;
      Vector      vecQuery  = null;
      Vector      vecResult = null;


      //process queries in the #show if-then-else statement
      IfThenElse  ifThenElseQuery = null;
      Enumeration eShowQuery = null; 
      for(int i = 0; i < allShowVector.size(); i ++){
         ifThenElseQuery  = (IfThenElse) allShowVector.elementAt(i);
	 if(ifThenElseQuery.getShowVector() != null){
	    eShowQuery = ifThenElseQuery.getShowVector().elements();
            int queryRes = 0;

	    while(eShowQuery.hasMoreElements()) {
               vecQuery = (Vector) eShowQuery.nextElement();
               vecResult = new Vector();
               clearQuery();
               queryRes = doQuery(false, strNode, vecQuery, vecResult,  
                          0,  null, null, null, null);
               if(queryRes == 1){
                  print(toStringQuery(strNode, vecQuery, vecResult));
               }
               else if(queryRes == 0){
             	  print(toStringQuery(strNode, vecQuery, null));
               }
               else if(queryRes == 2){
                  // Suppress output
               }
	    }
	 }
	 if(ifThenElseQuery.getLHSCondition() != null){
	    processIfThenElse(strNode,ifThenElseQuery);
	 }
      }
   }


   /*-----------------------------------------------------------------------*/
   /**
   *   Display leaves
   */
   public void showLeaves() throws Exception{

      //mark the execution of leaves command
      theoryFlag = true;

      //any default queries?
      int   iAllShowSize = allShowVector.size();
      if(iAllShowSize == 0){
          handleOutputln("No show or showif statement in theory\n");
          //handleOutputln("or you have not yet compiled the source code.\n");
          return;
      }

      Enumeration    eNodes  = nameTable.elements();
      EleInNameTable ele     = null;
      String         strNode = null;
	
      while(eNodes.hasMoreElements()){
         ele     = (EleInNameTable) eNodes.nextElement();
         strNode = ele.getStrKey();
         if(ele.isNode() && !identifierTable.contains(strNode)
                && !hideifTable.containsKey(strNode)
                && !hideTable.containsKey(strNode)) {
            showTheory(strNode);
         }
      }

   }

   /**
   *   Display newly augmented leaves
   */
   public void showAugleaves() throws Exception{
      //any default queries?
      int   iAllShowSize = allShowVector.size();
      if(iAllShowSize == 0){
          handleOutputln("No show or showif statement in theory\n");
          //handleOutputln("or you have not yet compiled the source code.\n");
          return;
      }

      Enumeration    eNodes  = nameTable.elements();
      EleInNameTable ele     = null;
      String         strNode = null;
	
      while(eNodes.hasMoreElements()){
         ele     = (EleInNameTable) eNodes.nextElement();
         strNode = ele.getStrKey();
         if(ele.isNode() && !orgNameTable.containsKey(strNode) 
                && !identifierTable.contains(strNode)
                && !hideifTable.containsKey(strNode)
                && !hideTable.containsKey(strNode)) {
            showTheory(strNode);
         }
      }

   }


   /**
   *   
   */
   public void analyze() throws Exception{
      //list all identifiers appearing only once in the whole program
      warnRedundantId();

      if(theoryFlag){
         //Display all un matched rules
         displayUnMatchedRules();
      }
      else
         System.out.println("For a listing of unmatched paths please run leaves or theory command first");
   }

   /**
   *   Display all unmatched rules
   */   
   private void displayUnMatchedRules() throws Exception{

      //check all the rule flags and output the rules that were not matched at
      //all
      Enumeration eNodes     = nameTable.elements();
      EleInNameTable ele     = null;
      Vector matchFlags      = new Vector();
      Vector vecRules        = new Vector();
      StringBuffer strResult = new StringBuffer();
      StringBuffer tmpStrResult;
      StringBuffer hiddenLeaves = new StringBuffer(); 
      String strNode;
      boolean  appendFlag;
      boolean strResultFlag = false, hiddenLeavesFlag = false;

      strResult.append(
                "\nThe following rules were never matched during querying:\n");
      hiddenLeaves.append("\nLeaves Explicitly hidden: \n");
      while(eNodes.hasMoreElements()){
         ele        = (EleInNameTable) eNodes.nextElement();
         strNode       = (String) ele.getStrKey();
         if(ele.isNode()){
            if(identifierTable.contains(strNode) 
                  && (hideifTable.contains(strNode) 
                       || hideTable.contains(strNode))){
               hiddenLeaves.append(strNode+" ");
               hiddenLeavesFlag = true;
            }
            else {
               tmpStrResult = new StringBuffer();
               appendFlag    = false;
               matchFlags = (Vector) ele.getMatchFlags();
               vecRules   = (Vector) ele.getVecRules();
               tmpStrResult.append("\nIn Node "+strNode+":\n");

               if(!matchFlags.contains(Boolean.TRUE)){
                  tmpStrResult.append("  All rules\n");
                  appendFlag = true;
                  strResultFlag = true;
               }
               else {
                  for (int i = 0; i < matchFlags.size(); i++){
                     if(matchFlags.elementAt(i) == Boolean.FALSE){
                        tmpStrResult.append("  "+
                            ((Rule)vecRules.elementAt(i)).toString());
                        appendFlag = true;
                        strResultFlag = true;
                     }
                  }
               }
               if(appendFlag) 
                  strResult.append(tmpStrResult.toString());
            }
         }
      }

      if(strResultFlag)
         System.out.println(strResult.toString());
      if(hiddenLeavesFlag)
         System.out.println(hiddenLeaves.toString());
   }


   private boolean processIfThenElse(String strNode, 
                IfThenElse ifThenElseQuery) throws Exception{
      //process queries in the #showif statement
      Enumeration eAllQuery = null;
      Vector      vecQuery  = null;
      Vector      vecResult = null;
      Vector      vecIf     = null;
      Vector      vecIfWhat = null;
      Vector      vecThen   = null;
      Vector      vecElse   = null;
      Vector      vecThenIf = null;
      Vector      vecElseIf = null;
      Vector 	  vecElsif  = null;
      boolean     flag      = false;

      vecIf = (Vector) ifThenElseQuery.getLHSCondition();
      vecIfWhat = (Vector) ifThenElseQuery.getRHSCondition();
      vecThen = (Vector) ifThenElseQuery.getThenVector();
      vecElse = (Vector) ifThenElseQuery.getElseVector();
      vecThenIf = (Vector) ifThenElseQuery.getThenIf();
      vecElseIf = (Vector) ifThenElseQuery.getElseIf();
      vecElsif = (Vector) ifThenElseQuery.getElsif();

      clearQuery();
      vecResult = new Vector();
      //evaluate #showif's condition
      if(doQuery(false, strNode, vecIf, vecResult, 0, 
             null, null, null, null) == 1){
       	 if(vecResult.size() == vecIfWhat.size()){
            flag = true;
            for(int i = 0; i < vecResult.size(); i ++){
               if(((String)vecResult.elementAt(i)) != 
                       ((String)vecIfWhat.elementAt(i))){
                  flag = false;
                  break;
               }
            }
         }
                      
         //if match showif
         if(flag){
	    if(vecThen != null){
               eAllQuery = vecThen.elements();
               int queryRes = 0;

               while(eAllQuery.hasMoreElements()){
                  vecQuery = (Vector) eAllQuery.nextElement();
                  vecResult = new Vector();
                  clearQuery();
                  queryRes = doQuery(false, strNode, vecQuery, vecResult,
                                       0, null, null, null, null);
                  if(queryRes == 1){
                     print(toStringQuery(strNode, vecQuery, vecResult));
                  } 
		  else if(queryRes == 0){
                     print(toStringQuery(strNode, vecQuery, null));
                  }
                  else if(queryRes == 2){
                     // Suppress output
                  }
               }
	    }
	    if(vecThenIf != null){
	       Enumeration eVecThenIf = vecThenIf.elements();
	       for(;eVecThenIf.hasMoreElements();){
		  IfThenElse newIfThenElse = 
				(IfThenElse) eVecThenIf.nextElement();
		  processIfThenElse(strNode,newIfThenElse);
	       }
	    }
         }
	 else { //if did not match 
	    boolean newFlag = false;
	    if(vecElsif != null){
	       Enumeration eVecElsif = vecElsif.elements();
	       IfThenElse newIfThenElse = new IfThenElse(); 
	       for(;eVecElsif.hasMoreElements();){
		   newIfThenElse = (IfThenElse) eVecElsif.nextElement();
		   newFlag = processIfThenElse(strNode,newIfThenElse);
		   if(newFlag)
		      break;
	       }
	    }
	    if(!newFlag){
	       if(vecElse != null){
                  eAllQuery = vecElse.elements();
                  int queryRes = 0;
                  while(eAllQuery.hasMoreElements()){
                     vecQuery = (Vector) eAllQuery.nextElement();
                     vecResult = new Vector();
                     clearQuery();
                     queryRes = doQuery(false, strNode, vecQuery, vecResult,
                                          0, null, null, null, null);
                     if(queryRes == 1){
                        print(toStringQuery(strNode, vecQuery, 
                                                vecResult));
                     } 
                     else if(queryRes == 0){
                        print(toStringQuery(strNode, vecQuery, null));
                     }
                     else if(queryRes == 2){
                        //Suppress output
                     }
                  }
	       }
	       if(vecElseIf != null){
		  Enumeration eVecElseIf = vecElseIf.elements();
		  for(;eVecElseIf.hasMoreElements();){
		      IfThenElse newIfThenElse 
				= (IfThenElse) eVecElseIf.nextElement();
		      processIfThenElse(strNode,newIfThenElse);
		  }
	       }
	    }
	 }
      }
      return flag;
   }
	
/*****************************************************************************
                       Some toString????? functions
*****************************************************************************/
   /**
   *   we ask for     strNode:<vecAtoms>
   *      result is:  vecResult
   */
   public String toStringQuery(String strNode, 
                               Vector vecAtoms, Vector vecResult){
      StringBuffer strResult = new StringBuffer();
       
      strResult.append(strNode + ":<");
      for(int i = 0; i < vecAtoms.size(); i ++){
         strResult.append(" " + vecAtoms.elementAt(i).toString());
      }
      strResult.append(" > ==>");
      if(vecResult != null){
         for(int i = 0; i < vecResult.size(); i ++){
            strResult.append(" " + vecResult.elementAt(i).toString());
         }
      }
      strResult.append("\n");
      return strResult.toString();
   }

   /**
   *   current variable mapping information
   */
   private String toStringMapTable(Hashtable mapTable){
      StringBuffer strResult = new StringBuffer();
      if(mapTable != null){
          Enumeration eMap = mapTable.keys();
          while(eMap.hasMoreElements()){
              String strKey = (String) eMap.nextElement();
	      Vector vecVal = new Vector();
	      
		   vecVal=(Vector)mapTable.get(strKey);
	      	   String strVal = new String();
		   int i;

		   for(i=0;i<vecVal.size();i++)
		      strVal=strVal+((String) vecVal.elementAt(i))+" ";

                   strResult.append(strKey + " <==> " + 
                                      strVal + "  ");
          }
          strResult.append("\n");
      }
      return strResult.toString();
   }

   /**
   *  Information in Name Table
   */
   private String toStringNameTable(){ 
      StringBuffer strResult = new StringBuffer();
      for(Enumeration e = nameTable.elements(); e.hasMoreElements() ;) {
           strResult.append(
                   ((EleInNameTable)e.nextElement()).toString());
      }
      strResult.append("\n");
      return strResult.toString();
   }

   /**
   *   All the variables defined. 
   */
   private String toStringVarsTable(){ 
      StringBuffer strResult = new StringBuffer();
      for(Enumeration e = varsTable.keys(); e.hasMoreElements();){
           String  strKey = (String) e.nextElement();
           strResult.append("#vars  " + strKey + ": ");
           Enumeration eVars = ((Vector)varsTable.get(strKey)).elements();
           while(eVars.hasMoreElements())
                 strResult.append(" " + (String)eVars.nextElement());
           strResult.append(".\n");
      }
      return strResult.toString();
   }


   /**
   *   All the #show statements defined.
   */
   private String toStringShow(){
      if(allShowVector.size() == 0)
          return "";

      StringBuffer strResult = new StringBuffer();
      for(Enumeration e = allShowVector.elements(); e.hasMoreElements();) {
         strResult.append("#show ");
         IfThenElse curIf = (IfThenElse) e.nextElement();
	 Vector vec = (Vector) curIf.getShowVector();
	 if(vec != null){
	    for(Enumeration e1 = vec.elements(); e1.hasMoreElements(); ) {
	        Vector vec1 = (Vector) e1.nextElement();
              	strResult.append("\n      < ");
            	for(int i = 0; i < vec1.size(); i ++){
            		strResult.append(vec1.elementAt(i).toString() + " ");
              	}
                strResult.append(">");
	    }
	 }

	 if(curIf.getLHSCondition() != null)
	 	strResult = (StringBuffer) 
			toStringIfThenElse(strResult, curIf, false); 
         strResult.append(".\n");
      }
      return strResult.toString();
   } 

   private StringBuffer toStringIfThenElse(StringBuffer strResult, 
                IfThenElse curIf, boolean isElsif) {
      if(!isElsif)
         strResult.append("\n#if ");
      strResult.append("< ");
      Vector v1 = (Vector) curIf.getLHSCondition();
      for(int i = 0; i < v1.size(); i ++){
         strResult.append(v1.elementAt(i).toString() + " ");
      }
      strResult.append("> == ");

      Vector v2 = (Vector) curIf.getRHSCondition();
      for(int i = 0; i < v2.size(); i ++){
         strResult.append(v2.elementAt(i).toString() + " ");
      }

      Vector v3 = (Vector) curIf.getThenVector();
      if(v3 != null){
         strResult.append("\n  #then");
         for(int i = 0; i < v3.size(); i ++){
            Vector v4 = (Vector) v3.elementAt(i);
            strResult.append("\n      < ");
            for(int j = 0; j < v4.size(); j ++){
               strResult.append(v4.elementAt(j).toString() + " ");
            }
            strResult.append(">");
         }
      }

      Vector vecThenIf = (Vector) curIf.getThenIf();	
      if(vecThenIf != null){
	 Enumeration eVecThenIf = vecThenIf.elements();
	 IfThenElse newCurIf = new IfThenElse();
	 for(;eVecThenIf.hasMoreElements();){
      	    newCurIf = (IfThenElse) eVecThenIf.nextElement();
	    strResult = toStringIfThenElse(strResult, newCurIf,false);
	 }
      }

      Vector vecElsif = (Vector) curIf.getElsif();	
      if(vecElsif != null){
	  Enumeration eVecElsif = vecElsif.elements();
	  IfThenElse newCurIf = new IfThenElse();
	  for(;eVecElsif.hasMoreElements();){
             strResult.append("\n  #elsif");
      	     newCurIf = (IfThenElse) eVecElsif.nextElement();
	     strResult = toStringIfThenElse(strResult, newCurIf ,true);
	  }
      }
	
      Vector v5 = (Vector) curIf.getElseVector();
      Vector vecElseIf = (Vector) curIf.getElseIf();	
      if(v5 != null || vecElseIf != null)
         strResult.append("\n  #else");
      if(v5 != null){
         for(int i = 0; i < v5.size(); i ++){
             Vector v6 = (Vector) v5.elementAt(i);
             strResult.append("\n      < ");
             for(int j = 0; j < v6.size(); j ++){
                strResult.append(v6.elementAt(j).toString() + " ");
             }
             strResult.append(">");
         }
      }

      if(vecElseIf != null){
	  Enumeration eVecElseIf = vecElseIf.elements();
	  IfThenElse newCurIf = new IfThenElse();
	  for(;eVecElseIf.hasMoreElements();){
      	     newCurIf = (IfThenElse) eVecElseIf.nextElement();
	     strResult = toStringIfThenElse(strResult, newCurIf, false);
	  }
      }

      if(!isElsif)
         strResult.append("\n  #endif");

      return strResult;
   }

   /**
   *   All the #hide statement defined.
   */
   private String toStringHide(){
      if(hideTable.isEmpty())
         return "";

      StringBuffer strResult = new StringBuffer();
      strResult.append("\n#hide");
      for(Enumeration e = hideTable.keys(); e.hasMoreElements();)
         strResult.append(" " + e.nextElement());
      strResult.append(".\n");
      return strResult.toString();
   }

   /**
   *   All the #hideif statement defined.
   */
   private String toStringHideif(){
      if(hideifVector.isEmpty() && hideifNotVector.isEmpty())
         return "";

      StringBuffer strResult = new StringBuffer();
      Vector lhsAtoms = null;
      Vector rhsAtoms = null;
      strResult.append("\n#hideif");
      for(int i = 0; i<hideifNotVector.size(); i+=2){
	  lhsAtoms = (Vector) hideifNotVector.elementAt(i);
	  rhsAtoms = (Vector) hideifNotVector.elementAt(i+1);
	  strResult.append("\n    < ");
	  for(int j = 0;j<lhsAtoms.size();j++)
             strResult.append(lhsAtoms.elementAt(j).toString() + " ");
	  strResult.append("> != ");
	  for(int j = 0;j<rhsAtoms.size();j++)
             strResult.append(rhsAtoms.elementAt(j).toString() + " ");
      }
      for(int i = 0; i<hideifVector.size(); i+=2){
	  lhsAtoms = (Vector) hideifVector.elementAt(i);
	  rhsAtoms = (Vector) hideifVector.elementAt(i+1);
	  strResult.append("\n    < ");
	  for(int j = 0;j<lhsAtoms.size();j++)
             strResult.append(lhsAtoms.elementAt(j).toString() + " ");
	  strResult.append("> = ");
	  for(int j = 0;j<rhsAtoms.size();j++)
             strResult.append(rhsAtoms.elementAt(j).toString() + " ");
      }
      strResult.append(".\n");
      strResult.append("Nodes hidden by #hideif: ");
      if(hideifTable.isEmpty())
      	  strResult.append(" You should first run theory ");
      else
      	 for(Enumeration e = hideifTable.keys(); e.hasMoreElements();)
            strResult.append(" " + e.nextElement());
      strResult.append(".\n");
      return strResult.toString();
   }

   public String toString(){

      return "Dumping Theory\n" + 
             toStringNameTable() + toStringVarsTable() + 
	     toStringShow() + toStringHide() + toStringHideif();
     
   }
}
