package org.bouncycastle.crypto.examples;

import com.sun.kjava.*;
import java.io.*;
import java.lang.*;

import org.bouncycastle.util.test.*;
import org.bouncycastle.util.encoders.*;

import org.bouncycastle.crypto.*;
import org.bouncycastle.crypto.engines.*;
import org.bouncycastle.crypto.modes.*;
import org.bouncycastle.crypto.params.*;

/**
 * CryptoTest is a simple graphics application for the J2ME CLDC.
 *
 * It has hardcoded values for the key and plain text. It also
 * performs the standard testing for the chosen cipher, and 
 * displays the results.
 *
 * This example shows how to use the light-weight API and 
 * symmetric ciphers.
 *
 */
public class CryptoTest extends Spotlet
{
	Graphics g = Graphics.getGraphics();
	private static Button exitButton;
	private static Button encryptButton;
	private static Button decryptButton;
	private static Button testButton;

	private static RadioGroup rbg = null;
	private static RadioButton[] cipherList = null;

	private boolean doneEncrypt = false;

	private String key = "0123456789abcdef0123456789abcdef";
	private String plainText = "www.bouncycastle.org";
	private byte[] keyBytes = null;
	private byte[] cipherText = null;
	private BufferedBlockCipher cipher = null;

	private String[] cipherNames = { "DES", "DESede", "IDEA", 
								     "Rijndael", "Twofish" };

	public static void main(String[] args)
	{
		(new CryptoTest()).register(NO_EVENT_OPTIONS);
	}

	public CryptoTest()
	{
		encryptButton = new Button("Encrypt", 1, 128);
		decryptButton = new Button("Decrypt", 1, 145);
		exitButton = new Button("Exit", 138, 145);

		g.clearScreen();
		g.drawString("Key   is; "+key.substring(1,16)+"...", 5, 2, g.PLAIN);
		g.drawString("Input is; "+plainText, 5, 16, g.PLAIN);

		initCipherList();
		drawCipherList();

		exitButton.paint();
		encryptButton.paint();
		decryptButton.paint();
	}

	private void initCipherList() 
	{
		rbg = new RadioGroup(cipherNames.length);
		cipherList = new RadioButton[cipherNames.length];
		for (int i=0; i< cipherNames.length ; i++)
		{
			// System.out.println(cipherNames[i]);
			cipherList[i] = new RadioButton(rbg_x,rbg_y+(14*i),cipherNames[i]);
			rbg.add(cipherList[i]);
		}
		rbg.setSelected(cipherList[0]);
	}

	private void drawCipherList()
	{
		for (int i=0; i< cipherList.length ; i++)
		{
			cipherList[i].paint();
		}
	}

	private void handleRadioButtons(int x, int y)
	{
		for (int i=0; i< cipherList.length ; i++)
		{
			if (cipherList[i].pressed(x,y))
			{
				if (doneEncrypt)
				{
					status("Cannot change cipher, decrypt first");
					return;
				}
				rbg.setSelected(cipherList[i]);
			}
		}
	}

	private int whichCipher()
	{
		for (int i=0; i<cipherList.length ; i++)
		{
			if (cipherList[i].isSelected())
			{
				return i;
			}
		}
		return 0;
	}
	/**
	 * pendown() handles all the user interactions that are possible
	 * for the example.
	 */
	public void penDown(int x, int y)
	{
		if (exitButton.pressed(x,y))
		{
			System.exit(0);
		}

		if (encryptButton.pressed(x,y))
		{
			if (doneEncrypt)
			{
				status("Already encrypted, use decrypt");
				return;
			}
			cipherText = performEncrypt(Hex.decode(key.getBytes()), plainText);
			doneEncrypt = true;
			String ctS = new String(Hex.encode(cipherText));
			if (ctS.length() > 30)
			{
				ctS = ctS.substring(1,30);
			}
			status(ctS);
				
		}

		if (decryptButton.pressed(x,y))
		{
			if (!doneEncrypt)
			{
				status("Not encrypted, use encrypt first");
				return;
			}
			plainText = performDecrypt(Hex.decode(key.getBytes()), cipherText);
			doneEncrypt = false;
			status(plainText);
		}

		handleRadioButtons(x,y);
	}
	
	// encryption wrappers
	private final byte[] performEncrypt(byte[] key, String plainText)
	{
		byte[] ptBytes = plainText.getBytes();

		cipher = new PaddedBlockCipher(
									new CBCBlockCipher(getEngineInstance()));

		String name = cipher.getUnderlyingCipher().getAlgorithmName();
		message("Using "+name);

		cipher.init(true, new KeyParameter(key));

		byte[] rv = new byte[cipher.getOutputSize(ptBytes.length)];
	
		int oLen = cipher.processBytes(ptBytes, 0, ptBytes.length, rv, 0);
		try
		{
			cipher.doFinal(rv, oLen);
		}
		catch (CryptoException ce)
		{
			message("Ooops, encrypt exception");
			status(ce.toString());
		}
		return rv;
	}

	private final String performDecrypt(byte[] key, byte[] cipherText)
	{
		cipher.init(false, new KeyParameter(key));

		byte[] rv = new byte[cipher.getOutputSize(cipherText.length)];
	
		int oLen = cipher.processBytes(cipherText, 0, cipherText.length, rv, 0);
		try
		{
			cipher.doFinal(rv, oLen);
		}
		catch (CryptoException ce)
		{
			message("Ooops, decrypt exception");
			status(ce.toString());
		}
		return new String(rv);
	}

	private final BlockCipher getEngineInstance()
	{
		// returns a block cipher according to the current
		// state of the radio button lists.  This is only
		// done prior to encryption.
		BlockCipher rv = null;

		switch (whichCipher())
		{
			case 0:
				rv = new DESEngine();
				break;
			case 1:
				rv = new DESedeEngine();
				break;
			case 2:
				rv = new IDEAEngine();
				break;
			case 3:
				rv = new RijndaelEngine();
				break;
			case 4:
				rv = new TwofishEngine();
				break;
			default:
				rv = new DESEngine();
				break;
		}
		return rv;
	}

	private final static int rbg_x = 1;
	private final static int rbg_y = 32;

	// utility display functions
	private final static int status_x = 1;
	private final static int status_y = 100;

	private String[] currentStatus = new String[2];

	private void message(String s)
	{
		status(s, 0);
	}

	private void status(String s)
	{
		status(s, 1);
	}

	private void status(String s, int line)
	{
		if (currentStatus[line] != null)
		{
			clearStatus(line);
		}
		g.drawString(s, status_x, status_y+(14*line));
		currentStatus[line] = s;
	}

	private void clearStatus(int line)
	{
		if (currentStatus[line] != null)
		{
			g. drawString(currentStatus[line], 
						  status_x, status_y+(14*line), g.ERASE);
		}
		currentStatus[line] = null;
	}
}

