package org.bouncycastle.asn1.x509;

import java.io.*;
import java.util.*;

import org.bouncycastle.asn1.*;

public class X509Name
    implements DEREncodable
{
    /**
     * country code - StringType(SIZE(2))
     */
    public static final DERObjectIdentifier C = new DERObjectIdentifier("2.5.4.6");

    /**
     * organization - StringType(SIZE(1..64))
     */
    public static final DERObjectIdentifier O = new DERObjectIdentifier("2.5.4.10");

    /**
     * organizational unit name - StringType(SIZE(1..64))
     */
    public static final DERObjectIdentifier OU = new DERObjectIdentifier("2.5.4.11");

    /**
     * Title
     */
    public static final DERObjectIdentifier T = new DERObjectIdentifier("2.5.4.12");

    /**
     * common name - StringType(SIZE(1..64))
     */
    public static final DERObjectIdentifier CN = new DERObjectIdentifier("2.5.4.3");

    /**
     * device serial number name - StringType(SIZE(1..64))
     */
    public static final DERObjectIdentifier SN = new DERObjectIdentifier("2.5.4.5");

    /**
     * locality name - StringType(SIZE(1..64))
     */
    public static final DERObjectIdentifier L = new DERObjectIdentifier("2.5.4.7");

    /**
     * state, or province name - StringType(SIZE(1..64))
     */
    public static final DERObjectIdentifier ST = new DERObjectIdentifier("2.5.4.8");


    /**
     * email address (RSA PKCS#9 extension) - IA5String
     * <p>
     * note: if you're trying to be ultra orthodox, don't use this! It shouldn't be in here.
     */
    public static final DERObjectIdentifier EmailAddress = new DERObjectIdentifier("1.2.840.113549.1.9.1");

    /*
     * others...
     */
    public static final DERObjectIdentifier DC = new DERObjectIdentifier("0.9.2342.19200300.100.1.25");

    /**
     * look up table translating OID values into their common symbols.
     */
    public static Hashtable OIDLookUp = new Hashtable();

    /**
     * look up table translating common symbols into their OIDS.
     */
    public static Hashtable SymbolLookUp = new Hashtable();

    static
    {
        OIDLookUp.put(C, "C");
        OIDLookUp.put(O, "O");
        OIDLookUp.put(T, "T");
        OIDLookUp.put(OU, "OU");
        OIDLookUp.put(CN, "CN");
        OIDLookUp.put(L, "L");
        OIDLookUp.put(ST, "ST");
        OIDLookUp.put(SN, "SN");
        OIDLookUp.put(EmailAddress, "EmailAddress");
        OIDLookUp.put(DC, "DC");

        SymbolLookUp.put("C", C);
        SymbolLookUp.put("O", O);
        SymbolLookUp.put("T", T);
        SymbolLookUp.put("OU", OU);
        SymbolLookUp.put("CN", CN);
        SymbolLookUp.put("L", L);
        SymbolLookUp.put("ST", ST);
        SymbolLookUp.put("SN", SN);
        SymbolLookUp.put("EmailAddress", EmailAddress);
        SymbolLookUp.put("DC", DC);
    }

    private Vector                  ordering = new Vector();
    private Vector                  values = new Vector();

    public static X509Name getInstance(
        Object  obj)
    {
        if (obj instanceof X509Name)
        {
            return (X509Name)obj;
        }
        else if (obj instanceof ASN1Sequence)
        {
            return new X509Name((ASN1Sequence)obj);
        }

        throw new IllegalArgumentException("unknown object in factory");
    }

    /**
     * Constructor from DERConstructedSequence.
     *
     * the principal will be a list of constructed sets, each containing an (OID, String) pair.
     */
    public X509Name(
        ASN1Sequence  seq)
    {
        Enumeration e = seq.getObjects();

        while (e.hasMoreElements())
        {
            DERSet  set = (DERSet)e.nextElement();
            DERConstructedSequence  s = (DERConstructedSequence)set.getSequence();

            ordering.addElement(s.getObjectAt(0));
            values.addElement(((DERString)s.getObjectAt(1)).getString());
        }
    }

    /**
     * constructor from a table of attributes.
     * <p>
     * it's is assumed the table contains OID/String pairs, and the contents
     * of the table are copied into an internal table as part of the 
     * construction process.
     * <p>
     * <b>Note:</b> if the name you are trying to generate should be
     * following a specific ordering, you should use the constructor
     * with the ordering specified below.
     */
    public X509Name(
        Hashtable  attributes)
    {
        this(null, attributes);
    }

    /**
     * constructor from a table of attributes with ordering.
     * <p>
     * it's is assumed the table contains OID/String pairs, and the contents
     * of the table are copied into an internal table as part of the 
     * construction process. The ordering vector should contain the OIDs
     * in the order they are meant to be encoded or printed in toString.
     */
    public X509Name(
        Vector      ordering,
        Hashtable   attributes)
    {
        if (ordering != null)
        {
            for (int i = 0; i != ordering.size(); i++)
            {
                this.ordering.addElement(ordering.elementAt(i));
            }
        }
        else
        {
            Enumeration     e = attributes.keys();

            while (e.hasMoreElements())
            {
                this.ordering.addElement(e.nextElement());
            }
        }

        for (int i = 0; i != this.ordering.size(); i++)
        {
            DERObjectIdentifier     oid = (DERObjectIdentifier)this.ordering.elementAt(i);

            if (OIDLookUp.get(oid) == null)
            {
                throw new IllegalArgumentException("Unknown object id - " + oid.getId() + " - passed to distinguished name");
            }

            if (attributes.get(oid) == null)
            {
                throw new IllegalArgumentException("No attribute for object id - " + oid.getId() + " - passed to distinguished name");
            }

            this.values.addElement(attributes.get(oid)); // copy the hash table
        }
    }

    /**
     * takes two vectors one of the oids and the other of the values.
     */
    public X509Name(
        Vector  ordering,
        Vector  values)
    {
        if (ordering.size() != values.size())
        {
            throw new IllegalArgumentException("ordering vector must be same length as values.");
        }

        for (int i = 0; i < ordering.size(); i++)
        {
            this.ordering.addElement(ordering.elementAt(i));
            this.values.addElement(values.elementAt(i));
        }
    }

    /**
     * takes an X509 dir name as a string of the format "C=AU, ST=Victoria", or
     * some such, converting it into an ordered set of name attributes.
     */
    public X509Name(
        String  dirName)
    {
        X509NameTokenizer   nTok = new X509NameTokenizer(dirName);

        while (nTok.hasMoreTokens())
        {
            String  token = nTok.nextToken();
            int     index = token.indexOf('=');

            if (index == -1)
            {
                throw new IllegalArgumentException("badly formated directory string");
            }

            String  name = token.substring(0, index);
            String  value = token.substring(index + 1);

            DERObjectIdentifier oid = (DERObjectIdentifier)SymbolLookUp.get(name);
            if (oid == null)
            {
                throw new IllegalArgumentException("Unknown object id - " + name + " - passed to distinguished name");
            }

            this.ordering.addElement(oid);
            this.values.addElement(value);
        }
    }

    public DERObject getDERObject()
    {
        DERConstructedSequence seq = new DERConstructedSequence();

        for (int i = 0; i != ordering.size(); i++)
        {
            DERConstructedSequence  s = new DERConstructedSequence();
            DERObjectIdentifier     oid = (DERObjectIdentifier)ordering.elementAt(i);

            s.addObject(oid);
            if (oid.equals(EmailAddress))
            {
                s.addObject(new DERIA5String((String)values.elementAt(i)));
            }
            else
            {
                s.addObject(new DERPrintableString((String)values.elementAt(i)));
            }

            seq.addObject(new DERSet(s));
        }

        return seq;
    }

    public int hashCode()
    {
        DERConstructedSequence  seq = (DERConstructedSequence)this.getDERObject();
        Enumeration             e = seq.getObjects();
        int                     hashCode = 0;

        while (e.hasMoreElements())
        {
            DERSet                  set = (DERSet)e.nextElement();
            DERConstructedSequence  s = (DERConstructedSequence)set.getSequence();
            for (int i = s.getSize() - 1; i >= 0; i--)
            {
                hashCode ^= s.getObjectAt(i).hashCode();
            }
        }

        return hashCode;
    }

    public boolean equals(
        Object o)
    {
        if (o == null || !(o instanceof X509Name))
        {
            return false;
        }

        X509Name        other = (X509Name)o;

        DERConstructedSequence  sq1 = (DERConstructedSequence)this.getDERObject();
        DERConstructedSequence  sq2 = (DERConstructedSequence)other.getDERObject();

        if (sq1.getSize() != sq2.getSize())
        {
            return false;
        }

        Enumeration     e1 = sq1.getObjects();
        Enumeration     e2 = sq2.getObjects();

        while (e1.hasMoreElements())
        {
            DERSet                  set1 = (DERSet)e1.nextElement();
            DERConstructedSequence  s1 = (DERConstructedSequence)set1.getSequence();
            DERSet                  set2 = (DERSet)e2.nextElement();
            DERConstructedSequence  s2 = (DERConstructedSequence)set2.getSequence();
            if (s1.getSize() != s2.getSize())
            {
                return false;
            }

            for (int i = s1.getSize() - 1; i >= 0; i--)
            {
                if (!s1.getObjectAt(i).equals(s2.getObjectAt(i)))
                {
                    return false;
                }
            }
        }

        return true;
    }

    public String toString()
    {
        StringBuffer            buf = new StringBuffer();
        boolean                 first = true;
        Enumeration             e1 = ordering.elements();
        Enumeration             e2 = values.elements();

        while (e1.hasMoreElements())
        {
            Object                  oid = e1.nextElement();
            String                  sym = (String)OIDLookUp.get(oid);
            
            if (first)
            {
                first = false;
            }
            else
            {
                buf.append(", ");
            }

            if (sym != null)
            {
                buf.append(sym);
            }
            else
            {
                buf.append(((DERObjectIdentifier)oid).getId());
            }

            buf.append("=");
            buf.append((String)e2.nextElement());
        }

        return buf.toString();
    }
}
