package org.bouncycastle.asn1.pkcs;

import java.util.Enumeration;

import org.bouncycastle.asn1.*;
import org.bouncycastle.asn1.x509.*;

/**
 * a PKCS#7 signed data object.
 */
public class SignedData
    implements DEREncodable, PKCSObjectIdentifiers
{
    private DERInteger              version;
    private DERObject               digestAlgorithms;
    private ContentInfo             contentInfo;
    private DERObject               certificates;
    private DERObject               crls;

    public SignedData(
        ASN1Sequence seq)
    {
        Enumeration     e = seq.getObjects();

        version = (DERInteger)e.nextElement();
        digestAlgorithms = ((DERSet)e.nextElement()).getSequence();
        contentInfo = ContentInfo.getInstance(e.nextElement());

        while (e.hasMoreElements())
        {
            DERObject o = (DERObject)e.nextElement();

            //
            // an interesting feature of SignedData is that there appear to be varying implementations...
            // for the moment we ignore anything which doesn't fit.
            //
            if (o instanceof DERTaggedObject)
            {
                DERTaggedObject tagged = (DERTaggedObject)o;

                switch (tagged.getTagNo())
                {
                case 0:
                    certificates = tagged.getObject();
                    break;
                case 1:
                    crls = tagged.getObject();
                    break;
                default:
                    throw new IllegalArgumentException("unknown tag value " + tagged.getTagNo());
                }
            }
        }
    }

    public DERInteger getVersion()
    {
        return version;
    }

    public ContentInfo getContentInfo()
    {
        return contentInfo;
    }

    public DERObject getCertficates()
    {
        return certificates;
    }

    /**
     * <pre>
     *  SignedData ::= SEQUENCE {
     *      version Version,
     *      digestAlgorithms DigestAlgorithmIdentifiers,
     *      contentInfo ContentInfo,
     *      certificates
     *          [0] IMPLICIT ExtendedCertificatesAndCertificates
     *                   OPTIONAL,
     *      crls
     *          [1] IMPLICIT CertificateRevocationLists OPTIONAL,
     *      signerInfos SignerInfos }
     * </pre>
     */
    public DERObject getDERObject()
    {
        ASN1Sequence seq = new DERConstructedSequence();

        seq.addObject(version);
        seq.addObject(digestAlgorithms);
        seq.addObject(contentInfo);

        if (certificates != null)
        {
            seq.addObject(new DERTaggedObject(false, 0, certificates));
        }

        if (crls != null)
        {
            seq.addObject(new DERTaggedObject(false, 1, crls));
        }

        return seq;
    }
}
