package org.bouncycastle.asn1.pkcs;

import org.bouncycastle.asn1.*;
import org.bouncycastle.asn1.x509.*;

/**
 * PKCS10
 *
 * <pre>
 *  CertificationRequestInfo ::= SEQUENCE {
 *   version             INTEGER { v1(0) } (v1,...),
 *   subject             Name,
 *   subjectPKInfo   SubjectPublicKeyInfo{{ PKInfoAlgorithms }},
 *   attributes          [0] Attributes{{ CRIAttributes }}
 *  }
 *
 *  Attributes { ATTRIBUTE:IOSet } ::= SET OF Attribute{{ IOSet }}
 *
 *  Attribute { ATTRIBUTE:IOSet } ::= SEQUENCE {
 *    type    ATTRIBUTE.&id({IOSet}),
 *    values  SET SIZE(1..MAX) OF ATTRIBUTE.&Type({IOSet}{@type})
 *  }
 * </pre>
 */
public class CertificationRequestInfo
    implements DEREncodable
{
    DERInteger              version = new DERInteger(0);
    X509Name                subject;
    SubjectPublicKeyInfo    subjectPKInfo;
    DERConstructedSet       attributes = null;
    boolean                 isexplicit = false;

    public static CertificationRequestInfo getInstance(
        Object  obj)
    {
        if (obj instanceof CertificationRequestInfo)
        {
            return (CertificationRequestInfo)obj;
        }
        else if (obj instanceof ASN1Sequence)
        {
            return new CertificationRequestInfo((ASN1Sequence)obj);
        }

        throw new IllegalArgumentException("unknown object in factory");
    }

    public CertificationRequestInfo(
        X509Name                subject,
        SubjectPublicKeyInfo    pkInfo,
        DERConstructedSet       attributes)
    {
        this.subject = subject;
        this.subjectPKInfo = pkInfo;
        this.attributes = attributes;

        if ((subject == null) || (version == null) || (subjectPKInfo == null))
        {
            throw new IllegalArgumentException("Not all mandatory fields set in CertificationRequestInfo generator.");
        }
    }

    public CertificationRequestInfo(
        ASN1Sequence  seq)
    {
        version = (DERInteger)seq.getObjectAt(0);

        subject = X509Name.getInstance(seq.getObjectAt(1));
        subjectPKInfo = SubjectPublicKeyInfo.getInstance(seq.getObjectAt(2));

        DERTaggedObject tagobj = (DERTaggedObject)seq.getObjectAt(3);

        if (!tagobj.isEmpty())
        {
            DERConstructedSequence s = (DERConstructedSequence)tagobj.getObject();

            attributes = new DERConstructedSet();

            //
            // if there is only one attribute we loose the surrounding set, due
            // to implicit tagging, so the sequence we get back may be for an
            // attribute not a sequence of attributes.
            //
            if (s.getObjectAt(0) instanceof ASN1Sequence)
            {
                for (int i = 0; i != s.getSize(); i++)
                {
                    attributes.addObject((DEREncodable)s.getObjectAt(i));
                }
            }
            else
            {
                attributes.addObject((DEREncodable)s);
            }
        }

        isexplicit = tagobj.isExplicit();

        if ((subject == null) || (version == null) || (subjectPKInfo == null))
        {
            throw new IllegalArgumentException("Not all mandatory fields set in CertificationRequestInfo generator.");
        }
    }

    public DERInteger getVersion()
    {
        return version;
    }

    public X509Name getSubject()
    {
        return subject;
    }

    public SubjectPublicKeyInfo getSubjectPublicKeyInfo()
    {
        return subjectPKInfo;
    }

    public DERConstructedSet getAttributes()
    {
        return attributes;
    }

    public DERObject getDERObject()
    {
        DERConstructedSequence  seq = new DERConstructedSequence();

        seq.addObject(version);
        seq.addObject(subject);
        seq.addObject(subjectPKInfo);

        if (attributes == null)
        {
            seq.addObject(new DERTaggedObject(false, 0));
        }
        else
        {
            seq.addObject(new DERTaggedObject(isexplicit, 0, attributes));
        }

        return seq;
    }
}
