package org.bouncycastle.asn1.cms;

import org.bouncycastle.asn1.BERConstructedSequence;
import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.BERTaggedObject;

/** 
 * <br>
 * <pre>
 * RFC 2630
 * 
 * SignerInfo ::= SEQUENCE {
 * 	version CMSVersion,
 * 	sid SignerIdentifier,
 * 	digestAlgorithm DigestAlgorithmIdentifier,
 * 	signedAttrs [0] IMPLICIT SignedAttributes OPTIONAL,
 * 	signatureAlgorithm SignatureAlgorithmIdentifier,
 * 	signature SignatureValue,
 * 	unsignedAttrs [1] IMPLICIT UnsignedAttributes OPTIONAL 
 * }
 * </pre>
 * 
 * @version 1.0; 
 *     
 */

public class SignerInfo implements DEREncodable {

	/*
	 *
	 *  VARIABLES
	 *
	 */

	private CMSVersion                   version;
	private SignerIdentifier             sid;
	private DigestAlgorithmIdentifier    digestAlgorithm;
	private SignedAttributes             signedAttrs;
	private SignatureAlgorithmIdentifier signatureAlgorithm;
	private SignatureValue               signature;
	private UnsignedAttributes           unsignedAttrs;

	/*
	 *
	 *  CONSTRUCTORS
	 *
	 */
	
	public SignerInfo(CMSVersion                   _version,
					  SignerIdentifier             _sid,
					  DigestAlgorithmIdentifier    _digestAlgorithm,
					  SignedAttributes             _signedAttrs,
					  SignatureAlgorithmIdentifier _signatureAlgorithm,
					  SignatureValue               _signature,
					  UnsignedAttributes           _unsignedAttrs) {
		
		setVersion(_version);
		setSid(_sid);
		setDigestAlgorithm(_digestAlgorithm);
		setSignedAttrs(_signedAttrs);
		setSignatureAlgorithm(_signatureAlgorithm);
		setSignature(_signature);
		setUnsignedAttrs(_unsignedAttrs);
	}
	
	public SignerInfo(BERConstructedSequence _seq) {
		int i = 0;
		Object _tmp;
		
		version         = CMSVersion.getInstance(_seq.getObjectAt(i++));
		sid             = SignerIdentifier.getInstance(_seq.getObjectAt(i++));
		digestAlgorithm = DigestAlgorithmIdentifier.getInstance(_seq.getObjectAt(i++));
		
		_tmp = _seq.getObjectAt(i++); 
		if(_tmp instanceof BERTaggedObject) {
			signedAttrs = SignedAttributes.getInstance(_tmp);
			_tmp = _seq.getObjectAt(i++);
		}
		signatureAlgorithm = SignatureAlgorithmIdentifier.getInstance(_tmp);
		signature          = SignatureValue.getInstance(_seq.getObjectAt(i++));
		
		if(_seq.getSize() > i) {
			unsignedAttrs = UnsignedAttributes.getInstance(_seq.getObjectAt(i));
		}
	}

	public SignerInfo(SignerInfo _orig) {
		version            = _orig.version;
		sid                = _orig.sid;
		digestAlgorithm    = _orig.digestAlgorithm;
		signedAttrs        = _orig.signedAttrs;
		signatureAlgorithm = _orig.signatureAlgorithm;
		signature          = _orig.signature;
		unsignedAttrs      = _orig.unsignedAttrs;
	}

	public static SignerInfo getInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof SignerInfo) {
			return (SignerInfo)_obj;
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new SignerInfo((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid SignerInfo");
	} 

	public static SignerInfo newInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof SignerInfo) {
			return new SignerInfo((SignerInfo)_obj);
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new SignerInfo((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid SignerInfo");
	} 


	/*
	 *
	 *  BUSINESS METHODS
	 *
	 */

	public CMSVersion getVersion() {
		return version;
	}

	private void setVersion(CMSVersion _version) {
		version = _version;
	}


	public SignerIdentifier getSid() {
		return sid;
	}

	private void setSid(SignerIdentifier _sid) {
		sid = _sid;
	}


	public DigestAlgorithmIdentifier getDigestAlgorithm() {
		return digestAlgorithm;
	}

	private void setDigestAlgorithm(DigestAlgorithmIdentifier _digestAlgorithm) {
		digestAlgorithm = _digestAlgorithm;
	}


	public SignedAttributes getSignedAttrs() {
		return signedAttrs;
	}

	private void setSignedAttrs(SignedAttributes _signedAttrs) {
		signedAttrs = _signedAttrs;
	}


	public SignatureAlgorithmIdentifier getSignatureAlgorithm() {
		return signatureAlgorithm;
	}

	private void setSignatureAlgorithm(SignatureAlgorithmIdentifier _signatureAlgorithm) {
		signatureAlgorithm = _signatureAlgorithm;
	}


	public SignatureValue getSignature() {
		return signature;
	}

	private void setSignature(SignatureValue _signature) {
		signature = _signature;
	}


	public UnsignedAttributes getUnsignedAttrs() {
		return unsignedAttrs;
	}

	private void setUnsignedAttrs(UnsignedAttributes _unsignedAttrs) {
		unsignedAttrs = _unsignedAttrs;
	}



	public DERObject getDERObject() {
		BERConstructedSequence _seq = new BERConstructedSequence();
		_seq.addObject(version);
		_seq.addObject(sid);
		_seq.addObject(digestAlgorithm);

		if(signedAttrs != null) {
			_seq.addObject(new BERTaggedObject(false, 0, signedAttrs.getDERObject()));
		}

		_seq.addObject(signature);

		if(unsignedAttrs != null) {
			_seq.addObject(new BERTaggedObject(false, 1, unsignedAttrs.getDERObject()));
		}

		return _seq;
	}

	/*
	 *
	 *  INTERNAL METHODS
	 *
	 */


}


/**
 * Version History
 * ~~~~~~~~~~~~~~~
 * 1.0
 *     - Created [CK]
 * 
 */
