package org.bouncycastle.asn1.cms;

import org.bouncycastle.asn1.BERConstructedSequence;
import org.bouncycastle.asn1.BERTaggedObject;
import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERObject;

/** 
 * <br>
 * <pre>
 * RFC 2630
 * 
 * SignedData ::= SEQUENCE {
 * 	version CMSVersion,
 * 	digestAlgorithms DigestAlgorithmIdentifiers,
 * 	encapContentInfo EncapsulatedContentInfo,
 * 	certificates [0] IMPLICIT CertificateSet OPTIONAL,
 * 	crls [1] IMPLICIT CertificateRevocationLists OPTIONAL,
 * 	signerInfos SignerInfos 
 * }
 * </pre>
 * 
 * @version 1.0; 
 *     
 */

public class SignedData implements DEREncodable {

	/*
	 *
	 *  VARIABLES
	 *
	 */

	private CMSVersion                 version;
	private DigestAlgorithmIdentifiers digestAlgorithms;
	private EncapsulatedContentInfo    encapContentInfo;
	private CertificateSet             certificates;
	private CertificateRevocationLists crls;
	private SignerInfos                signerInfos;

	/*
	 *
	 *  CONSTRUCTORS
	 *
	 */

	public SignedData(CMSVersion                 _version,
					  DigestAlgorithmIdentifiers _digestAlgorithms,
					  EncapsulatedContentInfo    _encapContentInfo,
					  CertificateSet             _certificates,
					  CertificateRevocationLists _crls,
					  SignerInfos                _signerInfos) {
		
		setVersion(_version);
		setDigestAlgorithms(_digestAlgorithms);
		setEncapContentInfo(_encapContentInfo);
		setCertificates(_certificates);
		setCrls(_crls);
		setSignerInfos(_signerInfos);
	}
	
	public SignedData(BERConstructedSequence _seq) {
		int i = 0;
		version          = CMSVersion.getInstance(_seq.getObjectAt(i++));
		digestAlgorithms = DigestAlgorithmIdentifiers.getInstance(_seq.getObjectAt(i++));
		encapContentInfo = EncapsulatedContentInfo.getInstance(_seq.getObjectAt(i++));
		
		switch(_seq.getSize()) {
			case 4 :
				break;
			case 5 :
				BERTaggedObject _dto = (BERTaggedObject)_seq.getObjectAt(i++);
				switch(_dto.getTagNo()) {
					case 0 :
						certificates = CertificateSet.getInstance(_dto);
						break;
					case 1 :
						crls = CertificateRevocationLists.getInstance(_dto);
						break;
					default:
						throw new IllegalArgumentException("Invalid SignedData");
				}
				break;
			case 6 :
				certificates = CertificateSet.getInstance(_seq.getObjectAt(i++));
				crls         = CertificateRevocationLists.getInstance(_seq.getObjectAt(i++));
				break;
			default:
				throw new IllegalArgumentException("Invalid SignedData");
		}

		signerInfos = SignerInfos.getInstance(_seq.getObjectAt(i));
	}

	public SignedData(SignedData _orig) {
		version          = _orig.version;
		digestAlgorithms = _orig.digestAlgorithms;
		encapContentInfo = _orig.encapContentInfo;
		certificates     = _orig.certificates;
		crls             = _orig.crls;
		signerInfos      = _orig.signerInfos;
	}
	
	public static SignedData getInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof SignedData) {
			return (SignedData)_obj;
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new SignedData((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid SignedData");
	}

	public static SignedData newInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof SignedData) {
			return new SignedData((SignedData)_obj);
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new SignedData((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid SignedData");
	}

	/*
	 *
	 *  BUSINESS METHODS
	 *
	 */

	public CMSVersion getVersion() {
		return version;
	}

	private void setVersion(CMSVersion _version) {
		version = _version;
	}


	public DigestAlgorithmIdentifiers getDigestAlgorithms() {
		return digestAlgorithms;
	}

	private void setDigestAlgorithms(DigestAlgorithmIdentifiers _digestAlgorithms) {
		digestAlgorithms = _digestAlgorithms;
	}


	public EncapsulatedContentInfo getEncapContentInfo() {
		return encapContentInfo;
	}

	private void setEncapContentInfo(EncapsulatedContentInfo _encapContentInfo) {
		encapContentInfo = _encapContentInfo;
	}


	public CertificateSet getCertificates() {
		return certificates;
	}

	private void setCertificates(CertificateSet _certificates) {
		certificates = _certificates;
	}


	public CertificateRevocationLists getCrls() {
		return crls;
	}

	private void setCrls(CertificateRevocationLists _crls) {
		crls = _crls;
	}


	public SignerInfos getSignerInfos() {
		return signerInfos;
	}

	private void setSignerInfos(SignerInfos _signerInfos) {
		signerInfos = _signerInfos;
	}



	public DERObject getDERObject() {
		BERConstructedSequence _seq = new BERConstructedSequence();
		_seq.addObject(version);
		_seq.addObject(digestAlgorithms);
		_seq.addObject(encapContentInfo);

		if(certificates != null) {
			_seq.addObject(new BERTaggedObject(false, 0, certificates.getDERObject()));
		}
		
		if(crls != null) {
			_seq.addObject(new BERTaggedObject(false, 1, crls.getDERObject()));
		}
		
		_seq.addObject(signerInfos);
		return _seq;
	}

	/*
	 *
	 *  INTERNAL METHODS
	 *
	 */


}


/**
 * Version History
 * ~~~~~~~~~~~~~~~
 * 1.0
 *     - Created [CK]
 * 
 */
