package org.bouncycastle.asn1.cms;

import org.bouncycastle.asn1.BERConstructedSequence;
import org.bouncycastle.asn1.BERTaggedObject;
import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERObject;

/** 
 * <br>
 * <pre>
 * RFC 2630
 * 
 * KeyTransRecipientInfo ::= SEQUENCE {
 * 	version CMSVersion,  -- always set to 0 or 2
 * 	rid RecipientIdentifier,
 * 	keyEncryptionAlgorithm KeyEncryptionAlgorithmIdentifier,
 * 	encryptedKey EncryptedKey 
 * }
 * </pre>
 * 
 * @version 1.0; 
 *     
 */

public class KeyTransRecipientInfo implements DEREncodable {

	/*
	 *
	 *  VARIABLES
	 *
	 */

	private CMSVersion                       version;
	private RecipientIdentifier              rid;
	private KeyEncryptionAlgorithmIdentifier keyEncryptionAlgorithm;
	private EncryptedKey                     encryptedKey;

	/*
	 *
	 *  CONSTRUCTORS
	 *
	 */

	public KeyTransRecipientInfo(CMSVersion                       _version,
								 RecipientIdentifier              _rid,
								 KeyEncryptionAlgorithmIdentifier _keyEncryptionAlgorithm,
								 EncryptedKey                     _encryptedKey) {
		
		setVersion(_version);
		setRecipientIdentifier(_rid);
		setKeyEncryptionAlgorithm(_keyEncryptionAlgorithm);
		setEncryptedKey(_encryptedKey);
	}
	
	public KeyTransRecipientInfo(BERConstructedSequence _seq) {
		version                = CMSVersion.getInstance(_seq.getObjectAt(0));
		rid                    = RecipientIdentifier.getInstance(_seq.getObjectAt(1));
		keyEncryptionAlgorithm = KeyEncryptionAlgorithmIdentifier.getInstance(_seq.getObjectAt(2));
		encryptedKey           = EncryptedKey.getInstance(_seq.getObjectAt(3));
	}

	public KeyTransRecipientInfo(KeyTransRecipientInfo _orig) {
		version                = _orig.version;
		rid                    = _orig.rid;
		keyEncryptionAlgorithm = _orig.keyEncryptionAlgorithm;
		encryptedKey           = _orig.encryptedKey;
	}

	public static KeyTransRecipientInfo getInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof KeyTransRecipientInfo) {
			return (KeyTransRecipientInfo)_obj;
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new KeyTransRecipientInfo((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid KeyTransRecipientInfo");
	} 

	public static KeyTransRecipientInfo newInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof KeyTransRecipientInfo) {
			return new KeyTransRecipientInfo((KeyTransRecipientInfo)_obj);
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new KeyTransRecipientInfo((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid KeyTransRecipientInfo");
	} 

	/*
	 *
	 *  BUSINESS METHODS
	 *
	 */

	public CMSVersion getVersion() {
		return version;
	}

	private void setVersion(CMSVersion _version) {
		version = _version;
	}


	public RecipientIdentifier getRecipientIdentifier() {
		return rid;
	}

	private void setRecipientIdentifier(RecipientIdentifier _rid) {
		rid = _rid;
	}


	public KeyEncryptionAlgorithmIdentifier getKeyEncryptionAlgorithm() {
		return keyEncryptionAlgorithm;
	}

	private void setKeyEncryptionAlgorithm(KeyEncryptionAlgorithmIdentifier _keyEncryptionAlgorithm) {
		keyEncryptionAlgorithm = _keyEncryptionAlgorithm;
	}


	public EncryptedKey getEncryptedKey() {
		return encryptedKey;
	}

	private void setEncryptedKey(EncryptedKey _encryptedKey) {
		encryptedKey = _encryptedKey;
	}



	public DERObject getDERObject() {
		BERConstructedSequence _seq = new BERConstructedSequence();
		_seq.addObject(version);
		_seq.addObject(rid);
		_seq.addObject(keyEncryptionAlgorithm);
		_seq.addObject(encryptedKey);
		return _seq;
	}

	/*
	 *
	 *  INTERNAL METHODS
	 *
	 */


}


/**
 * Version History
 * ~~~~~~~~~~~~~~~
 * 1.0
 *     - Created [CK]
 * 
 */
