package org.bouncycastle.asn1.cms;

import org.bouncycastle.asn1.BERConstructedSequence;
import org.bouncycastle.asn1.BERTaggedObject;
import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERObject;

/** 
 * <br>
 * <pre>
 * RFC 2630
 * 
 * KeyAgreeRecipientInfo ::= SEQUENCE {
 * 	version CMSVersion,  -- always set to 3
 * 	originator [0] EXPLICIT OriginatorIdentifierOrKey,
 * 	ukm [1] EXPLICIT UserKeyingMaterial OPTIONAL,
 * 	keyEncryptionAlgorithm KeyEncryptionAlgorithmIdentifier,
 * 	recipientEncryptedKeys RecipientEncryptedKeys 
 * }
 * </pre>
 * 
 * @version 1.0; 
 *     
 */

public class KeyAgreeRecipientInfo implements DEREncodable {

	/*
	 *
	 *  VARIABLES
	 *
	 */

	private CMSVersion                       version;
	private OriginatorIdentifierOrKey        originator;
	private UserKeyingMaterial               ukm;
	private KeyEncryptionAlgorithmIdentifier keyEncryptionAlgorithm;
	private RecipientEncryptedKeys           recipientEncryptedKeys;
	
	/*
	 *
	 *  CONSTRUCTORS
	 *
	 */

	public KeyAgreeRecipientInfo(CMSVersion                       _version,
								 OriginatorIdentifierOrKey        _originator,
								 UserKeyingMaterial               _ukm,
								 KeyEncryptionAlgorithmIdentifier _keyEncryptionAlgorithm,
								 RecipientEncryptedKeys           _recipientEncryptedKeys) {
		
		setVersion(_version);
		setOriginator(_originator);
		setUserKeyingMaterial(_ukm);
		setKeyEncryptionAlgorithm(_keyEncryptionAlgorithm);
		setRecipientEncryptedKeys(_recipientEncryptedKeys);
	}
	
	public KeyAgreeRecipientInfo(BERConstructedSequence _seq) {
		int i = 0;
		version    = CMSVersion.getInstance(_seq.getObjectAt(i++));
		originator = OriginatorIdentifierOrKey.getInstance(_seq.getObjectAt(i++));

		if(_seq.getSize() == 5) {
			ukm = UserKeyingMaterial.getInstance(_seq.getObjectAt(i++));
		}

		keyEncryptionAlgorithm = KeyEncryptionAlgorithmIdentifier.getInstance(_seq.getObjectAt(i++));
		recipientEncryptedKeys = RecipientEncryptedKeys.getInstance(_seq.getObjectAt(i++));
	}
	
	public KeyAgreeRecipientInfo(KeyAgreeRecipientInfo _orig) {
		version                = _orig.version;
		originator             = _orig.originator;
		ukm                    = _orig.ukm;
		keyEncryptionAlgorithm = _orig.keyEncryptionAlgorithm;
		recipientEncryptedKeys = _orig.recipientEncryptedKeys;
	}
	
	public static KeyAgreeRecipientInfo getInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof KeyAgreeRecipientInfo) {
			return (KeyAgreeRecipientInfo)_obj;
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new KeyAgreeRecipientInfo((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid KeyAgreeRecipientInfo");
	} 

	public static KeyAgreeRecipientInfo newInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof KeyAgreeRecipientInfo) {
			return new KeyAgreeRecipientInfo((KeyAgreeRecipientInfo)_obj);
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new KeyAgreeRecipientInfo((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid KeyAgreeRecipientInfo");
	} 

	/*
	 *
	 *  BUSINESS METHODS
	 *
	 */

	public CMSVersion getVersion() {
		return version;
	}

	private void setVersion(CMSVersion _version) {
		version = _version;
	}


	public OriginatorIdentifierOrKey getOriginator() {
		return originator;
	}

	private void setOriginator(OriginatorIdentifierOrKey _originator) {
		originator = _originator;
	}


	public UserKeyingMaterial getUserKeyingMaterial() {
		return ukm;
	}

	private void setUserKeyingMaterial(UserKeyingMaterial _ukm) {
		ukm = _ukm;
	}


	public KeyEncryptionAlgorithmIdentifier getKeyEncryptionAlgorithm() {
		return keyEncryptionAlgorithm;
	}

	private void setKeyEncryptionAlgorithm(KeyEncryptionAlgorithmIdentifier _keyEncryptionAlgorithm) {
		keyEncryptionAlgorithm = _keyEncryptionAlgorithm;
	}


	public RecipientEncryptedKeys getRecipientEncryptedKeys() {
		return recipientEncryptedKeys;
	}

	private void setRecipientEncryptedKeys(RecipientEncryptedKeys _recipientEncryptedKeys) {
		recipientEncryptedKeys = _recipientEncryptedKeys;
	}



	public DERObject getDERObject() {
		BERConstructedSequence _seq = new BERConstructedSequence();
		_seq.addObject(version);
		_seq.addObject(new BERTaggedObject(true, 0, originator.getDERObject()));
		
		if(ukm != null) {
			_seq.addObject(new BERTaggedObject(true, 1, ukm.getDERObject()));
		}
		
		_seq.addObject(keyEncryptionAlgorithm);
		_seq.addObject(recipientEncryptedKeys);
		return _seq;
	}

	/*
	 *
	 *  INTERNAL METHODS
	 *
	 */


}


/**
 * Version History
 * ~~~~~~~~~~~~~~~
 * 1.0
 *     - Created [CK]
 * 
 */
