package org.bouncycastle.asn1.cms;

import org.bouncycastle.asn1.BERConstructedSequence;
import org.bouncycastle.asn1.BERTaggedObject;
import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERObject;

/** 
 * <br>
 * <pre>
 * RFC 2630
 * 
 * KEKRecipientInfo ::= SEQUENCE {
 * 	version CMSVersion,  -- always set to 4
 * 	kekid KEKIdentifier,
 * 	keyEncryptionAlgorithm KeyEncryptionAlgorithmIdentifier,
 * 	encryptedKey EncryptedKey 
 * }
 * </pre>
 * 
 * @version 1.0; 
 *     
 */

public class KEKRecipientInfo implements DEREncodable {

	/*
	 *
	 *  VARIABLES
	 *
	 */

	private CMSVersion                       version;
	private KEKIdentifier                    kekid;
	private KeyEncryptionAlgorithmIdentifier keyEncryptionAlgorithm;
	private EncryptedKey                     encryptedKey;

	/*
	 *
	 *  CONSTRUCTORS
	 *
	 */

	public KEKRecipientInfo(CMSVersion                       _version,
							KEKIdentifier                    _kekid,
							KeyEncryptionAlgorithmIdentifier _keyEncryptionAlgorithm,
							EncryptedKey                     _encryptedKey) {
	
		setVersion(_version);
		setKekid(_kekid);
		setKeyEncryptionAlgorithm(_keyEncryptionAlgorithm);
		setEncryptedKey(_encryptedKey);
	}
	
	
	public KEKRecipientInfo(BERConstructedSequence _seq) {
		version                = CMSVersion.getInstance(_seq.getObjectAt(0));
		kekid                  = KEKIdentifier.getInstance(_seq.getObjectAt(1));
		keyEncryptionAlgorithm = KeyEncryptionAlgorithmIdentifier.getInstance(_seq.getObjectAt(2));
		encryptedKey           = EncryptedKey.getInstance(_seq.getObjectAt(3));
	}

	public KEKRecipientInfo(KEKRecipientInfo _orig) {
		version                = _orig.version;
		kekid                  = _orig.kekid;
		keyEncryptionAlgorithm = _orig.keyEncryptionAlgorithm;
		encryptedKey           = _orig.encryptedKey;
	}

	public static KEKRecipientInfo getInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof KEKRecipientInfo) {
			return (KEKRecipientInfo)_obj;
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new KEKRecipientInfo((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid KEKRecipientInfo");
	}

	public static KEKRecipientInfo newInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof KEKRecipientInfo) {
			return new KEKRecipientInfo((KEKRecipientInfo)_obj);
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new KEKRecipientInfo((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid KEKRecipientInfo");
	}

	/*
	 *
	 *  BUSINESS METHODS
	 *
	 */

	public CMSVersion getVersion() {
		return version;
	}

	private void setVersion(CMSVersion _version) {
		version = _version;
	}


	public KEKIdentifier getKekid() {
		return kekid;
	}

	private void setKekid(KEKIdentifier _kekid) {
		kekid = _kekid;
	}


	public KeyEncryptionAlgorithmIdentifier getKeyEncryptionAlgorithm() {
		return keyEncryptionAlgorithm;
	}

	private void setKeyEncryptionAlgorithm(KeyEncryptionAlgorithmIdentifier _keyEncryptionAlgorithm) {
		keyEncryptionAlgorithm = _keyEncryptionAlgorithm;
	}


	public EncryptedKey getEncryptedKey() {
		return encryptedKey;
	}

	private void setEncryptedKey(EncryptedKey _encryptedKey) {
		encryptedKey = _encryptedKey;
	}



	public DERObject getDERObject() {
		BERConstructedSequence _seq = new BERConstructedSequence();
		_seq.addObject(version);
		_seq.addObject(kekid);
		_seq.addObject(keyEncryptionAlgorithm);
		_seq.addObject(encryptedKey);
		return _seq;
	}

	/*
	 *
	 *  INTERNAL METHODS
	 *
	 */


}


/**
 * Version History
 * ~~~~~~~~~~~~~~~
 * 1.0
 *     - Created [CK]
 * 
 */
