package org.bouncycastle.asn1.cms;

import org.bouncycastle.asn1.BERConstructedSequence;
import org.bouncycastle.asn1.BERTaggedObject;
import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERObject;

/** 
 * <br>
 * <pre>
 * RFC 2630
 * 
 * EnvelopedData ::= SEQUENCE {
 * 	version CMSVersion,
 * 	originatorInfo [0] IMPLICIT OriginatorInfo OPTIONAL,
 * 	recipientInfos RecipientInfos,
 * 	encryptedContentInfo EncryptedContentInfo,
 * 	unprotectedAttrs [1] IMPLICIT UnprotectedAttributes OPTIONAL 
 * }
 * </pre>
 * 
 * @version 1.0; 
 *     
 */

public class EnvelopedData implements DEREncodable {

	/*
	 *
	 *  VARIABLES
	 *
	 */

	private CMSVersion            version;
	private OriginatorInfo        originatorInfo;
	private RecipientInfos        recipientInfos;
	private EncryptedContentInfo  encryptedContentInfo;
	private UnprotectedAttributes unprotectedAttrs;

	/*
	 *
	 *  CONSTRUCTORS
	 *
	 */

	public EnvelopedData(CMSVersion            _version,
	                     OriginatorInfo        _originatorInfo,
	                     RecipientInfos        _recipientInfos,
	                     EncryptedContentInfo  _encryptedContentInfo,
	                     UnprotectedAttributes _unprotectedAttrs) {
		
		setVersion(_version);
		setOriginatorInfo(_originatorInfo);
		setRecipientInfos(_recipientInfos);
		setEncryptedContentInfo(_encryptedContentInfo);
		setUnprotectedAttrs(_unprotectedAttrs);
	}
	                     
	public EnvelopedData(BERConstructedSequence _seq) {
		Object _tmp;
		int i = 0;
		
		version = CMSVersion.getInstance(_seq.getObjectAt(i++));
		
		_tmp = _seq.getObjectAt(i++);
		if(_tmp instanceof BERTaggedObject) {
			originatorInfo = OriginatorInfo.getInstance(_tmp);
			_tmp = _seq.getObjectAt(i++);
		}
		recipientInfos = RecipientInfos.getInstance(_tmp);
		
		encryptedContentInfo = EncryptedContentInfo.getInstance(_seq.getObjectAt(i++));
		
		if(_seq.getSize() > i) {
			unprotectedAttrs = UnprotectedAttributes.getInstance(_seq.getObjectAt(i));
		}
		
	}
	
	public EnvelopedData(EnvelopedData _orig) {
		originatorInfo       = _orig.originatorInfo;
		recipientInfos       = _orig.recipientInfos;
		encryptedContentInfo = _orig.encryptedContentInfo;
		unprotectedAttrs     = _orig.unprotectedAttrs;
	}

	public static EnvelopedData getInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof EnvelopedData) {
			return (EnvelopedData)_obj;
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new EnvelopedData((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid EnvelopedData");
	}

	public static EnvelopedData newInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof EnvelopedData) {
			return new EnvelopedData((EnvelopedData)_obj);
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new EnvelopedData((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid EnvelopedData");
	}

	/*
	 *
	 *  BUSINESS METHODS
	 *
	 */

	public CMSVersion getVersion() {
		return version;
	}

	private void setVersion(CMSVersion _version) {
		version = _version;
	}


	public OriginatorInfo getOriginatorInfo() {
		return originatorInfo;
	}

	private void setOriginatorInfo(OriginatorInfo _originatorInfo) {
		originatorInfo = _originatorInfo;
	}


	public RecipientInfos getRecipientInfos() {
		return recipientInfos;
	}

	private void setRecipientInfos(RecipientInfos _recipientInfos) {
		recipientInfos = _recipientInfos;
	}


	public EncryptedContentInfo getEncryptedContentInfo() {
		return encryptedContentInfo;
	}

	private void setEncryptedContentInfo(EncryptedContentInfo _encryptedContentInfo) {
		encryptedContentInfo = _encryptedContentInfo;
	}


	public UnprotectedAttributes getUnprotectedAttrs() {
		return unprotectedAttrs;
	}

	private void setUnprotectedAttrs(UnprotectedAttributes _unprotectedAttrs) {
		unprotectedAttrs = _unprotectedAttrs;
	}



	public DERObject getDERObject() {
		BERConstructedSequence _seq = new BERConstructedSequence();
		
		_seq.addObject(version);
		if(originatorInfo != null) {
			_seq.addObject(new BERTaggedObject(false, 0, originatorInfo.getDERObject()));
		}
		_seq.addObject(recipientInfos);
		_seq.addObject(encryptedContentInfo);
		if(unprotectedAttrs != null) {
			_seq.addObject(new BERTaggedObject(false, 1, unprotectedAttrs.getDERObject()));
		}
		
		return _seq;
	}

	/*
	 *
	 *  INTERNAL METHODS
	 *
	 */


}


/**
 * Version History
 * ~~~~~~~~~~~~~~~
 * 1.0
 *     - Created [CK]
 * 
 */
