package org.bouncycastle.asn1.cms;

import org.bouncycastle.asn1.BERConstructedSequence;
import org.bouncycastle.asn1.BERTaggedObject;
import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERObject;

/** 
 * <br>
 * <pre>
 * RFC 2630
 * 
 * EncryptedData ::= SEQUENCE {
 * 	version CMSVersion,
 * 	encryptedContentInfo EncryptedContentInfo,
 * 	unprotectedAttrs [1] IMPLICIT UnprotectedAttributes OPTIONAL 
 * }
 * </pre>
 * 
 * @version 1.0; 
 *     
 */

public class EncryptedData implements DEREncodable {

	/*
	 *
	 *  VARIABLES
	 *
	 */
	
	private CMSVersion            version;
	private EncryptedContentInfo  encryptedContentInfo;
	private UnprotectedAttributes unprotectedAttrs;

	/*
	 *
	 *  CONSTRUCTORS
	 *
	 */

	public EncryptedData(CMSVersion _version, 
	                     EncryptedContentInfo _encryptedContentInfo,
	                     UnprotectedAttributes _unprotectedAttrs) {
		
		setVersion(_version);
		setEncryptedContentInfo(_encryptedContentInfo);
		setUnprotectedAttrs(_unprotectedAttrs);
	}
	
	public EncryptedData(BERConstructedSequence _seq) {
		version              = CMSVersion.getInstance(_seq.getObjectAt(0));
		encryptedContentInfo = EncryptedContentInfo.getInstance(_seq.getObjectAt(1));
		if(_seq.getSize() > 2) {
			unprotectedAttrs = UnprotectedAttributes.getInstance(_seq.getObjectAt(2));
		}
	}

	public EncryptedData(EncryptedData _orig) {
		version              = _orig.version;
		encryptedContentInfo = _orig.encryptedContentInfo;
		unprotectedAttrs     = _orig.unprotectedAttrs;
	}

	public static EncryptedData getInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof EncryptedData) {
			return (EncryptedData)_obj;
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new EncryptedData((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid EncryptedData");
	}

	public static EncryptedData newInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof EncryptedData) {
			return new EncryptedData((EncryptedData)_obj);
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new EncryptedData((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid EncryptedData");
	}

	/*
	 *
	 *  BUSINESS METHODS
	 *
	 */

	public CMSVersion getVersion() {
		return version;
	}

	private void setVersion(CMSVersion _version) {
		version = _version;
	}


	public EncryptedContentInfo getEncryptedContentInfo() {
		return encryptedContentInfo;
	}

	private void setEncryptedContentInfo(EncryptedContentInfo _encryptedContentInfo) {
		encryptedContentInfo = _encryptedContentInfo;
	}


	public UnprotectedAttributes getUnprotectedAttrs() {
		return unprotectedAttrs;
	}

	private void setUnprotectedAttrs(UnprotectedAttributes _unprotectedAttrs) {
		unprotectedAttrs = _unprotectedAttrs;
	}



	public DERObject getDERObject() {
		BERConstructedSequence _seq = new BERConstructedSequence();
		
		_seq.addObject(version);
		_seq.addObject(encryptedContentInfo);
		if(unprotectedAttrs != null) {
			_seq.addObject(new BERTaggedObject(false, 1, unprotectedAttrs.getDERObject()));
		}
		
		return _seq;
	}

	/*
	 *
	 *  INTERNAL METHODS
	 *
	 */


}


/**
 * Version History
 * ~~~~~~~~~~~~~~~
 * 1.0
 *     - Created [CK]
 * 
 */
