package org.bouncycastle.asn1.cms;

import org.bouncycastle.asn1.BERConstructedSequence;
import org.bouncycastle.asn1.BERTaggedObject;
import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERObject;

/** 
 * <br>
 * <pre>
 * RFC 2630
 * 
 * EncryptedContentInfo ::= SEQUENCE {
 * 	contentType ContentType,
 * 	contentEncryptionAlgorithm ContentEncryptionAlgorithmIdentifier,
 * 	encryptedContent [0] IMPLICIT EncryptedContent OPTIONAL 
 * }
 * </pre>
 * 
 * @version 1.0; 
 *     
 */

public class EncryptedContentInfo implements DEREncodable {

	/*
	 *
	 *  VARIABLES
	 *
	 */

	private ContentType                          contentType;
	private ContentEncryptionAlgorithmIdentifier contentEncryptionAlgorithm;
	private EncryptedContent                     encryptedContent;
	
	/*
	 *
	 *  CONSTRUCTORS
	 *
	 */

	public EncryptedContentInfo(ContentType                          _contentType, 
	                            ContentEncryptionAlgorithmIdentifier _contentEncryptionAlgorithm,
	                            EncryptedContent                     _encryptedContent) {
		
		setContentType(_contentType);
		setContentEncryptionAlgorithm(_contentEncryptionAlgorithm);
		setEncryptedContent(_encryptedContent);
	}
	
	public EncryptedContentInfo(BERConstructedSequence _seq) {
		contentType = ContentType.getInstance(_seq.getObjectAt(0));
		contentEncryptionAlgorithm = ContentEncryptionAlgorithmIdentifier.getInstance(_seq.getObjectAt(1));
		if(_seq.getSize() > 2) {
			encryptedContent = EncryptedContent.getInstance(_seq.getObjectAt(2));
		}
	}

	public EncryptedContentInfo(EncryptedContentInfo _orig) {
		contentType                = _orig.contentType;
		contentEncryptionAlgorithm = _orig.contentEncryptionAlgorithm;
		encryptedContent           = _orig.encryptedContent;
	}

	public static EncryptedContentInfo getInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof EncryptedContentInfo) {
			return (EncryptedContentInfo)_obj;
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new EncryptedContentInfo((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid EncryptedContentInfo");
	}

	public static EncryptedContentInfo newInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof EncryptedContentInfo) {
			return new EncryptedContentInfo((EncryptedContentInfo)_obj);
		}
		
		if(_obj instanceof BERConstructedSequence) {
			return new EncryptedContentInfo((BERConstructedSequence)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid EncryptedContentInfo");
	}

	/*
	 *
	 *  BUSINESS METHODS
	 *
	 */

	public ContentType getContentType() {
		return contentType;
	}

	private void setContentType(ContentType _contentType) {
		contentType = _contentType;
	}


	public ContentEncryptionAlgorithmIdentifier getContentEncryptionAlgorithm() {
		return contentEncryptionAlgorithm;
	}

	private void setContentEncryptionAlgorithm(ContentEncryptionAlgorithmIdentifier _contentEncryptionAlgorithm) {
		contentEncryptionAlgorithm = _contentEncryptionAlgorithm;
	}


	public EncryptedContent getEncryptedContent() {
		return encryptedContent;
	}

	private void setEncryptedContent(EncryptedContent _encryptedContent) {
		encryptedContent = _encryptedContent;
	}



	public DERObject getDERObject() {
		BERConstructedSequence _seq = new BERConstructedSequence();
		
		_seq.addObject(contentType);
		_seq.addObject(contentEncryptionAlgorithm);
		if(encryptedContent != null) {
			_seq.addObject(new BERTaggedObject(false, 0, encryptedContent.getDERObject()));
		}
		
		return _seq;
	}

	/*
	 *
	 *  INTERNAL METHODS
	 *
	 */


}


/**
 * Version History
 * ~~~~~~~~~~~~~~~
 * 1.0
 *     - Created [CK]
 * 
 */
