package org.bouncycastle.asn1.cms;

import org.bouncycastle.asn1.BERConstructedSequence;
import org.bouncycastle.asn1.BERTaggedObject;
import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.x509.X509CertificateStructure;

/** 
 * <br>
 * <pre>
 * RFC 2630
 * 
 * CertificateChoices ::= CHOICE {
 * 	certificate Certificate,  -- See X.509
 * 	extendedCertificate [0] IMPLICIT ExtendedCertificate,  -- Obsolete
 * 	attrCert [1] IMPLICIT AttributeCertificate -- See X.509 & X9.57
 * }
 * </pre>  
 * 
 * @version 1.0; 
 *     
 */

public class CertificateChoices implements DEREncodable {

	/*
	 *
	 *  VARIABLES
	 *
	 */

	private DEREncodable cert;

	/*
	 *
	 *  CONSTRUCTORS
	 *
	 */

	public CertificateChoices(X509CertificateStructure _cert) {
		setCertificate(_cert);
	}
	
	public CertificateChoices(ExtendedCertificate _cert) {
		setCertificate(_cert);
	}
	
	public CertificateChoices(DEREncodable _cert) {
		if(_cert instanceof BERTaggedObject) {
			BERTaggedObject _dto = (BERTaggedObject)_cert;
			switch(_dto.getTagNo()) {
				case 0 :
					cert = ExtendedCertificate.getInstance(_dto);
					break;
				case 1 :
					throw new IllegalArgumentException("AttributeCertificate not supported");
				default:
					throw new IllegalArgumentException("Invalid CertificateChoices");
			}
		}
		else if(_cert instanceof X509CertificateStructure) {
			cert = _cert;
		}
		else if(_cert instanceof BERConstructedSequence) {
			cert = new X509CertificateStructure((BERConstructedSequence)_cert);
		}
		else {
			throw new IllegalArgumentException("Invalid CertificateChoices");
		}
	}

	public CertificateChoices(CertificateChoices _orig) {
		cert = _orig.cert;
	}
	
	public static CertificateChoices getInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof CertificateChoices) {
			return (CertificateChoices)_obj;
		}
		
		if(_obj instanceof DEREncodable) {
			return new CertificateChoices((DEREncodable)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid CertificateChoices");
	} 

	public static CertificateChoices newInstance(Object _obj) {
		if(_obj == null) {
			return null;
		}
		
		if(_obj instanceof CertificateChoices) {
			return new CertificateChoices((CertificateChoices)_obj);
		}
		
		if(_obj instanceof DEREncodable) {
			return new CertificateChoices((DEREncodable)_obj);
		}
		
		if(_obj instanceof BERTaggedObject) {
			return getInstance(((BERTaggedObject)_obj).getObject());
		}
		
		throw new IllegalArgumentException("Invalid CertificateChoices");
	} 

	/*
	 *
	 *  BUSINESS METHODS
	 *
	 */

	public DEREncodable getCertificate() {
		return cert;
	}

	private void setCertificate(X509CertificateStructure _cert) {
		cert = _cert;
	}

	private void setCertificate(ExtendedCertificate _cert) {
		cert = _cert;
	}

	public DERObject getDERObject() {
		if(cert instanceof ExtendedCertificate) {
			return new BERTaggedObject(false, 0, cert.getDERObject());
		}
		
		return cert.getDERObject();
	}

	/*
	 *
	 *  INTERNAL METHODS
	 *
	 */


}


/**
 * Version History
 * ~~~~~~~~~~~~~~~
 * 1.0
 *     - Created [CK]
 * 
 */
