package org.bouncycastle.asn1;

import java.io.*;

public class DERObjectIdentifier
    extends DERObject
{
    String      identifier;

    public static DERObjectIdentifier getInstance(
        Object  obj)
    {
        if (obj == null || obj instanceof DERObjectIdentifier)
        {
            return (DERObjectIdentifier)obj;
        }

        if (obj instanceof ASN1OctetString)
        {
            return new DERObjectIdentifier(((ASN1OctetString)obj).getOctets());
        }

        if (obj instanceof ASN1TaggedObject)
        {
            return getInstance(((ASN1TaggedObject)obj).getObject());
        }

        throw new IllegalArgumentException("illegal object in getInstance");
    }

    DERObjectIdentifier(
        byte[]  bytes)
    {
        int             head = bytes[0] & 0xff;
        StringBuffer    objId = new StringBuffer();

        objId.append(Integer.toString(head / 40));
        objId.append('.');
        objId.append(Integer.toString(head % 40));
        
        int value = 0;

        for (int i = 1; i != bytes.length; i++)
        {
            int b = bytes[i] & 0xff;

            value = value * 128 + (b & 0x7f);
            if ((b & 128) == 0)             // end of number reached
            {
                objId.append('.');
                objId.append(Integer.toString(value));
                value = 0;
            }
        }

        this.identifier = objId.toString();
    }

    public DERObjectIdentifier(
        String  identifier)
    {
        this.identifier = identifier;
    }

    public String getId()
    {
        return identifier;
    }

    void encode(
        DEROutputStream out)
        throws IOException
    {
        OIDTokenizer            tok = new OIDTokenizer(identifier);
        ByteArrayOutputStream   bOut = new ByteArrayOutputStream();
        DEROutputStream         dOut = new DEROutputStream(bOut);

                                    // space for 5 7 bit numbers in an int
        byte[]                  iBuf = new byte[5];    
        
        dOut.write(
                    Integer.parseInt(tok.nextToken()) * 40
                    + Integer.parseInt(tok.nextToken()));

        while (tok.hasMoreTokens())
        {
            //
            // translate into base 128
            //
            int value = Integer.parseInt(tok.nextToken());
            int count = iBuf.length - 1;
            
            iBuf[count--] = (byte)(value % 128);
            value /= 128;

            while (value != 0)
            {
                iBuf[count--] = (byte)((value % 128) | 0x80);
                value /= 128;
            }
            dOut.write(iBuf, count + 1, iBuf.length - (count + 1));
        }

        dOut.close();

        byte[]  bytes = bOut.toByteArray();

        out.writeEncoded(OBJECT_IDENTIFIER, bytes);
    }

    public int hashCode()
    {
        return identifier.hashCode();
    }

    public boolean equals(
        Object  o)
    {
        if ((o == null) || !(o instanceof DERObjectIdentifier))
        {
            return false;
        }

        return identifier.equals(((DERObjectIdentifier)o).identifier);
    }
}
