package Proxy;

/***
 * Purpose: HHProxy is instantiated by Proxy for every message received
 *          from HH.  HHProxy has methods for handling file block requests
 *          from HH by retrieving file information from FM and using the
 *          downloadAndSend() method to retrieve file blocks from FSs.
 *          This class has methods for caching files in the cacheMap,
 *          which is a TreeMap of FNode objects. Messages from HH include
 *          requests for file blocks  and requests to flush cache.
 ***/

import javax.servlet.*;		//for servlet output/input streams
import javax.servlet.http.*;

import java.io.*;
import java.net.*;
import java.security.*;   //for MessageDigest
import javax.net.ssl.*;

import java.util.*;				//for Map interface

	//Every instance is a proxy for one HH
	public class HHProxy implements FileManagerConstants{

		ServletInputStream sis;		//input stream from HH
		ServletOutputStream sos;	//output stream to HH
	
		HttpURLConnection connection; 	

		FNode fNode;							//for cacheMap

		FMInfo fmi;

		
		OutputStream        outFMStream; //output stream to FM

		//Info related to one request/response for the Proxy servlet
	
		byte protocol;				//3, if from HH
		byte msgType;					//2, for file request from HH
		
		byte[] ownerIdArr;
		int ownerIdLen;
		byte fileIdArr;
		int fileIdLen;
		long timeStamp;
		long randNumber;
		byte[] pSetDigest;
		int digLen;

		int startIndex;					//starting index of requested blocks
		int numReqBlocks;				//number of blocks requested
		byte[][] sigsArr;				//Array to hold blocks sigs sent by HH in request

		byte[] reqFileId;				//initialized only for file request from HH
		
		DataOutputStream hDOS;
    Map cacheMap; //TreeMap of FNode objects. Common to all instances of HHProxy.
		              //caches maximum 5 files (FNode objects).
		
		public HHProxy(ServletInputStream sis, ServletOutputStream sos, Map cacheMap) throws Exception{
			this.sis = sis;
			this.sos = sos;
			this.cacheMap = cacheMap;

			DataInputStream hhdis = new DataInputStream(sis);
			hDOS= new DataOutputStream(sos);//o/p stream to HH
			
			//Required for initializing FM info
			fmi = new FMInfo();
			FMInfo.init();

			//Read ownerId, passwd and fileReq from sis 
			//Put ownerId in OwnerInfo and pw in OwnerProtocol we will build
			//Pass requested file name to fmReceiveFileInfo()

			ownerIdLen = hhdis.readByte();
			ownerIdArr = new byte[ownerIdLen];
			hhdis.readFully(ownerIdArr);

			int hhOwnerProtLen = hhdis.readInt();
			byte[] hhOProtArr = new byte[hhOwnerProtLen];
			hhdis.readFully(hhOProtArr);

				//Decode the OwnerProtocol
				System.out.println("Decoding byte array data for OwnerProtocol: ");

      	ByteArrayInputStream bais = new ByteArrayInputStream(hhOProtArr);
      	DataInputStream bdis = new DataInputStream(bais);

				protocol = bdis.readByte();	
				msgType = bdis.readByte();
				System.out.println(	"Protocol(3): " + protocol + " msgType(2): " 
														+ msgType);

				if(protocol != 3){
					System.out.println("Client is not HH. Disconnecting...");
					return;
				}

				//Read password stuff
				timeStamp = bdis.readLong();
				randNumber = bdis.readLong();
				digLen = bdis.readInt();
				pSetDigest = new byte[digLen];
				bdis.readFully(pSetDigest);

			switch(msgType){

				case 2:	//File Request from HH
									System.out.println("Received file request from HH");
									int fIdLen = bdis.readInt();
									reqFileId = new byte[fIdLen];
									bdis.readFully(reqFileId);

                  /***
									 * If file with reqFileId String is cached, send back requested
									 * blocks from cache.
									 * Otherwise, retrieve file information from FM, and use
									 * downloadAndSend() to download the file from FSs and cache
									 * for future requests.
									 ***/

									String fId = new String(reqFileId, DEF_ENCODING);

									startIndex = bdis.readInt();
									numReqBlocks = bdis.readInt();//Number of requested blocks

									sigsArr = new byte[numReqBlocks][20];
		
									System.out.println("Going to read block sigs from HH Array");
									for(int sIndex=0;sIndex<numReqBlocks;sIndex++)
										bdis.readFully(sigsArr[sIndex]);
									
									//We pass the sigs to compare with sigs stored in FNode
									if(sendFromCache(fId,startIndex,numReqBlocks,sigsArr)==true){
										System.out.println("Sent requested blocks from cache");
										return;
									}

									System.out.println("File not cached...Downloading and "
																			+ "sending");

									fmConnect();
									//FM will authenticate the HH
									fmRequestFileInfo(reqFileId);
									fmReceiveFileInfo(reqFileId);	//Calls downloadAndSend()
									bdis.close();				//for hhOProtArr
									hhdis.close();			//input stream from HH
									System.out.println("Finished receiving requested file info");

									break;

				case 5:		//Clear cache msg from HH
									
									System.out.println("Going to read fIdListArrLen.");
									int fIdListArrLen = bdis.readInt();
									System.out.println("fIdListArrLen: " + fIdListArrLen);
									byte[] fIdListArr = new byte[fIdListArrLen];
									bdis.readFully(fIdListArr);

									System.out.println("Extracted fIdListArr. Calling flushCache()");
									flushCache(fIdListArr, fIdListArrLen);
									System.out.println("Closing input stream from HH.");
									bdis.close();
									hhdis.close();

									break;
									
				default: 
									System.out.println("Unknown msgType " + msgType +
																		" from HH");

			}

		}	//end of constructor


		  /***
			 * Precondition:  flush-cache message received from HH. fIdListArr contains
			 *                byte encoding of list of file ids of HH.
			 * Postcondition: Removes FNode objects that with fileIds in fIdListArr from
			 *                cacheMap.
			 ***/

		public void flushCache(byte[] fIdListArr, int len){

			byte[] listArr = new byte[len];

			System.arraycopy(fIdListArr, 0, listArr, 0, len);


			ByteArrayInputStream bais = new ByteArrayInputStream(listArr);
			DataInputStream badis = new DataInputStream(bais);


			while(true){
				try{
					int fIdLen = badis.readInt();

					byte[] fIdArr = new byte[fIdLen];
					badis.readFully(fIdArr);
					String fId = new String(fIdArr, DEF_ENCODING);
					
					if(cacheMap.containsKey(fId)){

						System.out.println("Removing file: " + fId + " from cache.");
						cacheMap.remove(fId);


					}
					

				}catch(Exception e){
					System.out.println("flushCache-"+e.getMessage());
					break;

				}


			}//end of while
						
		}

      /***
			 * Precondition:   FNode is not already in cache.
			 * Postcondition:  This function adds the FNode (containing the file
			 *                 info and file blocks) to the cache. If required,
			 *                 it replaces the least recently used (LRU) cached
			 *                 FNode.
			 ***/

		public void addToCache(FNode fNode){

			fNode.updateTimeStamp();

			if(cacheMap.size()>= MAX_CACHE_SIZE){

				Set mSet = cacheMap.entrySet();
				Iterator it = mSet.iterator();

				long currTimeStamp = System.currentTimeMillis();
				String keyId=null;
				
				while(it.hasNext()){

					Map.Entry me = (Map.Entry)it.next();
					FNode tfNode = (FNode)me.getValue();
					if(tfNode.timeStamp<currTimeStamp){

						currTimeStamp = tfNode.timeStamp;
						keyId = new String((String)me.getKey());

					}
				
				}//while

				System.out.println("Removing least recently used file from cache");
				cacheMap.remove(keyId);

			}//MAX_CACHE_SIZE

			System.out.println("Going to add fNode to cacheMap");

			cacheMap.put(fNode.fId, fNode);

			System.out.println("Finished adding fNode to cacheMap");


		}//end of addToCache()

    /***
		 * Precondition:  request for file blocks received from HH.
		 * Postcondition: if file containing blocks requested by HH is in cache,
		 *                this method sends cached blocks to HH and returns true.
		 *                Returns false, if the file is not in cache.
		 ***/

		public boolean sendFromCache(String fId, int startIndex, int numReqBlocks, byte[][] sigsArr){
		
		FNode sfNode = (FNode)cacheMap.get(fId);

		long currTime = System.currentTimeMillis();
		
		
		if(sfNode == null){
			System.out.println("Could not sendFromCache()");				
			return false;

		}


		long lastAccessed = sfNode.timeStamp;
		System.out.println("lastAccessed time for fileId "+ sfNode.fId);
		System.out.println("is: " + sfNode.timeStamp);

		long idleTime = currTime - sfNode.timeStamp;

		//return false, if file is cached but has not been accessed for 10 minutes.
		//the stale file will automatically get replaced when the cache fills up.
		
		if(idleTime > 600000){

			System.out.println("File is available in cache but has not been accessed");
			System.out.println("for more than 10 minutes.");
			sfNode.timeStamp = 0;
			return false;
		}

		System.out.println("Requested file is in cache.");

		if(startIndex<0 || startIndex>=sfNode.noBlocks){
			System.out.println("Wrong startIndex in request: " + startIndex);
			return false;
		}

		if((startIndex+numReqBlocks) < 0){
			System.out.println("Negative (startIndex+numReqBlocks)!");
			return false;
		}

		if((startIndex+numReqBlocks)>=sfNode.noBlocks){
			System.out.println("Wrong numReqBlocks: " + numReqBlocks);
			numReqBlocks = sfNode.noBlocks-startIndex;
			System.out.println("Corrected to : " + numReqBlocks);
		}

		try{

			for(int bIndex=startIndex;bIndex<(startIndex+numReqBlocks); bIndex++){
				System.out.println("Sending block "+bIndex+" from cache");
				hDOS.write(sfNode.block[bIndex],0,sfNode.size[bIndex]);

			}

			hDOS.close();

		}catch(Exception e){

			System.out.println("Exception in sendFromCache(): " 
													+ e.getMessage());
		}

		sfNode.updateTimeStamp();	//Does this update in the referenced FNode-yes
		System.out.println("Finished sending requested blocks from cache");
		return true;	//all well

		}
		
		//Method that establishes https connection to FM.
	  public void fmConnect() throws Exception{

			System.out.println("Connecting to: " + fmi.fmURL);
			try{
				connection = (HttpURLConnection)((FMInfo.fmURL).openConnection());
			}catch(Exception e){
				System.out.println("Exception in fmConnect(): " + e.getMessage());
			}
			System.out.println("Opened https connection to: " + fmi.fmURL);
			connection.setDoOutput(true);
			connection.setRequestMethod("POST");
			connection.connect();
	
		}
	/***
	 *	Precondition: 	file with fileId rFileId not in Proxy cache.
	 *	Postcondition: 	sends request for file server lisiting and block sigs 
	 *									for file requested by HH to FM.
	 ***/

	public void fmRequestFileInfo(byte[] rFileId){

		OutputStream outFMStream;

		try{

			outFMStream = connection.getOutputStream();
			System.out.println("Going to request info for file" );
			DataOutputStream ndos = new DataOutputStream(outFMStream);
			ndos.write((byte)ownerIdLen);
			ndos.write(ownerIdArr); 										//the ownerId

			//OwnerProtocolP contains the passwd - need to change for every instance
			System.out.println("Generating OwnerProtocol with msgType: 6");
			OwnerProtocolP oProt = new OwnerProtocolP((byte)6, rFileId, timeStamp, randNumber, pSetDigest);
			byte[] oProtArr = oProt.byteEncode();
			int oLen = oProtArr.length;
			ndos.writeInt(oLen);
			try{
				ndos.flush();
			} catch(Exception ed){
				System.out.println(ed.getMessage());
			}
			ndos.write(oProtArr);
			ndos.flush();
			ndos.close();

			System.out.println("Finished sending request for File Info");
		}catch (Exception e){
			System.out.println("5.could not open output stream to FM");
		}

	}


	/***
	 * Precondition:	HHProxy has sent fmRequestFileInfo message to FM.
	 * Postcondition: 	Receives file info (list of FSs storing file and block sigs)
	 * 								from FM and calls downloadAndSend().
	 ***/
	
	public void fmReceiveFileInfo(byte[] fId) throws Exception{

		BriefFNode bFNode = new BriefFNode(fId);

		InputStream inFMStream;

		try{

			inFMStream = connection.getInputStream();
			System.out.println("Going to receive info for file");

			
			bFNode.byteDecode(inFMStream);	

			//Create a FNode that will cache the above mentioned info and
			//also the actual downloaded file blocks

			fNode = new FNode(bFNode);
			//Note: there are no file blocks in there yet
			
			inFMStream.close();

		}catch (Exception e){
			System.out.println("6.could not open input stream from FM");
			return;
		}

	//hhOutputStream is the servlet output stream back to the requesting HH
	System.out.println("Received FileInfo from FM. Calling downloadAndSend()");

	int attempt = 0;
	
	for(attempt=0; attempt<5; attempt++){

		System.out.println("Calling downloadAndSend()...");
		if(downloadAndSend(bFNode)){
			System.out.println("downloadAndSend succeeded in attempt: "+(attempt+1));
			break;
		}
		else{
			System.out.println("downloadAndSend failed in attempt: "+(attempt+1));
		}

	}

	if(attempt == 5){
		System.out.println("downloadAndSend() failed 5 times. Giving up...");
		return;
	}


	}//end of fmReceiveFileInfo()

  /***
	 * 	Precondition: 	bFNode has been initialized with information required 
	 * 									to downloadAndSend() the file.
	 *  Postcondition:	Get the list of FSs that have this file from bfNode.
	 *  								Send block reqs to different FSs and receive and check 
	 *  								block sigs. Transmit blocks in proper order back to the
	 *  								requesting HH
	 ***/

	public boolean downloadAndSend(BriefFNode bfNode) throws Exception{


		//Check for the condition where the total number of blocks is 
		//less than 5
		int maxSeriesNum;
		if(bfNode.noBlocks<5) 
			maxSeriesNum = bfNode.noBlocks;
		else
			maxSeriesNum = 5;

		//Five RetrieveFS objects are instantiated in five threads.
		System.out.println(	"maxSeriesNum: " + maxSeriesNum 
													+ " Num blocks: "+bfNode.noBlocks);

	 ThreadGroup retrieveFSGroup = new ThreadGroup("RetrievefromFS");

	switch(maxSeriesNum){

		case 5: 
	 		RetrieveFS firstFS = 	new RetrieveFS("FirstFS", retrieveFSGroup, 
										 				bfNode, 5/*Series Num*/);
		case 4:
	 		RetrieveFS secondFS = new RetrieveFS("SecondFS", retrieveFSGroup,
										 				bfNode, 4);
		case 3:
	 		RetrieveFS thirdFS = 	new RetrieveFS("ThirdFS", retrieveFSGroup,
										 				bfNode, 3);
		case 2:
	 		RetrieveFS fourthFS = new RetrieveFS("FourthFS", retrieveFSGroup,
											 			bfNode, 2);
		case 1:
	 		RetrieveFS fifthFS = 	new RetrieveFS("FifthFS", retrieveFSGroup,
									 					bfNode, 1);
	 		break;

		default:
	 		System.out.println("Wrong maxSeriesNum: Shouldn't be getting this");
			return false;
		
	}//end of switch

	/***
	 * Iterate through the 5 threads, blocking if necessary, to retrieve 
	 * the file blocks in proper order
	 ***/

	 Thread groupFSThread[] = new Thread[retrieveFSGroup.activeCount()];
	 retrieveFSGroup.enumerate(groupFSThread);

		boolean available = true;

		/***
		 *	groupFSThread[0] corresponds to series 4
		 *	groupFSThread[4] corresponds to series 0
		 ***/ 

		MessageDigest md = MessageDigest.getInstance("SHA-1");

		int count=0;			//block count
		
	 	while(/*more blocks available*/ available){
	 		for (int seriesNum=maxSeriesNum-1; seriesNum>=0; seriesNum--){
				//May block if necessary
				//Read from each of the 5 buffers in order.
				byte[] blockBytes = ((RetrieveFS)groupFSThread[seriesNum]).getBlock();			

				if(blockBytes == null){
					System.out.println("Finished retrieving all file blocks");
					available=false;
					break;			//Careful - will 'break' out of for only
				}

					boolean error = false;
					try{
						boolean badSig = false;
						md.update(blockBytes);

						byte[] newSig = md.digest();	//this also resets the digest

						//Check signature of retrieved block
						if(compareSigs(newSig, bfNode.blockSig[count])!=0)
							badSig = true;

						if(badSig){
							int badFSIndex = ((RetrieveFS)(groupFSThread[seriesNum])).FSIndex;
							bfNode.FS[badFSIndex] = null;
							throw new SigException();
						}

						System.arraycopy(blockBytes, 0, fNode.block[count], 0, blockBytes.length);
						fNode.size[count]=blockBytes.length;
						if((count >= startIndex) && (count < (startIndex+numReqBlocks))){	
							//send block to HH
							System.out.println("Sending block of series " 
														+ seriesNum + " to HH" + " Count: " + count);
							hDOS.write(blockBytes);
						}
					}
					catch(SigException se){
						System.out.println("downloadAndSend()-Caught SigException...");
						//Close connection, HH will try again
						hDOS.close();
						return false;
					}

					catch(Exception eh){
						System.out.println("Unable to write to HH-"+eh.getMessage());
						System.out.println("Still going to cache fNode.");
						error = true;
					}
				count++;			//One more block

	 		}//end of for

		}//end of while

		//Cache the blocks of this file; if needed remove least recently 
		//used FNode

		addToCache(fNode); 

		System.out.println("Finished retrieving all blocks for HH");

		hDOS.close();
		
		return true;

	}//end of downloadAndSend()



	/***
	 * Purpose: Helper method, verifies sigs of retrieved blocks by checking 
	 * 					160 bits of SHA-1 hash.
	 * Postcondition: Returns 0 if sig2 matches sig1, -1 otherwise
	 ***/

	public int compareSigs(byte[] sig1, byte[] sig2){

		try{
			for(int sigByte=0; sigByte<20; sigByte++)
					if(sig2[sigByte] != sig1[sigByte]) return -1;

		}catch(Exception e){
			System.out.println("compareSigs Exception-"+e.getMessage());
			return -1;
		}

		return 0;
	}

	//Helper function for debugging.
	public void displayBlock(byte[] b){
				
		for(int i=0;i<b.length;i++)
			System.out.println(b[i]);

	}
	
	}//end of HHProxy class

	/***
	 * Purpose: FMInfo contains data and methods to connect to FM using
	 * 					https.
	 ***/

  class FMInfo{

		public static URL fmURL;
		public static void init(){
			try{
					//Required for secure https connection
					System.setProperty("java.protocol.handler.pkgs","com.sun.net.ssl.internal.www.protocol");
					Security.addProvider(new com.sun.net.ssl.internal.ssl.Provider());

					fmURL = new URL("https", "ep.netlab.uky.edu", 8443, "/servlet/FM.FM");
			}catch(MalformedURLException m){System.out.println("Malformed URL");}
		}
	}//end of FMInfo


