package FS;

/***
 * Purpose: FileManagerProtocol contains data members and methods
 *          to code/decode messages between FM and FS, between
 *          DT and FS, and between Proxy and FS.
 ***/

import java.net.*;
import java.io.*;
import java.security.*;

class FileManagerProtocol implements FileManagerConstants{

    public byte protocol;      
    public byte msgType;
    public int dataLength;
    public byte[] data;        
		
    //Constructor
    public FileManagerProtocol(byte msgType, byte[] data) throws Exception{

        this.protocol = 1;
        this.msgType = msgType;

				//kill this test
        if(false && (data.length > MAX_DATA_SIZE)){
            this.dataLength = MAX_DATA_SIZE;
            throw new IOException("Data more than MAX_DATA_SIZE: ");
        }
        else this.dataLength = data.length;

        //Allocate space for the data buffer
        this.data = new byte[data.length];
        System.arraycopy(data,0,this.data,0,this.dataLength);
    }

    /***
		* Precondition: data members are initialized.
		* Postcondition: returns the data members, encoded as a byte array.
		***/
    public byte[] byteEncode() throws Exception{

        ByteArrayOutputStream buf = new ByteArrayOutputStream();
        DataOutputStream out = new DataOutputStream(buf);
        out.writeByte(protocol);
        out.writeByte(msgType);
        out.writeInt(dataLength);
        out.write(data);

        out.flush();

        return buf.toByteArray();
    }

    //Constructor
    public FileManagerProtocol() throws Exception{
    	//Does nothing. The data members will be initialized by byteDecode
    }

    /***
		 *Precondition:   The data members of the class are not initialized.
		 *Postcondition:  The data members are initialized using data from 
		 *								the input stream is.
		 * 								If the protocol is 1 (FM's message), this method performs 
		 * 								does an	authenticity and integity check on the message 
		 * 								and checks the timeStamp.
		 * 								This method returns false if message does not conform to 
		 * 								FileManagerProtocol or if the message is from FM and fails
		 * 								authentication or verification.
		 ***/
    public boolean byteDecode(InputStream is)throws Exception{


     	DataInputStream dis = new DataInputStream(is);

			try{
   	  	protocol = (byte)dis.readByte();
	    	msgType = (byte)dis.readByte();
  	  	dataLength = dis.readInt();

			}catch(Exception e){
				is.close();
				System.out.println("End of communication.");
				return false;
			}
    	//Allocate space for data buffer
     	data = new byte[dataLength];

      try{
				dis.readFully(data); //Returns error on premature EOStream
			}catch(Exception e){
				System.out.println("Could not read expected data bytes in " +
												"FileManagerProtocol.");
				return false;
			}

			if(protocol == 1 && msgType != SETUP_CONF){

				System.out.println("Msg from FM. Going to check authenticity and integity.");

				//Read the timeStamp in msg
				long timeStamp = dis.readLong();
				int sigLen = dis.readInt();
				byte[] fmSig = new byte[sigLen];
				dis.readFully(fmSig);
				
				//avoid replay
				boolean newerTS = false;

				if(timeStamp<=DiskInfo.prevFMTimeStamp){

					System.out.println("FM Msg contains older timeStamp. Rejecting...");
					return false;

				}
				//Store this timeStamp. Reject future messages from FM that have
				//this or lower timeStamp 
				DiskInfo.prevFMTimeStamp = timeStamp;
				
				ByteArrayOutputStream buf = new ByteArrayOutputStream();
				DataOutputStream out = new DataOutputStream(buf);

				out.writeByte(protocol);
				out.writeByte(msgType);
				out.writeInt(dataLength);
				out.write(data);
				out.flush();
				byte[] dataArr = buf.toByteArray();
				byte[] timeArr = getBytes(timeStamp); 

				Signature dsaSig = Signature.getInstance("SHA1withDSA", "SUN");
				dsaSig.initVerify(DiskInfo.pubKey);
				dsaSig.update(dataArr);
				dsaSig.update(timeArr);

				boolean verified = dsaSig.verify(fmSig);

				if(verified){

					System.out.println("Successfully verified FM's message.:");
					return true;

				}else{
					System.out.println("Message from FM does not match signature sent by FM.");	
					return false;
				}
			}//end of if(protocol == 1)
			return true;	//all well
    }

  /***
	 * The following helper method is from the book Wireless Java:
	 * Developing with Java[tm] 2 Micro Edition, by Jonathan Knudsen
	 *
	 * Postcondition: Returns a long data type as a byte array.
	 ***/
	private byte[] getBytes(long x){
	
		byte[] arr = new byte[8];
		for(int i = 0; i<8; i++){
			arr[i] = (byte)(x >> ((7-i) * 8));
		}

		return arr;
	}

	//Helper method, for debugging.
	public String toString(){

		String value = L_SEP + "Protocol: "+ protocol +
                       " msgType: " + msgType + 
                       " dataLength: " + dataLength + L_SEP;
		return value;
	}

} // End of Class

