package FS;

import java.io.*;
import java.net.*;
import java.util.*;
import java.security.*;

/***
 * 	Purpose: 	FSServer is the server run by FS on the machine
 * 						exporting disk space. FSServer expects 3 types of 
 * 						clients.
 * 	FM: 			- may submit a new file for storage
 * 						-	may submit dirty blocks of an old file
 * 						- may delete a file
 * 	DT: 			- may request blocks of a file
 * 	Proxy: 		- may request blocks of a file
 *
 * 	FSServer starts a new thread for each new connection
 *
 * 	(Note: I originally coded FSServer to be a HTTP server, and so 
 * 	some variables have the term http in their names.)
 ***/

public class FSServer implements Runnable {

	ServerSocket httpSocket;

	/***
	 * Precondition: httpSocket has already been instantiated.
	 ***/
	public FSServer(ServerSocket httpSocket){
		this.httpSocket = httpSocket;
	}
	
	public void run(){
		//Run forever accepting connections and spawning a thread to 
		//service each connection
		for(;;){
			try{
			System.out.println("Waiting for http client to connect");
			Socket httpClient = httpSocket.accept(); // blocks, waiting for a connection
			HttpProtocolHandler httpHandler = new HttpProtocolHandler(httpClient);
			Thread thread = new Thread(httpHandler);
			thread.start(); 	//spawns another thread and returns immediately
			} catch(IOException e){}
			
		}//end of for
	
	}//end of run

}//end of class

/***
 * Purpose: HttpProtocolHandler is instantiated by FSServer to service a 
 * 					connection request from FM, DT or Proxy and to act on the 
 * 					messages received from FM, DT, or Proxy.
 ***/
class HttpProtocolHandler implements Runnable , FileManagerConstants{

	Socket httpClient;
	InputStream is;
	OutputStream os;

	public HttpProtocolHandler(Socket httpClient){
		this.httpClient = httpClient;
	}
	
	public void run() {
		try{
			handleClient();
		} catch(Exception E){}
	}//end of run


		/***
		* Precondition: 	Client has already connected on httpClient socket
		* Postcondition:	Handles message sent by client (FM, DT or Proxy)
		***/
    void handleClient() throws Exception{
		
		is = new BufferedInputStream(httpClient.getInputStream());
		DataInputStream dis = new DataInputStream(is);

		os = httpClient.getOutputStream();

		FileManagerProtocol fmpServer = new FileManagerProtocol(); 	

		//decode FileManagerProtocol, FM's signature is verified in 
		//FileManagerProtocol if message is from FM.
		boolean goodMsg = fmpServer.byteDecode(is);

		if(!goodMsg){
			System.out.println("Bad message...Quitting.");
			return;
		}

		//Precondition: If msg is from FM, its authenticity and integrity are verified
		//							by the fmpServer object.
		
		switch(fmpServer.protocol) {
			
			/***
			 * Proxy and DT are not authenticated, since blocks are encrypted in the
			 * first place. The FM is authenticated with its public key being used to 
			 * verify the message signature
			 ***/
			case 1: //FM-FS 
			case 5: //DT-FS
			case 6:	//Proxy-FS

							//Now decide action based on msgType in fmp

							if(fmpServer.msgType == NEW_FILE && fmpServer.protocol==1){

							//extract the fileId from the data part of fmpServer
							//create a simple fNode, with the indexNum and blockSig
							//Store the blocks to disk
							FNode fn = new FNode();	

							BriefFileDesc bfd = new BriefFileDesc();
							bfd.byteDecode(fmpServer.data);
						
							fn.fileId = bfd.fileId;
							fn.fIdLength = bfd.fIdLength;
							fn.fileSize = bfd.maxEncFileSize;

							//need to check if sufficient disk space 
							//is available, based on fileSize
							DiskInfo.reduceSpaceLeft(fn.fileSize);

							fn.blockUnit = bfd.blockUnit;
							
							
							//Open a file output stream to write to disk
							//while saving the blockSig to FileNode

							FileOutputStream	fos; 

							try{
								//Generate a unique filename to use locally on FS
								Random rand = new Random(System.currentTimeMillis());

								File nfile;
								do{
									String fileName = "cf_" + Math.abs(rand.nextInt());
									nfile = new File(fileName);
									System.out.println("filename already taken: " + nfile);
								}while(nfile.exists());
								
								//Store the filename in the fNode
								fn.localFName = nfile.toString();
								
								fos = new FileOutputStream(nfile);

							}catch(Exception ec){
								System.out.println("Could not create an FileOutputStream for fileId: " 
																		+ fn.fileId);
								return;
							}
				
							int blockIndex;
							byte[] tempArr = new byte[(int)fn.blockUnit];
							byte[] tempSig = new byte[20];
							MessageDigest md = MessageDigest.getInstance("SHA-1");

							boolean readBlock = true; //try to read a block
							int numBytes = 0;
							int totBytes = 0;
							
							while(readBlock){
								numBytes = 0;
								totBytes = 0;

								try{
							  	blockIndex = dis.readInt();
										if(blockIndex >= 500){
											System.out.println("blockIndex: " + blockIndex 
																			+" exceeds maximum number of blocks(500)");
											return;
										}
										fn.noBlocks += 1; 				//since new file

										numBytes = dis.read(tempArr);	
										totBytes = numBytes;

										while(totBytes < (int)fn.blockUnit){
												if(numBytes == -1){
													System.out.println("No more bytes");
													break;
												}
												
										numBytes = dis.read(tempArr,totBytes,((int)fn.blockUnit - totBytes));	
										if(numBytes != -1) totBytes += numBytes;

										}

										//Calculate blockSig
										md.update(tempArr, 0,totBytes); 
										tempSig = md.digest();
								
										System.arraycopy(tempSig, 0, fn.blockSig[blockIndex], 0, tempSig.length); 

										//No need to 'seek' to the block index in the file 
										//because this is a new file
										//Want to write only totBytes bytes
										fos.write(tempArr,0,totBytes);			//write the block to file
										System.out.println("Wrote a block to file");
								
								}catch(EOFException eofe){
										readBlock = false;
								}catch(IOException ioe){
										readBlock = false;
								}

							}//end of while(readBlock)
						
							//Now that the fNode contains the required blockSigs, store it in 
							//the FileDirectory
							fos.close();
							System.out.println("Adding fNode to fileDirectory");

							FileDirectory.addNode(fn);

							}

							/***OLD_FILE***/

							//Precondition: Only changed blocks have been submitted
							else if (fmpServer.msgType == OLD_FILE  && fmpServer.protocol==1){


							//extract the fileId from the data of fmp
							//create a simple fNode, with the indexNum and blockSig
							//Store the blocks to disk
	
							System.out.println("Received OLD_FILE");
											
							FNode fn = new FNode();	

							BriefFileDesc bfd = new BriefFileDesc();
							bfd.byteDecode(fmpServer.data);
						
							fn.fileId = bfd.fileId;

							if(!FileDirectory.contains(fn.fileId)){

								System.out.println("***WRONG msgType: Not OLD_FILE***");
								return;

							}
							
							//Get old FNode
							FNode ofNode = (FNode)FileDirectory.getNode(fn.fileId);
							fn.noBlocks = ofNode.noBlocks;
							fn.localFName = ofNode.localFName;	
							for(int bIndex=0;bIndex<500;bIndex++)
								System.arraycopy(ofNode.blockSig[bIndex],0,fn.blockSig[bIndex],0,ofNode.sigSize);
							
							fn.fIdLength = bfd.fIdLength;
							fn.fileSize = bfd.maxEncFileSize;

							//At this point need to check if sufficient disk space 
							//is available, based on fileSize

							DiskInfo.increaseSpaceLeft(ofNode.fileSize);
							DiskInfo.reduceSpaceLeft(fn.fileSize);
							
							fn.blockUnit = bfd.blockUnit;
							
							
							//Open a file output stream to write to disk
							//while saving the blockSig to FileNode
							//need to generate a path to the directory

							RandomAccessFile oraf = new RandomAccessFile(ofNode.localFName,"rw");

							int blockIndex;
							byte[] tempArr = new byte[(int)fn.blockUnit];
							byte[] tempSig = new byte[20];
							MessageDigest md = MessageDigest.getInstance("SHA-1");

							boolean readBlock = true; //try to read a block
							int numBytes = 0;
							int totBytes = 0;
							
							while(readBlock){
								numBytes = 0;
								totBytes = 0;

								try{
							  	blockIndex = dis.readInt();

										if(blockIndex >= 500){
											System.out.println("blockIndex: " + blockIndex 
																			+" exceeds maximum number of blocks(500)");
											return;
										}
										if(blockIndex>=ofNode.noBlocks)
											fn.noBlocks += 1; 				//since old file

										numBytes = dis.read(tempArr);	
										totBytes = numBytes;

										while(totBytes < (int)fn.blockUnit){
												if(numBytes == -1){
													System.out.println("No more bytes");
													break;
												}
												
										numBytes = dis.read(tempArr,totBytes,((int)fn.blockUnit - totBytes));	
										if(numBytes != -1) totBytes += numBytes;

										}

										//Calculate blockSig
										md.update(tempArr, 0,totBytes); 
										tempSig = md.digest();
								
										//Replace old block signature
										System.arraycopy(tempSig, 0, fn.blockSig[blockIndex], 0, tempSig.length); 
										//need to 'seek' to the block index in the file because this 
										//is an old file.
										//Want to write only totBytes bytes.

										oraf.seek((blockIndex*fn.blockUnit));
										oraf.write(tempArr,0,totBytes);			//write the block to file
										if(totBytes<fn.blockUnit){
											System.out.println("Last block smaller than blockUnit. Truncating file if required.");
											long newLength = (blockIndex)*fn.blockUnit + totBytes;
											System.out.println("Old length: " + oraf.length());
											oraf.setLength(newLength);
											System.out.println("New length: " + oraf.length());
										}
								
								}catch(EOFException eofe){
										readBlock = false;
								}catch(IOException ioe){
										readBlock = false;
								}

							}//end of while(readBlock)
						
							//Now that the fNode contains the required blockSigs, enter it into 
							//the FileDirectory
							oraf.close();

							System.out.println("Writing modified FNode to FileDirectory.");
							FileDirectory.updateNode(fn);

							}
							
							/***DELETE FILE MESSAGE***/
						
							else if (fmpServer.msgType == RMFILE  && fmpServer.protocol==1){

								System.out.println("Received request to DELETE file.");

								BriefFileDesc bfd = new BriefFileDesc();
								bfd.byteDecode(fmpServer.data);

								if(!FileDirectory.contains(bfd.fileId)){
									System.out.println("***WRONG message: fileId does not exist***");
									return;
								}

								FNode ofNode = (FNode)FileDirectory.getNode(bfd.fileId);

								DiskInfo.increaseSpaceLeft(ofNode.fileSize);
								
								File dFile = new File(ofNode.localFName);
								System.out.println("Going to DELETE file: " + ofNode.localFName);
								boolean success = dFile.delete();
								System.out.println("Deleted file-"+success);

						 		if(success){
									FileDirectory.removeNode(bfd.fileId);
								}
							}	

							/***BLOCK_REQ***/

							else if (fmpServer.msgType == BLOCK_REQ){

							//Open a RandonAccessFile and keep it open for the duration 
							//of this conversation.
							//Keep seeking and returning blocks
				
							RequestArray rArr = new RequestArray();
							rArr.byteDecode(fmpServer.data);

							if(!FileDirectory.contains(rArr.fId)){
								System.out.println("Frivolous Request - Wrong File Id");
								dis.close();
								os.close();
								return;
							}

							FNode reqFN = FileDirectory.getNode(rArr.fId);
							
							//compare signature in block request with actual block signature.
							if(compareSHA(reqFN.blockSig[rArr.blockIndex-1], rArr.bSig)==0){

							}else{
								System.out.println("Frivolous Request - Wrong block signature");
								dis.close();
								os.close();
								return;
							}


							
							//Need to retrieve local file name from reqFN
							String fName = reqFN.localFName;
							System.out.println("Going to open local file: "+fName);
							RandomAccessFile raf = new RandomAccessFile(fName, "r");

							long pos;			//Start reading block from this position
							//Position before start of block
							pos = (rArr.blockIndex-1)*(reqFN.blockUnit);	
							
							try{
								raf.seek(pos);
							}catch(Exception e){
								System.out.println("Frivolous blockIndex: "+rArr.blockIndex);
								dis.close();
								os.close();
								raf.close();
								return;
							}

							boolean available = true;
							
							byte[] fileBlock = new byte[(int)reqFN.blockUnit];							
							int bytesRead = 0;
							int totBytes = 0;
							try{

								bytesRead = raf.read(fileBlock);	//Thats it!!!
								totBytes = bytesRead;

								while(totBytes<(int)reqFN.blockUnit){

										if(bytesRead==-1){
											System.out.println("No more bytes");
											available=false;		
											break;
										}
				
										bytesRead = raf.read(fileBlock, totBytes, (int)reqFN.blockUnit-totBytes);
										if(bytesRead != -1) totBytes += bytesRead;

								}
								
								System.out.println("Block index: " + rArr.blockIndex);

								/*** debugging
								System.out.println("Read following block from file: " 
																	+	fName);
								displayBlock(fileBlock);
								***/

							}catch(Exception e){
								System.out.println("Unable to read file block");
								dis.close();
								os.close();
								raf.close();
								return;
							}
							
							/***
							*	Wrap in FileManagerProtocol and send back requested block?
							* No, we want to return minimum info back (more seure).
							*	Just send back the encrypted block
							*	Assume that the receiver knows what to expect.
							***/
							os.write(fileBlock,0,totBytes);

							FileManagerProtocol nextFMP = new FileManagerProtocol();

							//Now repeat the process	
							while(available && (nextFMP.byteDecode(is))){

								//Assume request for same file - so reuse the 
								//RandomAccessFile handle. Can also reuse rArr

								rArr.byteDecode(nextFMP.data);

							//compare signature in block request with actual block signature.
							if(compareSHA(reqFN.blockSig[rArr.blockIndex-1], rArr.bSig)==0){

							}else{
								System.out.println("Frivolous Request - Wrong block signature");
								dis.close();
								os.close();
								return;
							}

								pos = (rArr.blockIndex-1)*(reqFN.blockUnit);	

								try{
									raf.seek(pos);
								}catch(Exception e){
									System.out.println("Frivolous blockIndex: "+rArr.blockIndex);
									dis.close();
									os.close();
									raf.close();
									return;
								}

								bytesRead=0;
								totBytes=0;

								try{
									//Zero out the block
									bytesRead = raf.read(fileBlock);	//Thats it!!!
									totBytes=bytesRead;
									while(totBytes<(int)reqFN.blockUnit){

										if(bytesRead==-1){
											break;
										}
				
										bytesRead = raf.read(fileBlock, totBytes, (int)reqFN.blockUnit-totBytes);
										if(bytesRead != -1) {
											totBytes += bytesRead;
											System.out.println("No more bytes");
											available=false;		
										}
									}

									System.out.println("Block index: " + rArr.blockIndex);

								}catch(Exception e){

									System.out.println("Unable to read file block");
									dis.close();
									os.close();
									raf.close();
									return;
								}

								os.write(fileBlock,0,totBytes);

							}

							//Close the RandomAccessFileStream
							raf.close();
							os.close();
							dis.close();

							}

							else {
								System.out.println("Unknown msgType received.");
							}
							
							break;

			case 4: //DT-FS - Same as case 1


							break;

			default: System.out.println("Unknown Protocol");

		}//end of switch

	}//end of handleClient


	/***
	* Purpose:  Compares 32 bits of SHA-1 signature sig2 to sig1
	*           Used for verifying whether the block signature 
	*           accompanying the block request matches the actual 
	*           block siganture.
	* Postcondition: returns 0 if sig2 matches sig1, -1 otherwise
	***/
	public int compareSHA(byte[] sig1, byte[] sig2){
		for(int sigByte=0; sigByte<8; sigByte++)
			if(sig2[sigByte] != sig1[sigByte]) return -1;
		return 0;
	}
		
	//Helper method, debugging
	public void displayBlock(byte[] block){

		System.out.println("*******");
		System.out.println("Block contains following bytes:");
		for(int i=0; i<block.length; i++)
			System.out.print(block[i]+" ");
		System.out.println("*******");
	}

}//end of class
