package FM;

import javax.servlet.*;
import javax.servlet.http.*;

import java.io.*;
import java.net.*;
import java.security.*;

/***
 * Purpose: The OwnerProtocol class is used to exchange messages between the 
 * 					FM and DT, FM and HH, and FM and Proxy. OwnerProtocol has data 
 * 					members and	methods for coding/decoding OwnerProtocol messages,
 * 					authenticating the user, and acting on the messages received 
 * 					from the user. The FM_Owner class is the driver (controls the 
 * 					methods	of this class) for this class.
 * 					
 *					Messages: 	1.  Submit File
 *											2.  Request File
 * 											5.	Request for File Deletion
 * 											6.	Request for FSListing and Block Sigs
 ***/

public class OwnerProtocol extends Thread implements FileManagerConstants {

	public byte protocol;    	/*2 for FM-DT protocol, 3 for FM-HH*/
	public byte msgType;			

	//Password stuff
	long timeStamp;
	long randNumber;
	int digLen;
	byte[] pSetDigest;

	public ServletInputStream osis;
	public ServletOutputStream osos;
	public BufferedInputStream bsis;
	public DataInputStream dis;
	
	public String ownerId;

	public OwnerNode ownerNode; //instantiated after ownerId has been read

	public FileDescription fDesc = null; 	//file submit
	/***OR***/
	public String reqFileId = null; 	//fileId of requested file (only for req)
	public String reqFileName = null;

	//constructor
	public OwnerProtocol(ServletInputStream osis, ServletOutputStream osos)throws Exception{
		
		this.osis = osis;
		this.osos = osos;

		System.out.println	("OwnerProtocol: Creating buffered input stream"
												+ " for servlet input");

    bsis = new BufferedInputStream(osis);
    dis = new DataInputStream(bsis);
	}

	/***
	 * Precondition: input stream from DT, HH or Proxy is initialized.
	 * Postcondition:	initializes data members based on message 
	 * 								received.
	 ***/
	
	public void byteDecode()throws Exception{

		System.out.println("OwnerProtocol: Reading from Owner...");
    int idLength = dis.read();
		System.out.println("OwnerProtocol-byteDecode: idLen " + idLength);	
    byte[] idArr = new byte[idLength];
    dis.readFully(idArr);

    this.ownerId = new String(idArr, DEF_ENCODING);
		System.out.println("OwnerProtocol-byteDecode: ownerId" + ownerId);	

    System.out.println("Owner Id read: " + this.ownerId);

		/***
		 * - 	Instantiate OwnerNode
		 * - 	We assume that the owner has already done the owner setup 
		 * 		through the webpage.
		 * -  Authentication is performed by the authenticateOwner() method.
		 ***/

		if((ownerNode = OwnerDirectory.getOwner(this.ownerId)) == null){
			System.out.println(	"Owner with ownerId: " + this.ownerId +
													" not in OwnerDirectory");
			return;
		}

		int oLen = dis.readInt(); 	//Length of OwnerProtocol
		
		byte[] oProt = new byte[oLen];
		dis.readFully(oProt);


	    ByteArrayInputStream bais = new ByteArrayInputStream(oProt);
	    DataInputStream bdis = new DataInputStream(bais);

	    System.out.println("Decoding byte array data for OwnerProtocol: ");

			this.protocol = (byte)bdis.read();
			this.msgType = (byte)bdis.read();
			System.out.println(	"Owner Protocol: " + this.protocol 
													+ " msgType: " + this.msgType);

			//Read password stuff - do not authenticate yet

			this.timeStamp = bdis.readLong();
			this.randNumber = bdis.readLong();
			this.digLen = bdis.readInt();

			pSetDigest = new byte[digLen];
			bdis.readFully(pSetDigest);
			System.out.println("Done with reading password stuff");

			System.out.println("Will verify password in authenticateOwner()");

			switch(msgType){

								
				case 1: {//file submit - read only FileDescription
								
								System.out.println("Decoding file description");
								int fDescLen = bdis.readInt();
								byte[] fDescArr = new byte[fDescLen];
								bdis.readFully(fDescArr);
								fDesc = new FileDescription();
								fDesc.byteDecode(fDescArr);

								System.out.println(	"Following File submitted:");
								System.out.println(fDesc);
								bdis.close();

								break;
								}

				case 2:	//file request - fileId string
								//Redundant - job taken over by Proxy

								int fdLen = bdis.readInt();
								byte[] tempId = new  byte[fdLen];
								bdis.readFully(tempId);
								this.reqFileName = new String(tempId, DEF_ENCODING);

								this.reqFileId = this.reqFileName;	

								System.out.println("\n******");
								System.out.println("reqFileName: " + this.reqFileName);
								System.out.println("reqFileId: " + this.reqFileName);
								System.out.println("******\n");

								System.out.println(this.reqFileId);
								System.out.println(	"Following file requested: " 
																		+ this.reqFileName);

								break;
				case 3:	//Request for file listing
								System.out.println(	"File listing request from owner: " 
																		+ this.ownerId);
								//No data in OwnerProtocol expected

								break;

				case 5: {//Request for file delete
								System.out.println("Received request for file Delete.");
								System.out.println("Decoding file description");
								int fDescLen = bdis.readInt();
								byte[] fDescArr = new byte[fDescLen];
								bdis.readFully(fDescArr);
								fDesc = new FileDescription();
								fDesc.byteDecode(fDescArr);

								System.out.println(	"Following File Deletion requested:");
								System.out.println(fDesc);
								bdis.close();
								break;
								}
								
				case 6:	//Request for FSList and Blocks Sigs.
								//Precondition: Info is being requested over https

								int fdLen2 = bdis.readInt();
								byte[] tempId2 = new  byte[fdLen2];
								bdis.readFully(tempId2);
								this.reqFileId = new String(tempId2, DEF_ENCODING);

								System.out.println("reqFileId: " + this.reqFileName);

								System.out.println(this.reqFileId);

								break;

				default:

		}//end of switch

	}//end of byteDecode


	/***
	 * Precondition: ownerId and password stuff have already been read.
	 * Postcondition: returns true/false based on result of authentication.
	 ***/

	public boolean authenticateOwner() throws Exception{
	 //gets ownerId and password stuff from class data members

		System.out.println("Attempting to authenticate Owner: " + ownerId);

		//Debugging
		System.out.println("Password stuff received from owner:");
		System.out.println("timeStamp: " + this.timeStamp);
		System.out.println("randNumber: " + this.randNumber);
		System.out.println("digLen: " + this.digLen);


		System.out.println("Checking for replay attack.");

		System.out.println("Comparing " + this.timeStamp+" : "+ownerNode.pwTimeStamp);
		
		if(this.timeStamp <= ownerNode.pwTimeStamp){

			System.out.println("Authentication information has old timeStamp.");
			System.out.println("Authentication failed!");
			return false;

		}
		
		//create a password digest by combining the above info with the 
		//stored password.

		byte[] cDigest = createDigest();

		//Compare created digest with digest received from owner
		int cLen = cDigest.length;

		boolean valid = false;
		
		//Compare lengths of digests
		if(cLen != digLen){

			System.out.println("Digests of different lengths");
			valid = false;
			return valid;

		}

		for(int digIndex = 0; digIndex<cLen; digIndex++){
			System.out.println(	"Comparing " + cDigest[digIndex] + ":" 
													+ this.pSetDigest[digIndex]);
			if(cDigest[digIndex] != this.pSetDigest[digIndex]){
				valid = false;
				return valid;
			}
		}

		System.out.println("Authentication Successful");	
		valid = true;			//all well
		ownerNode.pwTimeStamp = this.timeStamp;
		
		return valid;	
	}

	//Helper method: returns a digest of the timeStamp+randNumber
	//+password. Called by authenticateOwner() method.
	private byte[] createDigest() throws Exception{

		byte[] tsArr = getBytes(this.timeStamp);
		byte[] rmArr = getBytes(this.randNumber);

		byte[] pArr = ownerNode.password.getBytes(DEF_ENCODING);

		MessageDigest md = MessageDigest.getInstance("SHA-1");
		md.update(tsArr);
		md.update(rmArr);
		md.update(pArr);

		byte[] digest = md.digest();	//Also does a doFinal()

		return digest;

	}//end of createDigest

	/***
	*	The following helper method is from the book Wireless Java:
	*	Developing with Java[tm] 2 Micro Edition, by Jonathan Knudsen
	*	Postcondition: Returns a long data type as a byte array.
	***/
	private byte[] getBytes(long x){

		byte[] arr = new byte[8];
		for(int i = 0; i<8; i++){
			arr[i] = (byte)(x >> ((7-i) * 8));
		}

		return arr;
	}

	/***
	 * Precondition: 	data members ownerId, msgType and ownerNode are initialized.
	 * 								and the user has been authenticated.
	 * Postcondition: Acts on the message received from DT, HH or Proxy.
	 ***/

	/***
 	* File requests from DT, HH and Proxy are treated differently. DT saves 
 	* state (or explicitly requests it from FM) and consequently there
 	* is no need for DT to receive sigs of specific files. DT simply 
 	* receives the IP addresses of the FSs that store the requested file. 
 	* HH, however is stateless and before requesting Proxy for a file, retrieves
 	* the block sigs for the requested file from the FM. The Proxy must
 	* forward the ownerId and password digest that it received from the HH 
 	* to the FM and retrieve the block sigs from the FM.
 	***/
	public boolean actOnMsg()throws Exception{

		switch(msgType){
								

			case 1: /***
							*	FILE SUBMIT-could be NEW_FILE or OLD_FILE
							*
							*	Precondition: 	FileDescription has already been read
							*	Postcondition: 	calls storeAndForward() to forward file blocks 
							*									to five FSs
							***/
							System.out.println("Calling storeAndForward()");
							if(!storeAndForward()) return false;
							
							break;

			/***
			 * 	FILE REQUEST	
			 *	Precondition: 	OwnerNode has been instantiated, owner is authenticated
			 *									and the fileId of the requested had been read
			 *	Postcondition:	Either get the file from FSs - many2one(for HH) or send 
			 *									FS IPaddrs to DT for direct many2one, P2P file transfer 
			 ***/

			case 2: if(this.protocol == 2){ //Request from DT
								
								System.out.println("File request from DT. Calling sendFSInfo()");
								FNode fNode;
								System.out.println("Searching for following fileId-->"+reqFileId+"<--");
								if(ownerNode.fDir.contains(reqFileId)){
									//Retrieve the fNode
									fNode = ownerNode.fDir.getNode(reqFileId);
								}
								else{
									System.out.println("ERROR: Couldn't find fNode for fileId: " + reqFileId);
									return true;
								}

								DataOutputStream odos = new DataOutputStream(osos);
								sendFSInfo(odos,fNode); //send list of FS having file to DT				
								odos.close();
							
							}
							else if(this.protocol == 3){ 	//Request from HH

								//get the blocks of file in multiple threads from FSs
								//while simultaneously sending block by block to HH
								
								System.out.println("File request from HH. Calling downloadAndSend()");
								//downloadAndSend(); //Download many2one and send blocks to HH
								//No, now HH requests should go to Proxy.

							}
							else 
								System.out.println("ERROR: Do not understand protocol");

							break;
							
			case 3:	//Request for file listing. Dont care whether it is from DT or HH
							System.out.println("Calling sendBriefFileListing()");
							sendBriefFileListing();
							break;

			case 5: //Request for file DELETE from DT.

							System.out.println("Calling deleteFile().");
							deleteFile();
							

			case 6:	//Request for File Info - FSList and Block Sigs
							//Protocol will either be 2 (DT) or 4 (HHProxy)

							sendFileInfo();
							break;
			
			default:
							System.out.println("Unrecognized msgType: " + msgType); 

		}//end of switch


		return true;
		
		
	}//end of actOnMsg()

	//Sends list of FSs storing the requested file to DT or Proxy
	public void sendFSInfo(DataOutputStream odos, FNode fNode) throws Exception{

		System.out.println("Sending FS information to DT or Proxy");

		byte[] fsListArr = fNode.getFSList();	//Advantage of OOPs :-) 
		int fsListLength = fsListArr.length;
		odos.writeInt(fsListLength);
		odos.write(fsListArr);
		System.out.println("Finished sending FS information");
		//Dont close stream, may still need to send blockInfo
	}

	//Sends block sigs to DT, HH or Proxy.
	public void sendBlockInfo(DataOutputStream odos, FNode fNode) throws Exception{

		System.out.println("Sending blockInfo to DT or HHProxy");
		
		byte[] blockInfoArr = fNode.getBlockInfo();	//Advantage of OOPs :-) 

		odos.write(blockInfoArr);
		System.out.println("Finished sending block information");

	}
	
	/***
	 * 	Precondition: Owner is authenticated and connection is over https
	 *	Postcondition:First sends FSList and then the Block Sigs for the 
	 *								requested file
	 ***/

	public void sendFileInfo() throws Exception{

		DataOutputStream odos = new DataOutputStream(osos);
		
		FNode fNode;
		System.out.println("Searching for following fileId-->"+reqFileId+"<--");
		if(ownerNode.fDir.contains(reqFileId)){
			//Retrieve the fNode
			fNode = ownerNode.fDir.getNode(reqFileId);
		}
		else{
			System.out.println("ERROR: Couldn't find fNode for fileId: " + reqFileId);
			return;
		}

		System.out.println("Calling sendFSInfo()");
		sendFSInfo(odos, fNode);

		System.out.println("Calling sendBlockSigs()");
		sendBlockInfo(odos, fNode);

		odos.close();
	}

	//Sends important file info including block sigs.
	//Does not send the FSList since HH does not require that.

	public void sendBriefFileListing() throws Exception{

		DataOutputStream odos = new DataOutputStream(osos);

		if(this.protocol == 3){  //request from HH

			byte[] proxyURLArr = PROXY_URL.getBytes(DEF_ENCODING);
			System.out.println("Sending PROXY_URL: " + PROXY_URL + " to HH");
		
			odos.writeInt(proxyURLArr.length);
			odos.write(proxyURLArr);

			System.out.println("Sending brief File Listing to Owner");
			byte[] bfListing = ownerNode.fDir.getFileListing();
		
			odos.writeInt(bfListing.length);
			odos.write(bfListing);

			System.out.println("Finished sending brief File Listing to Owner");
			odos.flush();

			return;
		}

		//request from DT
		
		System.out.println("Sending brief File Listing to Owner");
		byte[] bfListing = ownerNode.fDir.getBFNodeListing();
		
		odos.writeInt(bfListing.length);
		odos.write(bfListing);

		System.out.println("Finished sending brief File Listing to Owner");
		odos.flush();

	}

	//Helper method, for debugging
	public void displayBlock(byte[] block, int totBytes){

	System.out.println("Block has following bytes:");
	System.out.println("\n******");
	for(int i=0;i<totBytes;i++)
		System.out.print(block[i]+" ");
	System.out.println("******\n");

	}

	/***
	 * Precondition: user is authenticated. msgType is DELETE.
	 * Postcondition: file delete messages are sent to the FSs
	 * 								storing the file.
	 ***/

	public boolean deleteFile() throws Exception{

		boolean oldFile = ownerNode.fDir.contains(fDesc.fileId);	
		if(oldFile){
			System.out.println("deleteFile(): Request for DELETING an oldFile.");
		}
		else {
			System.out.println("deleteFile(): fileId does not exist in Owner's File Directory.");
			return false;
		}

		FNode ofNode = (FNode)ownerNode.fDir.getNode(fDesc.fileId);
		ownerNode.spaceLeft += fDesc.maxEncFileSize;


		BriefFileDesc bfd = new BriefFileDesc(fDesc);
		FileManagerProtocol consumerFMP = new FileManagerProtocol((byte)RMFILE, bfd.byteEncode());

		byte[] fileInfo = consumerFMP.byteEncode();	

		
		//Send delete msgs to 5 FSs
		for(int fsIndex=0; fsIndex<5; fsIndex++){

			String fsIP = ofNode.FS[fsIndex];

    	OutputStream out;
    	DataOutputStream dos;
    	Socket fsClient;

    	try{
    		fsClient = new Socket(fsIP, FSPORT);
    	}catch(Exception ec){
    		System.out.println( "could not open connection to fsIP: " + fsIP +
    												" on port: " + FSPORT);
				continue;
    	}

    	out = fsClient.getOutputStream();
    	dos = new DataOutputStream(out);

			long timeStamp = System.currentTimeMillis();
			byte[] fmSig = FMSign.sign(fileInfo,timeStamp);
			System.out.println("Sending fileInfo to FS: " + fsIP);
			dos.write(fileInfo); 						//send FMP packet
			System.out.println("Sending timeStamp:"+timeStamp);
			dos.writeLong(timeStamp);				//New-to prevent replay attack
			System.out.println("Sending sigLen:"+fmSig.length);
			dos.writeInt(fmSig.length);			//New-length of sig of data+timeStamp
			System.out.println("Sending fmSig.");
			dos.write(fmSig);								//New-sig of data+timeStamp
			dos.close();

		}//end of for

		//Now remove fNode from owner's file directory.
		
		System.out.println("Removing fNode from owner's File Directory.");
		ownerNode.fDir.removeNode(ofNode.fileId);
		System.out.println("Done.");
					
						
		//Send delete conf msg to DT
		DataOutputStream odos = new DataOutputStream(osos);
		System.out.println("Going to send RMFILE_CONF to DT.");
		odos.writeInt(RMFILE_CONF);
		odos.close();
						
		return true;

	}//end of deleteFile()
	
	/****
	 * Precondition: 	ownerNode has been instantiated.
	 * 							 	OwnerProtocol (which contains the FileDescription)
	 * 								has already been read
	 * Postcondition:	Receive file being submitted, store pertinent info about 
	 * 								file and owner, forward the file blocks to 5 FSs.
	 ****/

	public boolean storeAndForward(){

	//First transfer relevant info from the FileDescription just read 
	//to an FNode object.
		
	FNode fileNode = new FNode();
	fileNode.fileId = fDesc.fileId; //store fileId of submitted file

	boolean oldFile = ownerNode.fDir.contains(fileNode.fileId);

	/*Modified later - java wont accept true/false as switch cases :-( */
		int NEWFILE;


		if(oldFile){ 
					NEWFILE=0;
					System.out.println("Received an OLDFILE from owner.");
		}
		else{
					NEWFILE=1;
					System.out.println("Received a NEWFILE from owner.");
		}

	switch(NEWFILE){

		case 1:		{
							//Lock fDir 
						
							//Check for available space
							if ( fDesc.maxEncFileSize > ownerNode.spaceLeft ){
								System.out.println("File Submit exceeds available space");
								return false;
							}
							//update spaceLeft
							ownerNode.spaceLeft -= fDesc.maxEncFileSize;

							fileNode.ownerId = fDesc.ownerId;
							fileNode.idLength = fDesc.idLength;

							System.out.println(	"Storing ownerId in fileNode : " 
																	+ fDesc.ownerId);

							fileNode.fIdLength = fDesc.fIdLength;
							fileNode.fileSize = fDesc.maxEncFileSize;
							fileNode.fileName = fDesc.fName;
							fileNode.blockUnit  = fDesc.blockUnit;
							fileNode.noBlocks = (int)Math.ceil((double)(fDesc.maxEncFileSize)/fDesc.blockUnit);
							
							//Get list of IP addresses of FSs where this file will be stored 
							//Need to check if getReplServers returns null

								try{
									System.arraycopy(FileServerDirectory.getReplServers(fileNode.fileSize), 0,fileNode.FS, 0, 5);
								}catch(Exception e){
									System.out.println(	"Could not get 5 replication servers-"
																			+e.getMessage());
									return false;
								}

							int sigSize = 0;
							try{
								sigSize = (int)dis.readInt();
							}catch(Exception e){
								System.out.println("EXCEPTION: " + e.getMessage());
							}

							System.out.println("sigSize of block: " + sigSize);
							
							//Needed so that space is allocated for the blockSigs
							fileNode.setBlockInfo(sigSize,fDesc);

							System.out.println("Expected number of encrypted blocks: " + fileNode.noBlocks);

							byte[] tempBlock = new byte[(int)fileNode.blockUnit];

							byte[] tempSig = new byte[fileNode.sigSize];

							FileBlock fileBlock = new FileBlock((int)fileNode.blockUnit);
							
							/***
							 * Start 5 other threads and pass fBlock and a FS IP address.
							 * Since get() of FileBlock is synchronized, the threads will
							 * simply wait until a block is available
							 ***/
						
							for(int FSIndex=0; FSIndex < 5;FSIndex++){
								new Thread(new FSConsumer(fDesc, fileBlock, FSIndex, fileNode,true)).start(); 
							}

							int numBytes = 0;
							int totBytes = 0;
							System.out.println("Num blocks: "+fileNode.noBlocks);
							
							for(int blockIndex = 0; blockIndex<fileNode.noBlocks; blockIndex++){
								//Read a signature and a block repeatedly
								numBytes = 0;
								totBytes = 0;

								//Last block may not have blockUnit bytes - IMPORTANT
								int numSigBytes = -1;
								int totSigBytes = -1;
								try{
									numSigBytes = dis.read(tempSig);

									totSigBytes = numSigBytes;
									while(totSigBytes<fileNode.sigSize){
											if(numSigBytes==-1) break;
											numSigBytes = dis.read(tempSig, totSigBytes, fileNode.sigSize-totSigBytes);
											if(numSigBytes != -1) totSigBytes += numSigBytes;
									}
									
								}catch(Exception e){
										System.out.println(	"EXCEPTION: " + e.getMessage());
								}
								
								try{
									numBytes = dis.read(tempBlock);
								}catch(Exception e){
									System.out.println("EXCEPTION: " + e.getMessage());
								}
								
								System.out.println(	"numSigBytes= " + numSigBytes + " numBytes= " 
																		+ numBytes);
								totBytes = numBytes;
								System.out.println("Read " + numBytes + " block bytes from DT");
								System.out.println("Setting totBytes to: "+totBytes);

								while(totBytes < (int)fileNode.blockUnit){
            			System.out.println(	"Bytes read from DT in one read: " 
																			+ numBytes);
            			if(numBytes == -1){
             				System.out.println("No more bytes");
             				break;
            			}

									System.out.println("Could not read all bytes of block...reading again");
									try{
          					numBytes = dis.read(tempBlock,totBytes,((int)fileNode.blockUnit - totBytes));
									}catch(Exception e){
										System.out.println("EXCEPTION: " + e.getMessage());
									}
          				
									if(numBytes != -1) totBytes += numBytes;

        				}

								System.out.println("Total bytes read in block: " + totBytes);

								//write the block signature to fileNode
								System.arraycopy(tempSig, 0, fileNode.blockSig[blockIndex], 0, tempSig.length);	
								//Last block may not have blockUnit bytes? Solved.

								//SYNCHRONIZED - wont return till the previous block has been 
								//grabbed by 5 FSs (from 5 other threads)!
								System.out.println("FM: Submitting following to fileBlock " + blockIndex);
								displayBlock(tempBlock,totBytes);
								fileBlock.put(tempBlock, blockIndex, totBytes);
								
							}

							//Required to send a terminate to the consumer threads	
							fileBlock.put(tempBlock, -1, 0);
								
							//Add the fNode to the file directory - last step
							ownerNode.fDir.addNode(fileNode);
							break;
							}


		case 0:		//OLD_FILE
							{
							//Get the FNode of OLD_FILE
							FNode ofNode = (FNode)ownerNode.fDir.getNode(fileNode.fileId);

							//update spaceLeft
							ownerNode.spaceLeft -= fDesc.maxEncFileSize;
							ownerNode.spaceLeft += ofNode.fileSize;

							//Check for available space
							if ( fDesc.maxEncFileSize > ownerNode.spaceLeft ){
								System.out.println("File Submit exceeds available space");
								return false;
							}
							fileNode.ownerId = fDesc.ownerId;
							fileNode.idLength = fDesc.idLength;

							System.out.println(	"Storing ownerId in fileNode : " 
																	+ fDesc.ownerId);

							fileNode.fIdLength = fDesc.fIdLength;
							fileNode.fileSize = fDesc.maxEncFileSize;
							fileNode.fileName = fDesc.fName;
							fileNode.blockUnit  = fDesc.blockUnit;
							fileNode.noBlocks = (int)Math.ceil((double)(fDesc.maxEncFileSize)/fDesc.blockUnit);

							//Get list of IP addresses of FSs where this file will be stored 

							//Copy old set of FS to new FNode
							System.arraycopy(ofNode.FS, 0, fileNode.FS, 0, 5);

							int sigSize = 0;
							try{
								sigSize = (int)dis.readInt();
							}catch(Exception e){
								System.out.println("EXCEPTION: " + e.getMessage());
							}

							System.out.println("sigSize of block: " + sigSize);
							
							//Needed so that space is allocated for the blockSigs
							fileNode.setBlockInfo(sigSize,fDesc);

							//Copy old sigs to new fileNode
							System.arraycopy(ofNode.blockSig, 0, fileNode.blockSig, 0, 500);
							
							System.out.println("Expected number of encrypted blocks: " + fileNode.noBlocks);

							byte[] tempBlock = new byte[(int)fileNode.blockUnit];

							byte[] tempSig = new byte[fileNode.sigSize];

							FileBlock fileBlock = new FileBlock((int)fileNode.blockUnit);
							
							/***
							 * Start 5 other threads and pass fBlock and a FS IP address.
							 * Since get() of FileBlock is synchronized, the threads will
							 * simply wait until a block is available
							 ***/
						
							for(int FSIndex=0; FSIndex < 5;FSIndex++){
								new Thread(new FSConsumer(fDesc, fileBlock, FSIndex, fileNode,false)).start(); 
							}

							int numBytes = 0;
							int totBytes = 0;
							int blockIndex=0;

							System.out.println("Num blocks: "+fileNode.noBlocks);
							
							boolean available = true;		//false, when no more blocks 
																					//are available

							while(available){
							//Read a signature and a block repeatedly
								numBytes = 0;
								totBytes = 0;

								//Last block may not have blockUnit bytes - IMPORTANT
								int numSigBytes = -1;
								int totSigBytes = -1;

								try{
									numSigBytes = dis.read(tempSig);
									totSigBytes = numSigBytes;
									while(totSigBytes<fileNode.sigSize){
											if(numSigBytes==-1) break;
											numSigBytes = dis.read(tempSig, totSigBytes, fileNode.sigSize-totSigBytes);
											if(numSigBytes != -1) totSigBytes += numSigBytes;
									}
									
									blockIndex = dis.readInt();
								}catch(Exception e){
									System.out.println("EXCEPTION: " + e.getMessage());
								}
								
								try{
									numBytes = dis.read(tempBlock);
								}catch(Exception e){
									System.out.println("EXCEPTION: " + e.getMessage());
								}
								
								System.out.println("numSigBytes= " + numSigBytes + " numBytes= " + numBytes);
								totBytes = numBytes;

								while(totBytes < (int)fileNode.blockUnit){
            			System.out.println(	"Bytes read from DT in one read: " 
																			+ numBytes);
            			if(numBytes == -1){
             				System.out.println("No more bytes");
										available=false;
             				break;
            			}

									System.out.println("Could not read all bytes of block...reading again");
									try{
          				numBytes = dis.read(tempBlock,totBytes,((int)fileNode.blockUnit - totBytes));
									}catch(Exception e){System.out.println("EXCEPTION: " + e.getMessage());}
          				if(numBytes != -1) totBytes += numBytes;

        				}

								System.out.println("Total bytes read in block: " + totBytes);

								//write the block signature to fileNode

								System.arraycopy(tempSig, 0, fileNode.blockSig[blockIndex], 0, tempSig.length);	
								//Last block may not have blockUnit bytes? Solved.

								//SYNCHRONIZED - wont return till the previous block has been 
								//grabbed by 5 FSs (from 5 other threads)!
								System.out.println("FM: Submitting to fileBlock " + blockIndex);
								fileBlock.put(tempBlock, blockIndex, totBytes);
								
							}

							//Required to send a terminate to the consumer threads	
							fileBlock.put(tempBlock, -1, 0);
								
							//Add the new fNode to the file directory-replaces ofNode
							ownerNode.fDir.updateNode(fileNode);

							break;
							}	
	}
	
	return true;		//all well

	}//end of storeAndForward()

} //end of class

/***
 * Purpose:	Five FSConsumer objects are instantiated by the storeAndForward()
 * 					method in five threads. Each FSConsumer object sends the file 
 * 					blocks to one FS.
 ***/

class FSConsumer implements Runnable, FileManagerConstants{

				private FileBlock fBlock; 
				private int fsNum;
				private String fsIP;
				int numBlocks;
				boolean newFile = true; //No danger in resending an old file
				FNode fNode;

				//data members that will be byteEncoded
				String fId;							//fileId

				byte[] blockBytes;
				FileDescription fDesc;

				//constructor	
				public FSConsumer(FileDescription fd, FileBlock fb, int FSNum, FNode fn, boolean newFile){

								fNode = fn;
								fBlock 	= fb;
								fsNum 	= FSNum;
								fsIP		= fn.FS[FSNum];
								System.out.println("\nReplication FS " + fsNum + ": "+ fsIP);
								this.numBlocks = fn.noBlocks;
								this.newFile = newFile;
								fId = fn.fileId;

								/***
								*	WRONG EARLIER: We are putting the block index in addition to
								*	the block bytes. So we need to allocate more space. 
								*	blockBytes = new byte[(int)fn.blockUnit];
								*	(the +4 is for the index)
								***/

								blockBytes = new byte[(int)fn.blockUnit+4];	
								fDesc = fd;
				}

				public void run(){
				//send fileId, blockIndex, blockSig, block 
					try{	
				
					OutputStream out; 
					DataOutputStream dos=null;
					Socket fsClient = null; 
					boolean error = false;
					try{
						//Open a connection to the FS 
						fsClient = new Socket(fsIP, FSPORT);
					}catch(Exception ec){
						System.out.println(	"could not open connection to fsIP: " + fsIP +
																" on port: " + FSPORT);
						error = true;	
					}
					if(!error){
					out = fsClient.getOutputStream();
					dos = new DataOutputStream(out);
					}
					System.out.println("Opened connection to FS: " + fsIP);
					//send FileManagerProtocol packet, including fileId in the data region
					//BriefFileDesc as part of data region

					BriefFileDesc bfd = new BriefFileDesc(fDesc);	
					
					FileManagerProtocol consumerFMP;
					
					if(this.newFile)
						consumerFMP = new FileManagerProtocol((byte)NEW_FILE, bfd.byteEncode()); 
					else
						consumerFMP = new FileManagerProtocol((byte)OLD_FILE, bfd.byteEncode()); 
					
				  byte[] fileInfo = consumerFMP.byteEncode();

					/***
					 * Authenticating FM to FS.
					 *
					 * Before we send the msg to FS we will sign it using FM's 
					 * private key and send the msgSig after the msg itself. The 
					 * FS will compare the msgSig we send with the msgSig it 
					 * generates using the FM's public key. This will serve 2 
					 * purposes - authenticate the FM to the FS and ensure that 
					 * the msg is not tampered with. We also put the timestamp 
					 * in the msg to prevent replay attacks.
					 ***/

					long timeStamp = System.currentTimeMillis();
					byte[] fmSig = FMSign.sign(fileInfo,timeStamp);
					if(!error){
						System.out.println("Sending fileInfo");
						dos.write(fileInfo); 						//send FMP packet
						System.out.println("Sending timeStamp:"+timeStamp);
						dos.writeLong(timeStamp);				//New-to prevent replay attack
						System.out.println("Sending sigLen:"+fmSig.length);
						dos.writeInt(fmSig.length);			//New-length of sig of data+timeStamp
						System.out.println("Sending fmSig.");
						dos.write(fmSig);								//New-sig of data+timeStamp
					}

					boolean available = true;	
					while(available){
						//get block
						//send block
						byte[] tempBytes = fBlock.get(this.fsNum);

						if(tempBytes == null) {
							System.out.println("No more blocks to send to FS " + fsIP);
							available = false;
							break; 																		//break out of the for loop
						}
						//tempBytes contains both the blockIndex and the block
						//last block incomplete? Solved.
						System.arraycopy(tempBytes, 0, blockBytes, 0, tempBytes.length); 
						if(!error){
						//write to the FS
						System.out.println("Writing block " + " to FS " + fsIP);
						dos.write(blockBytes, 0, tempBytes.length);
						}
						//Why transmit the block sig when FS can calculate this itself
						//and screen requests that contain the wrong sig (to prevent dos 
						//attacks)
						
						//What if man-in-the-middle changes the block in transmission

					}
					System.out.println("Closing connection to FS " + fsIP);
					System.out.println("");	
					if(!error){
					dos.close();		//close connection to FS
					}
				}	
				catch(Exception e){
					System.out.println("Exception in run() of FSConsumer");
				}

				}//end of run()

} //end of class FSConsumer







