package FM;

	/***
	 * Purpose: This file contains classes that define the data structures
	 * 					that the FM uses to store state information.
	 ***/ 					

import java.util.*;
import java.io.*;
import java.net.*;
import java.security.*;

/***
 * Purpose: The FNode object stores information about a file.
 * 					The FNode object is stored in the FileDirectory
 * 					tree.
 ***/

class FNode implements FileManagerConstants, Serializable{

	public String ownerId;						
	public int idLength;
	public String fileId = null;			// Will also be used as the key 
																		// of the (key, FNode) pair of the
																		// FileDirectory TreeMap
	int fIdLength = 0;
	public String fileName;
	public String[] FS = new String[5]; 	// File Server IPs where this file is
																				// replicated
	
	int sigSize;													// size of block signature
	public int noBlocks;                	// Number of encrypted blocks in the file
	public long blockUnit;    	          // size of one (encrypted) block
	
	//entry 0 corresponds to the signature of block 0 and so on.
	byte[][] blockSig;

	public long fileSize;
	
	// If more blocks are needed, point to another FNode.
	// Just a provision for future implementation
	FNode nextNode = null; 								//Initialised only if more than 500 
																				//blocks are required for the file  						
//empty constructor - data members will be set explicitly
public FNode() {}

public void setBlockInfo(int sSize, FileDescription fDesc){

	sigSize = sSize;
	blockSig = new byte[500][sigSize]; //500 blocksigs each sig of size sigSize
	fileSize = fDesc.maxEncFileSize;

}

//Called after entire file has been submitted
public void setNumBlocks(int numBlocks){
	noBlocks = numBlocks;
}

/***
 * Postcondition: Returns list of FSs storing this file
 * 								as a byte array for transmission over
 * 								the network.
 ***/

public byte[] getFSList() throws Exception{

		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream	bados	= new DataOutputStream(baos);

		
		for(int fsIndex = 0; fsIndex < 5; fsIndex++){

			byte[] fsArr = this.FS[fsIndex].getBytes(DEF_ENCODING);
			bados.writeInt(fsArr.length);
			bados.write(fsArr);

		}

		bados.flush();
		return baos.toByteArray();
}

/***
 * Postcondition: Returns sigSize, noBlocks, blockUnit followed by 
 * 								block sigs. CAREFUL while decoding...
 ***/
public byte[] getBlockInfo() throws Exception{

		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream	bados	= new DataOutputStream(baos);

		bados.writeInt(sigSize);
		bados.writeInt(noBlocks);
		bados.writeLong(blockUnit);

		byte[] sig = new byte[this.sigSize];

		for(int blockIndex = 0; blockIndex < this.noBlocks; blockIndex++){

			System.arraycopy(this.blockSig[blockIndex], 0, sig, 0, this.sigSize);
			bados.write(sig);

		}

		bados.flush();
		return baos.toByteArray();
		
}
/***
 * Postcondition: Returns a brief version of FNode, specifically
 * for transmitting back to HH
 ***/

public byte[] encodeBrief() throws Exception{

		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream	bados	= new DataOutputStream(baos);

		//Wont need this when we begin storing fileId as byte[]
		byte[] fileIdArr = fileId.getBytes(DEF_ENCODING);
		bados.writeInt(fileIdArr.length);
		bados.write(fileIdArr);
		
		//File Name may be different from fileId
		byte[] nameArr = fileName.getBytes(DEF_ENCODING);
		bados.writeInt(nameArr.length);
		bados.write(nameArr);

		bados.writeInt(sigSize);
		bados.writeInt(noBlocks);
		bados.writeLong(blockUnit);
		bados.writeLong(fileSize);
		for(int bIndex=0; bIndex<noBlocks; bIndex++)
			bados.write(blockSig[bIndex]);

		bados.flush();
		return baos.toByteArray();
}

	//Helper function, for debugging
public String toString(){
	
	String fNodeInfo;
	
	fNodeInfo = " File Id: " + fileId + " Name: " + fileName + 
				"\n FS 0: " + FS[0] + " FS 1: " + FS[1] + " FS 2: " + FS[2] +
				"\n FS 3: " + FS[3] + " FS 4: " + FS[4] + "\n";
	fNodeInfo += " Owner Id: " + ownerId;

	return fNodeInfo;
}

}// end of class

/***
 * Purpose: An FSNode object contains information about a file server.
 * 					An FSNode object is stored in the FileServerDirectory tree.
 ***/

class FSNode implements FileManagerConstants, Serializable{

	public int idLength;
	public String serverId; 	//The id of the file server
	public long totSpace; 		//Total space on offer
	public long spaceLeft;    //space left unused out of totSpace

	public String fileServerIP;
	public String fileManagerIP;

	public String[] ownerList; 	//just a reference list of Owners here	 		
	public int numOwners = 0; 	//number of current owners here
															//should be ownerList.length.
															//not used in current implementation

	public boolean active=true;			//false is FS has been shutdown
	
	/***
	 * Precondition: 	DInfo contains setup information sent to the FM by 
	 * 								FS.
	 * Postcondition: FSNode data members are initialized.
	 ***/
	public FSNode(DiskInfo DInfo) throws Exception{

		//make a new serverId using SHA-1 on the server's IP address	
		//and a random number

		MessageDigest md = MessageDigest.getInstance("SHA-1");
		md.update((DInfo.myIP).getBytes());

		//add random data to digest
		Random rand = new Random(System.currentTimeMillis());
		byte[] randBytes = new byte[20];
		rand.nextBytes(randBytes);

		md.update(randBytes);

		byte digest[]  = md.digest();	
		System.out.println("Encoding digest using " + DEF_ENCODING);

		serverId = new String(digest, DEF_ENCODING);
		byte[] encBuf = serverId.getBytes(DEF_ENCODING);
		idLength = encBuf.length;

		/*** Debugging...
		System.out.println();
		System.out.println("SHA-1Hash: " + digest);
		System.out.println("length of SHA-1Hash is: " + digest.length);
		System.out.println();	

		System.out.println();
		System.out.println("serverId(string): " + serverId);
		System.out.println("idLength: " + idLength);
		System.out.println("digestLength: " + digest.length);
		System.out.println();	
		***/

		totSpace = DInfo.totSpace;

		//space left unused out of totSpace
		spaceLeft = DInfo.spaceLeft;

		ownerList = new String[20]; 	//max 20 owners.
																	//not used.
		
		fileServerIP = new String(DInfo.myIP.trim());
		fileManagerIP = new String(DInfo.fileManagerIP.trim());
		
		System.out.println("Finished constructing FSNode");
	}
	
}//end of class

/***
 * Purpose: The OwnerDirectory constains the TreeMap of all OwnerNode
 * 					data structures. OwnerNode is defined in file OwnerNode.java
 * 					One OwnerNode object contains information about one Diskster
 * 					user. The OwnerDirectory class had methods to save state 
 * 					it state to file and to initialize itself from that stored 
 * 					state, if required.
 ***/

public class OwnerDirectory implements FileManagerConstants{

	static TreeMap ownerTree; //static OwnerDirectory common to all instances

	//constructor
	public OwnerDirectory(){
		ownerTree = new TreeMap();
		/***
		 * If setup file exists, re-initialize.
		 ***/

		if(ODFILE.exists()){
			
			System.out.println("Going to read ownerTree from file: " + ODFILE);
			try{
				FileInputStream fis = new FileInputStream(ODFILE);
				ObjectInputStream ois = new ObjectInputStream(fis);
				ownerTree = (TreeMap)ois.readObject();
				System.out.println("Initialized ownerTree from saved state.");
				ois.close();
			}catch (Exception e){
				System.out.println("OwnerDirectory: Exception-" + e.getMessage());
			}
		}
		else {
			System.out.println(ODFILE + " does not exist.");
			System.out.println("Starting out with empty ownerTree.");
		}
		
	}

	//Purpose: Saves OwnerDirectory state to file
	public static void writeToFile(){

		try{
			FileOutputStream fos = new FileOutputStream(ODFILE);
			ObjectOutputStream oos = new ObjectOutputStream(fos);
			oos.writeObject(ownerTree);
			oos.flush();
			oos.close();
		}catch(Exception ee){
				System.out.println(	"OwnerDirectory writeToFile: Exception-"
														+ ee.getMessage());
		}
					
	}
	//Checks if OwnerNode with ownerId oId exists in the OwnerDirectory
	public static boolean containsNode(String oId){
		return ownerTree.containsKey(oId);
	}
	
	//Adds OwnerNode oNode to the OwnerDirectory	
	public static boolean addOwner(OwnerNode oNode){
		
		if(ownerTree.containsKey(oNode.ownerId)) return false;
		
		ownerTree.put(oNode.ownerId, oNode);
		return true;
	}	

	//Retrieves OwnerNode with ownerId from OwnerDirectory
	public static OwnerNode getOwner(String ownerId){
	
		System.out.println("Retrieving OwnerNode for ownerId: " + ownerId);
		OwnerNode on = (OwnerNode)ownerTree.get(ownerId);
		on.updateTimeStamp(System.currentTimeMillis());
		return on;

	}

	//Removes OwnerNode with oId from the OwnerDirectory
	public static boolean removeNode(String oId){

		ownerTree.remove(oId);
		return true;

	}

}//end of class

//Tree of FNode(s) of a particular owner

/***
 * Purpose: FileDirectory contains the TreeMap of all FNode 
 * 					objects of a particular user. FileDirectory itself
 * 					is stored as a data member of OwnerNode.
 ***/

class FileDirectory implements FileManagerConstants, Serializable{

	TreeMap fileTree;
	
	public FileDirectory(){
	fileTree = new TreeMap();
	}

	public boolean addNode(FNode fNode){
	
		if(fileTree.containsKey(fNode.fileId)) return false;
		
		fileTree.put(fNode.fileId, fNode);
		return true; 
	}

	public boolean removeNode(String fId){

		fileTree.remove(fId);
		return true;

	}
	
	/***
	 * 	Precondition:		Called when the fNode is already present in 
	 * 									the FileDirectory. Called when a file has 
	 * 									been modified by its owner.
	 * 	Postcondition: 	Replaces an FNode object in the FileDirectory.
	 ***/
public boolean updateNode(FNode ufNode){
	
	//redundant based on our precondition
	if(!fileTree.containsKey(ufNode.fileId)) return false;
	
	fileTree.put(ufNode.fileId, ufNode);
	
	return true;
	
	}
	
//Retrieves FNode object with fileId from the FileDirectory.
public FNode getNode(String fileId){
	return (FNode)(fileTree.get(fileId));
}

/***
 * Postcondition: Returns true if fileId is already present, 
 * 								false otherwise
 ***/
	public boolean contains(String cfId){
		boolean conFlag;

		System.out.println("Checking for following fId in fDir: " + cfId);
		conFlag = fileTree.containsKey(cfId);
		System.out.println("result of check: " + conFlag);
		return conFlag;
	}

	/***
	 *	Postcondition: 	Returns a listing of BriefFNode(s) of this 
	 * 									owner, in a format fit to be transmitted to HH
	 ***/
	public byte[] getBFNodeListing() throws Exception{

	System.out.println("In getBFNodeListing()");
	ByteArrayOutputStream baos = new ByteArrayOutputStream();
	DataOutputStream	bados	= new DataOutputStream(baos);

	Set fSet = fileTree.entrySet();
	Iterator it = fSet.iterator();

	while(it.hasNext()) {
		Map.Entry me = (Map.Entry)it.next();
		FNode fNode = (FNode)me.getValue();
		byte[] bfNodeArr = fNode.encodeBrief();
		bados.writeInt(bfNodeArr.length);
		bados.write(bfNodeArr);
	}

	bados.flush();
	return baos.toByteArray();

	}

	/***
	 * Postcondition: 	Returns a listing of file names, fIds of this 
	 * 									owner, in a format fit to be transmitted to HH
	 ***/
	public byte[] getFileListing() throws Exception{

	System.out.println("In getFileListing()");
	ByteArrayOutputStream baos = new ByteArrayOutputStream();
	DataOutputStream	bados	= new DataOutputStream(baos);

	Set fSet = fileTree.entrySet();
	Iterator it = fSet.iterator();

	while(it.hasNext()) {

		Map.Entry me = (Map.Entry)it.next();
		FNode fNode = (FNode)me.getValue();

		byte[] nameArr = fNode.fileName.getBytes(DEF_ENCODING);
		byte[] idArr = fNode.fileId.getBytes(DEF_ENCODING);

		
		bados.writeInt(nameArr.length);
		bados.write(nameArr);

		bados.writeInt(idArr.length);
		bados.write(idArr);

	}

	bados.flush();
	return baos.toByteArray();

	}


}//end of class



/***
 * Purpose: The FileServerDirectory contains the TreeMap to store
 * 					the FSNode objects of the file servers registered with
 * 					FM. This class has methods to save it state to a file, 
 * 					if required.
 ***/

class FileServerDirectory implements FileManagerConstants{
	
	//static - common fSTree for all instances
	static TreeMap fSTree;		
	//for circular first fit.
	static String prevReplServer = new String("x");
	
	public FileServerDirectory(){
		fSTree = new TreeMap();

		/***
		 * If setup file exists, re-claim state.
		 ***/

		if(FSDFILE.exists()){
			
			System.out.println("Going to read fSTree from file: " + FSDFILE);
			try{
				FileInputStream fis = new FileInputStream(FSDFILE);
				ObjectInputStream ois = new ObjectInputStream(fis);
				fSTree = (TreeMap)ois.readObject();
				System.out.println("Initialized fSTree from saved state.");
			}catch (Exception e){
				System.out.println("FileServerDirectory: Exception-" + e.getMessage());
			}
		}
		else {
			System.out.println(FSDFILE + " does not exist.");
			System.out.println("Starting out with empty fSTree.");
		}

	}

	//Saves FileServerDirectory state to file
	public static void writeToFile(){

		try{
			FileOutputStream fos = new FileOutputStream(FSDFILE);
			ObjectOutputStream oos = new ObjectOutputStream(fos);
			oos.writeObject(fSTree);
			oos.flush();
			oos.close();
		}catch(Exception ee){
				System.out.println("FileServerDirectory writeToFile: Exception-"+ee.getMessage());
		}
					
	}
	
	//Stores an FSNode object to the FileServerDirectory.	
	public static boolean addNode(FSNode fSNode){

		System.out.println("Attempting to add fSNode");
		
		if(fSTree.containsKey(fSNode.serverId)){
		System.out.println("serverId already exists in FileServerDirectory");
		return false;
		}

		//Actually add the node
		fSTree.put(fSNode.serverId,fSNode);
	
		System.out.println("Added fSNode");
		return true;

		}
//Retrieves an FSNode object from the FileServerDirectory
public static FSNode getNode(String serverId){
	try{
		return (FSNode)(fSTree.get(serverId));
	}catch(Exception e){
		System.out.println("FileDirectory-Exception-Could not get FS: " + serverId);
		return null;
	}
}
//Helper function, traces the FileServerDirectory
public static void showTrace(){

	System.out.println("\nRegistered File Servers - serverIds:");
	
	Set fSSet = fSTree.entrySet();
	Iterator it = fSSet.iterator();
	
	while(it.hasNext()) {
		Map.Entry me = (Map.Entry)it.next();
		System.out.println("	" + me.getKey());
	}
	
	System.out.println();
}

//NOT USED
public static FSNode getFSNode(long spaceReq){ 

	System.out.println("In getFSNode() of FileServerDirectory");
	Set fSSet = fSTree.entrySet();
	Iterator it = fSSet.iterator();

	while(it.hasNext()) {
		Map.Entry me = (Map.Entry)it.next();
//		System.out.print(me.getKey() + ":");
// 		System.out.println(me.getValue());
		FSNode fsNode = (FSNode)me.getValue();
		if(fsNode.spaceLeft >= spaceReq) return fsNode;
	}

 	System.out.println("Unable to find File Server to satisfy space request");
	System.out.println("Returning null");
	
	return null;

	}

/***
 * Postcondition: Iterates through the FileServerDirectory and returns the 
 * 								IP addresses of the first five FSs that can satisfy the 
 * 								space request, in a circular first fit manner.
 ***/

	public static String[] getReplServers(long size){
	
		long spaceReq = size;
		String[] replServers = new String[5]; //Replication server list

		Set fSSet = fSTree.entrySet();
		Iterator it = fSSet.iterator();

		FSNode fsNode; 
		int FSIndex = 0;
		while(FSIndex < 5) {
			Map.Entry me = (Map.Entry)it.next();
			fsNode = (FSNode)me.getValue();

			if((fsNode.spaceLeft >= spaceReq)&&(fsNode.fileServerIP.compareTo(prevReplServer)!=0)&&(fsNode.active)) {
				FSIndex +=1; 											//Got one FSNode
				//Reference - should work
				replServers[FSIndex-1] = fsNode.fileServerIP; 	
				//Reduce spaceLeft on FSNode	
			}

			//Since we iterate through the FSDirectory, this ensures that 
			//we return 5 different FSNode objects
			if(!it.hasNext() && FSIndex<5){ //reset the iterator
				System.out.println("Reached end of fSSet.Resetting iterator");
				it = fSSet.iterator();
			}
		}//end of while

		prevReplServer = new String(replServers[0]);
		System.out.println("Returning following FS Server IPs for file replication:");
		for(int i=0; i<5; i++)
			System.out.println("	" + replServers[i]);
		return replServers;
	}

}//end of class

