
package FM;

/***
*	Purpose: 	This static class takes any given data and generates a signature
* 					of the data using the FM's private key. In other words, it signs 
* 					data on behalf of the FM. The recipient of the signature can use 
* 					this for authenticating the FM and also for verifying the integrity 
* 					of the data.
***/

import java.io.*;
import java.security.*;

public class FMSign implements FileManagerConstants{

	private static PrivateKey priv;
	public static PublicKey publ;
	
	//Supports serialization. Can be saved to file.
	private static KeyPair KP;	
	
	private static Signature dsaSig;
	
	public FMSign(){

		//If KeyPair file exists, read KeyPair from there, else generate.

		boolean set = false;
					
		if(KEYFILE.exists()){ //KeyPair file exists-later
			try{
			FileInputStream fis = new FileInputStream(KEYFILE);
			ObjectInputStream ois = new ObjectInputStream(fis);
			KP = (KeyPair)ois.readObject();
			System.out.println("Got DSA KeyPair from file.");
			priv = KP.getPrivate();
			System.out.println("Initialized FM's private key.");
			publ = KP.getPublic();
			System.out.println("Initialized FM's public key.");
			set = true;
			}catch(Exception e){
				System.out.println("FMSign-Unable to read "+KEYFILE);
				System.out.println(e.getMessage());
				System.out.println("Going to generate new KeyPair");
				set=false;
			}

		}

		if(!set){ //KeyPair file does not exist
			System.out.println("Getting instance of KeyPairGenerator.");
			try{
				KeyPairGenerator KPG = KeyPairGenerator.getInstance("DSA", "SUN");
				SecureRandom r = new SecureRandom();
				//Permitted key strength in US is 1024
				System.out.println("Setting DSA public/private key strength to 1024.");
				KPG.initialize(1024, r);

				KP = KPG.generateKeyPair();
				System.out.println("Got DSA KeyPair.");
				priv = KP.getPrivate();
				System.out.println("Initialized FM's private key.");
				publ = KP.getPublic();
				System.out.println("Initialized FM's public key.");
			}catch(Exception e){
				System.out.println("FMSign()-Cannot make KeyPair-"+e.getMessage());
			}
		}
		
		System.out.println("Getting instance of Signature for SHA1withDSA.");
		try{
			dsaSig = Signature.getInstance("SHA1withDSA", "SUN");

			System.out.println("Initializing dsaSig with FM's private key.");
			dsaSig.initSign(priv);

			System.out.println("Ready to sign data.");	
		}catch(Exception e){
			System.out.println("FMSign()-Cannot make Signature-"+e.getMessage());
		}
		
	}

	/***
	 * in-parameter: 	the data that needs to be signed with FM's private key.
	 * out-parameter: the signature generated with the private key.
	 ***/
			
	public synchronized static byte[] sign(byte[] data, long timeStamp){

		try{
				ByteArrayOutputStream buf = new ByteArrayOutputStream();
				DataOutputStream out = new DataOutputStream(buf);

				dsaSig.update(data);
				
				byte[] ts = getBytes(timeStamp);
				dsaSig.update(ts);
				byte[] sig = dsaSig.sign();	//This also resets the dsaSig

				out.write(sig);
				out.flush();
				return buf.toByteArray();

		}catch(Exception e){
			System.out.println("FMSign sign() Exception-" + e.getMessage());
			return null;
		}
		
	}//end of sign()

	/***
	 * Postcondition: Returns FM's public key as a byte array, encoded in 
	 * 								X.509 format
	 ***/

	public static byte[] getPublicKeyEncoded(){
		System.out.println("Returning FM's public key, encoded in X.509 format.");
		return publ.getEncoded();
	}

	//Helper function
	private static byte[] getBytes(long x){

		byte[] arr = new byte[8];
		for(int i = 0; i<8; i++){
			arr[i] = (byte)(x >> ((7-i) * 8));
		}

		return arr;
	}

	/***
	 * Postcondition: Saves public and private keys to file keyFile.properties
	 ***/

	public static void writeToFile(){
		try{
		FileOutputStream fos = new FileOutputStream(KEYFILE);
		ObjectOutputStream oos = new ObjectOutputStream(fos);
		oos.writeObject(KP);
		oos.close();
		fos.close();
		}catch (Exception e){
			System.out.println("FMSign-Could not writeToFile()-"+e.getMessage());
		}
	}



} //end of class




