package FM;

/***
 * Purpose: The FM instantiates this class in a separate thread.
 * 					The run() method instantiates a CleanTask object 
 * 					every 24 hours.
 ***/ 

import java.util.Timer;
import java.util.TimerTask;
import java.util.*;
import java.io.*;
import java.net.*;

public class Cleaner implements FileManagerConstants{

	Timer timer;

	public Cleaner(){
		timer = new Timer();
		
		//Start after a day, and repeat once a day
		timer.schedule(new CleanTask(),86400*1000, 86400*1000);	
	}

	/***
	 * Purpose: This class checks for users who have not accessed
	 * 					files for more than 60 days and removes their files
	 * 					and accounts.
	 ***/

	class CleanTask extends TimerTask implements FileManagerConstants{

		public void run(){

		System.out.println("Running CleanTask.");

		long maxIdleTime = 5184000;	/*60 days in seconds*/

		//Get an iterator on OwnerDirectory
		Set fSet = OwnerDirectory.ownerTree.entrySet();
		Iterator it = fSet.iterator();
		while(it.hasNext()) {

			Map.Entry me = (Map.Entry)it.next();
			OwnerNode oNode = (OwnerNode)me.getValue();
			
			long currTime = System.currentTimeMillis();
			
			long diff = currTime - oNode.lastAccessed;
			long diffSec = diff/1000;

			/***	
			*	If OwnerNode has not been accessed in the last 60 days
			*	remove all files associated with this owner and remove
			*	the Owner's account (owmerNode)
			***/

			if(diffSec >= maxIdleTime){
				System.out.println("Removing owner with id: " + oNode.ownerId);
				try{
					removeOwner(oNode);
				}catch(Exception e){
					System.out.println("Could not remove oNode.");
				}
			}
		}

		}//end of run

	/***
	 * Purpose:				Remove an Owner some reason, say account is idle 
	 * 								for long.
	 * Postcondition: All files belonging to this owner on all FSs are 
	 * 								deleted. The owner's account (ownerNode) is 
	 * 								removed from the FM's OwnerDirectory.
	 ***/

	public boolean removeOwner(OwnerNode ownerNode) throws Exception{

		//Get an iterator on Owner's fDir	
		Set fSet = ownerNode.fDir.fileTree.entrySet();
		Iterator it = fSet.iterator();
		while(it.hasNext()) {

			Map.Entry me = (Map.Entry)it.next();
			FNode ofNode = (FNode)me.getValue();

			BriefFileDesc bfd = new BriefFileDesc(ofNode);

			FileManagerProtocol consumerFMP = new FileManagerProtocol((byte)RMFILE, bfd.byteEncode());
				
			byte[] fileInfo = consumerFMP.byteEncode();	

			//Send delete msgs to 5 FSs
			for(int fsIndex=0; fsIndex<5; fsIndex++){

				String fsIP = ofNode.FS[fsIndex];

    		OutputStream out;
    		DataOutputStream dos;
    		Socket fsClient;

    		try{
    			fsClient = new Socket(fsIP, FSPORT);
    		}catch(Exception ec){
    			System.out.println( "could not open connection to fsIP: " + fsIP +
    												" on port: " + FSPORT);
					continue;
    		}

    		out = fsClient.getOutputStream();
    		dos = new DataOutputStream(out);

				long timeStamp = System.currentTimeMillis();
				byte[] fmSig = FMSign.sign(fileInfo,timeStamp);
				System.out.println("Sending fileInfo to FS: " + fsIP);
				dos.write(fileInfo); 						//send FMP packet
				System.out.println("Sending timeStamp:"+timeStamp);
				dos.writeLong(timeStamp);				//New-to prevent replay attack
				System.out.println("Sending sigLen:"+fmSig.length);
				dos.writeInt(fmSig.length);			//New-length of sig of data+timeStamp
				System.out.println("Sending fmSig.");
				dos.write(fmSig);								//New-sig of data+timeStamp
				dos.close();

			}//end of for

		ownerNode.fDir.removeNode(ofNode.fileId);		
		System.out.println("Removed an FNode.");

		}//end of while 	

		System.out.println("Removing ownerNode from OwnerDirectory.");
		OwnerDirectory.removeNode(ownerNode.ownerId);
		System.out.println("Done.");

		return true;


	}//end of removeOwner()

	}//end of CleanTask

}//end of Cleaner
