
package DT;

/***
 * Purpose: Five RetrieveFS objects are instantiated by the downloadAndSend()
 *          method of DeskTop. 
 *          Each RetrieveFS object retrieves blocks in its series(say, 0,5,10...)
 *          Each RetrieveFS object tries to keep a stock of 5 blocks ready for
 *          consumption
 *
 *          This class is used on the DT and Proxy, not the FM
 ***/

import java.io.*;
import java.util.*;
import java.net.*;

//Implements a producer-consumer relationship
public class RetrieveFS extends Thread implements FileManagerConstants{

	FNode fn;
	
  //mySeries would be 1,2,3,4 or 5
	//Also helps in deciding which FS to try first
	int mySeries;		

	Socket fsClient;
	byte[][] blocks;

	//Array parallel to blocks[] - empty implies 
	//that the block has been consumed
	boolean[] empty;	
	int[] realSize;		//We may end up with a block smaller than
										//blockUnit.

	int toRead;				//index of block that must be consumed next
	int lastBlockPut; //index of the last block produced

	//milliseconds to block on read, before the connection is broken
	final int TIMEOUT = 1000;

	//Flag to indicate the retrieval is finished
	boolean flagAllOut;	
	int FSIndex;

	RetrieveFS(String threadName, ThreadGroup tg, FNode fn, int seriesNum){
		super(tg, threadName);
		this.fn = fn;
		mySeries = seriesNum; 

		//At any point, can store max 5 blocks
		blocks = new byte[5][(int)fn.blockUnit];
		empty = new boolean[5];
		realSize = new int[5];

		for(int i=0; i<5; i++) empty[i] = true;
		for(int i=0; i<5; i++) realSize[i] = (int)this.fn.blockUnit;
		
		toRead = 0;
		lastBlockPut = -1;
		flagAllOut=false;	//not yet all-out
		this.start();		//Start the thread
	}


	/***
	 * Retrieve and store 5 blocks in my series.
	 * Replace blocks as they are consumed.
	 * Close the connection when all the blocks my series are read from FS
	 * If you cannot get the blocks from your FS, try the next FS
	 ***/
	
	public void run(){

	//Need a synchronized putBlock? Yes
	//keep calling putBlock() :-)
	//Attempt to connect to the FS according to mySeries
	//FSIndex will be 0-4 for mySeries 1-5
	
	FSIndex = mySeries-1; //Try to connect to mySeries FS first

	Socket fsClient;
	OutputStream os ;
	InputStream is;
	
	while(true){
		try{
			System.out.println("RetrieveFS: series:"+mySeries+" trying:" + fn.FS[FSIndex]);

			fsClient = new Socket(fn.FS[FSIndex], FSPORT);
			os = fsClient.getOutputStream();
			is = fsClient.getInputStream();
			//The maximum time-limit to block on a read
			fsClient.setSoTimeout(TIMEOUT);
			break;					//jump out if you get a connection

		}catch(Exception e){
			//try the next FS
			System.out.println(	"RetrieveFS: series:"+mySeries+" :(Exception):" 
													+ e.getMessage()+ ":" + fn.FS[FSIndex]);

			FSIndex = (FSIndex+1)%5;
			if((FSIndex+1) == mySeries){
				//implies, unsuccessfully tried all 5 FSs
				System.out.println("\nCouldn't connect to ANY of the replication"
														+ " File Servers\n");
				this.putBlock(null, true/*the end*/,0);
				return;				//break out of the function
			}
		}
	}//end of while


	DataOutputStream dos = new DataOutputStream(os);
	DataInputStream dis = new DataInputStream(is);

	byte[] blockBytes = new byte[(int)fn.blockUnit];

	//fileId + blockIndex + blockSig
	RequestArray reqArray = new RequestArray(fn);	

	int blockIndex;

	boolean available=true;
	//Request a new block from mySeries each time
	for(blockIndex=mySeries; blockIndex<=fn.noBlocks; blockIndex+=5)	
	{	
		try{
			byte[] req = reqArray.byteEncode(blockIndex);
			FileManagerProtocol fmProto = new FileManagerProtocol((byte)BLOCK_REQ, req);
			byte[] encFMProto = fmProto.byteEncode();
		
			//write this to output stream
			dos.write(encFMProto);
		}catch(Exception e){ 
			System.out.println("Exception while trying to send FileManagerProtocol"
													+ " e: blockIndex: " + blockIndex
													+ " e: fn.noBlocks: " + fn.noBlocks
													+ " e: mySeries: " + mySeries
													+	" e: msg: " + e.getMessage());
		}
		
		//read a block from the input stream and putBlock()

		int numBytes = -1;
		int totBytes = -1;
		try{//blocking call

			//What if you cannot get all the bytes in one read?
			//LATER...Done
			try{
				numBytes = dis.read(blockBytes);
			}catch(Exception ee){
				available=false;
			}
			totBytes = numBytes;

			while(totBytes < (int)fn.blockUnit){
				if(!available) break;
				if(totBytes==-1) {
					available=false;
					break;
				}

				if(numBytes == -1){
					System.out.println("No more bytes");
					available=false;
					break;
				}

					try{
							numBytes = dis.read(blockBytes,totBytes,((int)fn.blockUnit-totBytes));
					}catch(IOException eee){
						available=false;
					}
				if(numBytes != -1 && available) totBytes += numBytes;
			}



			if(totBytes != -1)
				this.putBlock(blockBytes, false/*not the end*/,totBytes);	

			if(!available) break;

		}catch(Exception e){
			System.out.println("***Exception: Did not receive expected block from FS");
			System.out.println(e.getMessage());
			try{//try to close streams
				dis.close();
				dos.close();
			}catch(Exception ee){
				System.out.println("Could not close i/o streams with FS");
				System.out.println(ee.getMessage());
			}
			break; 	//Assume this FS is of no further use for this transaction

		}
	}	
	
	//Check if we have reached the greatest possible blockIndex for 
	//this file
	if((blockIndex + 5)<= fn.noBlocks){
		System.out.println("Did not retrieve all blocks in my series.");
		System.out.println("Trying to resume with another FS...");
		//Do something...
	}
	else{
		//Successfully retrieved all blocks in my series. Terminate.
		//One more putBlock with 'end=true'
		try{
			System.out.println("\nSuccessfully retrieved all blocks in series: " 
													+ mySeries + "\n");
		
			this.putBlock(blockBytes, true/*the end*/,0);
			dis.close();
			dos.close();
		}catch(Exception e){
			System.out.println("Could not putBlock() with end=true");
			System.out.println(e.getMessage());
			return;	
		}
	}
				
} //end of run()

  /***
	 * Precondition:  if end is false, inBlock contains valid data.
	 * Postcondition: if buffer if full (has 5 blocks), waits for a block
	 *                to be consumed. Puts inBlock in buffer.
	 ***/

	public synchronized void putBlock(byte[] inBlock, boolean end, int rSize){
		//rSize is the actual size of the block, may be less than blockUnit
		//the last block

		while(empty[((lastBlockPut+1)%5)] == false){
			try {
				wait();
			} catch (InterruptedException e){
				System.out.println("putBlock: Exception-" + e.getMessage());
			}
		}
		lastBlockPut = (lastBlockPut+1)%5;

		if(end){
			//Termination condition
			empty[lastBlockPut]=false; 			//now available
			realSize[lastBlockPut] = rSize;
			blocks[lastBlockPut] = null;
			notifyAll();
		}
		else{
		System.arraycopy(inBlock, 0, blocks[lastBlockPut], 0, rSize);
		empty[lastBlockPut]=false; 			//now available
		realSize[lastBlockPut] = rSize;
		notifyAll();
		}

	}//end of putBlock

/***
 * Postcondition: gets a block according to index toRead from the
 *                buffer. returns null if there are no more blocks.
 *                Waits if buffer is empty, but more blocks are
 *                expected.
 ***/
public synchronized byte[] getBlock() {

	while(empty[toRead] == true){
		try {
			wait();
		} catch (InterruptedException e){
			System.out.println("getBlock Exception1: " + e.getMessage());
		}
	}	
	
	//terminating condition
	if(blocks[toRead]==null){
		notifyAll();
		System.out.println("No more blocks to get: returning null");
		return null;
	}	

		empty[toRead] = true;					//block consumed
		int oldRead = toRead;
		toRead = (toRead+1)%5;				//the next toRead

		//Want to avoid unnecessary copying of buffers
		if(realSize[oldRead] < (int)fn.blockUnit){
			System.out.println("get() of RetrieveFS. Returning smaller block of "
													+ "size: " + realSize[oldRead]);
			byte[] smallerBlock = new byte[realSize[oldRead]];
			System.arraycopy(blocks[oldRead],0,smallerBlock,0, realSize[oldRead]);

			notifyAll();
			return smallerBlock;

		}
		else{
			byte[] tBlock = new byte[(int)fn.blockUnit];
			System.arraycopy(blocks[oldRead],0,tBlock,0, (int)fn.blockUnit);
			notifyAll();
			return tBlock;
		}

	}

}//End of class

/***
 * Purpose: RequestArray is a helper class to RetrieveFS. RequestArray
 *          encodes a request for a block that is sent to FS.
 *          The request is encoded as fileId + blockIndex + blockSig
 *
 ***/

class RequestArray implements FileManagerConstants{
				
	FNode myFNode;

	String fId;
	int blockIndex;
	byte[] bSig;		//the block signature in the request
									//No space allocated yet.

	public RequestArray(FNode fileN){
		try{
			myFNode = fileN;
		}catch(Exception e){
			System.out.println("Exception in RequestArray constructor: " 
										   + e.getMessage());
		}
	}

	//Empty constructor. Data members will be filled in by byteDecode
	public RequestArray(){
		bSig = new byte[16];		//sigSize is fixed at 16 bytes
	}
	
	//Encodes request as fileId + bIndex + blockSig
	public byte[] byteEncode(int bIndex) throws Exception{

		ByteArrayOutputStream buf = new ByteArrayOutputStream();
		DataOutputStream out = new DataOutputStream(buf);

		byte[] idArr = myFNode.fileId.getBytes(DEF_ENCODING);

		out.writeInt(idArr.length);
		out.write(idArr);
		out.writeInt(bIndex);
		byte[] sig = new byte[myFNode.sigSize];
		System.arraycopy(myFNode.blockSig[bIndex-1], 0, sig, 0, myFNode.sigSize);
		out.write(sig);
		out.flush();
		return buf.toByteArray();

	}
	
	//decodes RequestArray packet. Used on FS, not on Proxy or DT.
	public void byteDecode(byte[] data) throws Exception{

  	ByteArrayInputStream is = new ByteArrayInputStream(data);
	  DataInputStream dis = new DataInputStream(is);

		int fIdLen = dis.readInt();
		byte[] idArr = new byte[fIdLen];
		dis.readFully(idArr);

		fId = new String(idArr, DEF_ENCODING); 	//fileId
		blockIndex = dis.readInt();							//Block Index
		dis.readFully(bSig);		//block signature

		dis.close();

	}

}


