
package DT;

/***
 * Purpose: The OwnerProtocol class is used to exchange messages between the
 *          FM and DT, FM and HH, and FM and Proxy. OwnerProtocol has data
 *          members and methods for coding/decoding OwnerProtocol messages,
 *          authenticating the user, and acting on the messages received
 *          from FM. The FM_Owner class is the driver (controls the
 *          methods of this class) for this class.
 *
 *          Messages:   1.  Submit File
 *                      2.  Request File
 *                      5.  Request for File Deletion
 *                      6.  Request for FSListing and Block Sigs
 ***/

import java.net.*;
import java.io.*;
import java.util.*;
import java.security.*;

public class OwnerProtocolDT implements FileManagerConstants{

	byte protocol = 2;	/*2 for FM-DT protocol, 3 for FM-HH*/
	byte msgType = 0;

	String password;
	Random pRandom;

	FileDescription fDesc; 	//submit
	String fileId;					//request

	//for submit File 
	public OwnerProtocolDT(String pWord, FileDescription fDesc, byte msgType){

		this.fDesc = fDesc;
		this.msgType = msgType;
		this.password = pWord;	//Will be scrambled in Encode()
		this.pRandom = new Random(System.currentTimeMillis());
	}

	//for File Request (FS list from FM)
	public OwnerProtocolDT(String pWord, String fileId, byte msgType){

		this.fileId = fileId;
		this.msgType = msgType;
		this.password = pWord;	//Will be scrambled in Encode()
		this.pRandom = new Random(System.currentTimeMillis());
	}
	
	//Requesting File Listing
	public OwnerProtocolDT(String pWord, byte msgType){
		
		this.msgType = msgType;
		this.password = pWord;	//Will be scrambled in Encode()
		this.pRandom = new Random(System.currentTimeMillis());
	}

	//Other messages
	public OwnerProtocolDT(byte msgType){}

	/***
	 * Precondition: 	protocol, msgType, password and pRandom are initialized.
	 * 								Depending on the msgType, other data members may
	 * 								also be required to be initialized.
	 * Postcondition: Returns a byte encoded array of the OwnerProtocol message
	 * 								that can be sent to FM.
	 ***/

	public byte[] byteEncode()throws Exception{
		
    ByteArrayOutputStream buf = new ByteArrayOutputStream();
    DataOutputStream out = new DataOutputStream(buf);


		out.writeByte(protocol);
		out.writeByte(msgType);
		
		//writePass() method writes the password digest stuff to FM
		writePass(out,this.password);

		switch(msgType){
			case 1: {//submit file
							byte [] fDescArr = fDesc.byteEncode();
							out.writeInt(fDescArr.length);
							out.write(fDescArr);
							break;
							}	
			case 2:	{//request FS list from FM
							byte[] fIdArr = this.fileId.getBytes(DEF_ENCODING);
							out.writeInt(fIdArr.length);
							out.write(fIdArr);
							break;
							}
			case 3:{//Request file listing.
							//No data to be sent

							break;
						 }

			case 5: {//Delete file from Diskster
							byte [] fDescArr = fDesc.byteEncode();
							out.writeInt(fDescArr.length);
							out.write(fDescArr);
							break;
							}
			default:
			}

		 out.flush();
     return buf.toByteArray();

		}

	/***
	 * Precondition: 	writePass() is passed the current data output stream 
	 * 								and password.
	 * Postcondition:	Writes the current timestamp, a random number
	 * 								and the SHA-1 digest of the timestamp, random number
	 * 								and password to the data output stream. 
	 * 								Does not close the output stream
	 ***/

	public void writePass(DataOutputStream pdos, String pWord) throws Exception{
		
		long timeStamp = System.currentTimeMillis();
		long randomNumber	=	pRandom.nextLong();

		MessageDigest md = MessageDigest.getInstance("SHA-1");
		
		byte[] tsArr = getBytes(timeStamp);
		byte[] rmArr = getBytes(randomNumber);
		byte[] pArr = pWord.getBytes(DEF_ENCODING);

		md.update(tsArr);
		md.update(rmArr);
		md.update(pArr);

		byte[] pSetDigest = md.digest();

		//timestamp
		pdos.writeLong(timeStamp);
		//random number
		pdos.writeLong(randomNumber);
		//the digest of the above two along with the password
		pdos.writeInt(pSetDigest.length);	//digest length, in case we use
																			//another digest later
		pdos.write(pSetDigest);

	}

	/***
	 * The following helper method is from the book Wireless Java:
	 * Developing with Java[tm] 2 Micro Edition, by Jonathan Knudsen
	 * Postcondition: Returns a long data type as a byte array.
	 ***/

	private byte[] getBytes(long x){

		byte[] arr = new byte[8];
		for(int i = 0; i<8; i++){
			arr[i] = (byte)(x >> ((7-i) * 8));
		}
		return arr;
	}

}//end of class


