package DT;

/***
 * Purpose: This file defines 2 classes, FMInfo and DeskTop. DeskTop contains 
 * 					the main() method and is the main driver class of package DT while
 * 					FMInfo is a helper class containing data and methods to connect
 * 					to FM.
 ***/

import java.io.*;
import java.net.*;
import javax.net.ssl.*;		//for https connection
import java.security.*;		//for MessageDigest

  /***
	 * Purpose: FMInfo contains data and methods to connect to FM using
	 *          https.
	 ***/

	class FMInfo{

		public static URL fmURL;

		public static void init(){
		try{
			//Required for secure https connection
			System.setProperty("java.protocol.handler.pkgs","com.sun.net.ssl.internal.www.protocol");
			Security.addProvider(new com.sun.net.ssl.internal.ssl.Provider());

			fmURL = new URL("https", "ep.netlab.uky.edu", 8443, "/servlet/FM.FM");
		}catch(MalformedURLException m){
			System.out.println("Malformed URL");
		}
		
		}

	}//end of class FMInfo

/***
 * Purpose: DeskTop presents the user interface to the user and 
 * 					contains data members and methods to submit files to 
 * 					to FM, delete files from FM, refresh DT state and save
 * 					DT state to file.
 ***/

public class DeskTop implements FileManagerConstants{

	OutputStream 				outFMStream; //output stream to FM
	InputStream					inFMStream;	 //input stream from FM
	FMInfo fmi;
	FileDirectory fileDirectory;
	HttpURLConnection connection;

	OwnerInfo oi;

	public static void main(String args[])throws Exception{
		
		DeskTop dt = new DeskTop();
		dt.fmi = new FMInfo();
		dt.fileDirectory = new FileDirectory();

		FMInfo.init();

		/***
		 * Try to initialize OwnerInfo from saved state. Initialize
		 * OwnerInfo interactively with user only if no previous
		 * OwnerInfo is available from file owner.properties.
		 ***/
		
		File ownerFile = new File("owner.properties");
		if(ownerFile.exists()){
			//Get OwnerInfo from file (saved state)
			FileInputStream fis = new FileInputStream(ownerFile);
			ObjectInputStream ois = new ObjectInputStream(fis);
			System.out.println("Initializing OwnerInfo from saved state.");
			dt.oi = (OwnerInfo)ois.readObject();		
		}
		else //get OwnerInfo from user	
			dt.oi = new OwnerInfo();

		//save state
		System.out.println("Saving OwnerInfo state to file...");
		dt.saveOwnerInfo(dt.oi);


		//Present text-based user interface to user

		//For keyboard input
		InputStreamReader isr = new InputStreamReader(System.in);
		StreamTokenizer st = new StreamTokenizer(isr);
		int choiceType = -1;
		String choice;
		int choiceNum = -1;
		
		while(choiceNum != 5){

		System.out.println("\nFiles submitted to Diskster:\n");
		try{	
			String[] fList = FileDirectory.getFileNames();
			for(int fIndex=0;fIndex<FileDirectory.numFiles; fIndex++){
				System.out.println("	" + fList[fIndex]);
		}
			System.out.println("");
		}catch(Exception ee){
			System.out.println(ee.getMessage());
			System.exit(1);
		}	

		System.out.println("1: Submit a file to Diskster.");
		System.out.println("2: Request a file from Diskster.");
   	System.out.println("3: Delete a file from Diskster.");
   	System.out.println("4: Refresh state information from FM");
		System.out.println("5: Shutdown");

		System.out.println("Enter choice:");
		
		choiceType = st.nextToken();
		if(choiceType != StreamTokenizer.TT_NUMBER) continue;
		
		choiceNum = (int)st.nval;	
		
		switch(choiceNum){
			case 1: { //file submit
							System.out.println("File Submission.");
							System.out.print("Enter name of file: ");
							choiceType = st.nextToken();		
							while(choiceType != StreamTokenizer.TT_WORD){
								System.out.println("Not a valid filename.");
								System.out.print("Enter name of file: ");
								choiceType = st.nextToken();		
							}
							choice = new String(st.sval);

							long sTime = System.currentTimeMillis();
							dt.fmConnect();		
							String fId = dt.submitFile(choice);
							dt.placeBoInput(fId);
							long eTime = System.currentTimeMillis();
							System.out.println("Time Taken: " 
																	+ (float)((eTime-sTime)/(float)1000) 
                         					+ " second(s).");
							System.out.println("Saving File Directory state to file...");
							//save state
							FileDirectory.writeToFile();
							break;
							}
			case 2:	{	//file request
							System.out.println("File Request.");
							System.out.print("Enter name of file: ");
							choiceType = st.nextToken();		
							while(choiceType != StreamTokenizer.TT_WORD){
								System.out.println("Not a valid filename.");
								System.out.print("Enter name of file: ");
								choiceType = st.nextToken();		
							}
							choice = new String(st.sval);
							long sTime = System.currentTimeMillis();
							dt.fmConnect();		
							String fileId = dt.requestFile(choice);
							dt.receiveFile(fileId);
							long eTime = System.currentTimeMillis();
							System.out.println("Time Taken: " 
																	+ (float)((eTime-sTime)/(float)1000) 
                         					+ " seconds.");
							System.out.println("Saving File Directory state to file...");
							//save state
							FileDirectory.writeToFile();
							break;
							}
			case 3: {	//file deletion request
							System.out.println("Request for file DELETION.");
							System.out.print("Enter name of file: ");
							choiceType = st.nextToken();		
							while(choiceType != StreamTokenizer.TT_WORD){
								System.out.println("Not a valid filename.");
								System.out.print("Enter name of file: ");
								choiceType = st.nextToken();		
							}
							choice = new String(st.sval);
							long sTime = System.currentTimeMillis();
							dt.fmConnect();		
							String fileId = dt.deleteFile(choice);
							dt.receiveDeleteConf(fileId);
							long eTime = System.currentTimeMillis();
							System.out.println("Time Taken: " 
																	+ (float)((eTime-sTime)/(float)1000) 
                         					+ " seconds.");
							System.out.println("Saving File Directory state to file...");
							//save state
							FileDirectory.writeToFile();
							break;
							}

			case 4: {	//refresh state request
							System.out.println("Refresh state from FM.");
							long sTime = System.currentTimeMillis();
							dt.fmConnect();		
							dt.fmRequestFileListing();
							dt.fmReceiveFileListing();
							long eTime = System.currentTimeMillis();
							System.out.println("Time Taken: " 
																	+ (float)((eTime-sTime)/(float)1000) 
                         					+ " seconds.");
							System.out.println("Saving File Directory state to file...");
							//save state
							FileDirectory.writeToFile();
							break;
							}
			case 5: {	//shutdown
							break;
							}

			default: System.out.println("Wrong Choice. Please choose again.");

		}	//end of switch

		}//end of while

		System.out.println("Saving OwnerInfo state to file...");
		//save state
		dt.saveOwnerInfo(dt.oi);

		//save state
		System.out.println("Saving File Directory state to file...");
		FileDirectory.writeToFile();
		System.out.println("Exiting from Diskster...");
		

	}//end of main

	//Method that establishes https connection to FM.
	public void fmConnect() {
		try{
		System.out.println("Connecting to: " + fmi.fmURL);

		connection = (HttpURLConnection)((FMInfo.fmURL).openConnection());
		connection.setDoOutput(true);
		connection.setRequestMethod("POST");
		connection.connect();
		outFMStream = new BufferedOutputStream(connection.getOutputStream());	

		}catch(Exception e){
			System.out.println("Could not connect to File Manager: "+fmi.fmURL);
			System.out.println("Exiting Diskster...");
			System.exit(1);
		}

	}
	
	//Started out as a placebo method to read FM response. Now
	//acts on authentication-failed response from FM.
	public void placeBoInput(String fId){

		try{
			InputStream inFMStream = connection.getInputStream();
			DataInputStream dis = new DataInputStream(inFMStream);
				
			byte res = dis.readByte();	
			if(res==(byte)0){
				System.out.println("OwnerId or Password wrong.");
				System.out.println("Could NOT submit file to Diskster.");
				if(fId != null)
					FileDirectory.removeNode(fId);
			}
		  while(true){
				try{
      		res = dis.readByte();
				}catch(Exception e){
					System.out.println("No more bytes to read from FM.");
					break;
				}
			}

			dis.close();
			
		}catch(Exception w){
		}
	}

	/***
	* Postcondition: 	Submits a file to FM using HTTPS post.	Returns fileId
	* 								on success, null on failure.
	***/
	public String submitFile(String fName) throws Exception{
	
	File file = new File(fName);
	if(file.length()>96000){
		System.out.println("ERROR: The file you are trying to submit is of size: "
												+ (file.length()/1000)	+ "KB.");
		System.out.println("The maximum permitted size of a single file is 96 KB.");
		System.out.println("Please split the file and try again.");
		return null;

	}

		FileInputStream inFile = null;

		//By default assume, new file.
		boolean oldFile = false;

		//Used for getting the signature of the block
		MessageDigest md = null;
		try{
			md = MessageDigest.getInstance("SHA-1");
		}catch(Exception w){
			System.out.println("submitFile Exception-"+w.getMessage());
			return null;
		}
	  try{
			inFile = new FileInputStream(fName);
		}
		catch(FileNotFoundException fnf){
						System.out.println("File Not Found");
						return null;
		}	

		//Make an FNode and enter into FileDirectory.
		//Put the fileId into the FileDescription
		
		FNode fNode = null;
		try{
		 fNode = new FNode(fName, oi);
		}catch(Exception w){
			System.out.println("submitFile Exception 2-"+w.getMessage());
		}
		FNode ofNode = null;
		try{
			ofNode = new FNode(fName, oi);	//Initialized only for oldFile
		}catch(Exception w){
			System.out.println("submitFile Exception 3-"+w.getMessage());
		}
		if(!FileDirectory.addNode(fNode)){
			oldFile = true;
			System.out.println("fName: " + fName + " already in FileDirectory");
			System.out.println("Retrieving old FNode.");
			ofNode = FileDirectory.getNode(fNode.fileId);
		}


		try{
			outFMStream.write((byte)oi.idLength);
			outFMStream.write(oi.idArr);	//the ownerId
		}catch(Exception w){
			System.out.println("submitFile Exception 4-"+w.getMessage());
		}
		//Create an instance of Encryptor.
		Encryptor eTor = new Encryptor((oi.DESKey).getBytes());

		//We need the inBlockSize and the outputSize to make the 
		//FileDescription also
		
		int inBlockSize = eTor.blockSize;
		byte[] inBlock = new byte[inBlockSize];
		
	
		FileDescription fd = new FileDescription(fName,oi);

		//Calculates max possible size of the encrypted file and adds to 
		//FileDescription
		//IMPORTANT: Functions added later - dont forget to update everywhere

		fd.putId(fNode);
		fd.putEncFileSize(inBlockSize,eTor.outputSize);

		//OwnerProtocolDT contains the passwd

		//Pass the password entered by the user	
		OwnerProtocolDT oProt = new OwnerProtocolDT(oi.password, fd,(byte)1);

		DataOutputStream ndis = null;
		int lenOwnerProt = 0;
		try{	
			byte[] oProtArr = oProt.byteEncode();	
			ndis = new DataOutputStream(outFMStream);
			lenOwnerProt = oProtArr.length;
			if(!(lenOwnerProt > 0)){
				System.out.println("Invalid Owner Protocol. Quitting...");
				return null;
			}
		ndis.writeInt(lenOwnerProt);
		ndis.flush();
		outFMStream.write(oProtArr);
		}catch(Exception w){
			System.out.println("submitFile Exception 6-"+w.getMessage());
		}

		//read the file and output the blocks

		try{
		int inL;
		int outL;
		
		System.out.println("Writing File blocks");

		boolean sentSigSize = false;
		int blockIndex = 0;
		int totBytes=0;
		while((inL = inFile.read(inBlock, 0, inBlockSize)) > 0){

			totBytes=inL;
			while(totBytes<inBlockSize){
				if(inL==-1){
 					System.out.println("read returned -1. Breaking..."); 
         	break;
				}
				inL=inFile.read(inBlock, totBytes, inBlockSize-totBytes);
				if(inL != -1) totBytes += inL;	
			}
			
				//Debugging...
				if((totBytes != inBlockSize) && (inL != -1)){
					System.out.println(	"WARNING: totBytes: " + totBytes + " inBlockSize: "
															+inBlockSize + " inL: " + inL);
					System.exit(1);
				}

			byte[] outBlock2 = eTor.encrypt(inBlock,totBytes);

			//BLOCK SIGNATURE  - digest is automatically reset
			md.update(outBlock2);
			byte[] blockSig = md.digest();

			//store the length of the signature once and also transmit it once.
			if(!sentSigSize){
				sentSigSize = true;
				fNode.setBlockInfo(blockSig.length, fd);
				outFMStream.flush();
				ndis.writeInt(blockSig.length);
				ndis.flush();
			}

			
			//Store block signature locally, used for verification later
			//and increment block index
			System.arraycopy(blockSig, 0, fNode.blockSig[blockIndex++], 0, blockSig.length);

			//Send the block and then send the block signature
			//Note that the signature itself is not encrypted.
			if(!oldFile){	//new file
			outFMStream.write(blockSig, 0, blockSig.length);
			outFMStream.flush();
			System.out.println("Sending block. Index: " + (blockIndex-1)
													+ " Length: " + outBlock2.length);	
			outFMStream.write(outBlock2, 0, outBlock2.length);
			outFMStream.flush();

			}else{	//old file
				//resend only changed blocks
				if(compareSHA(ofNode.blockSig[blockIndex-1],blockSig) !=0){ //different
					System.out.println(	"Sending CHANGED block. Index: " 
															+ (blockIndex-1));
					outFMStream.write(blockSig, 0, blockSig.length);
					outFMStream.flush();
					ndis.writeInt((blockIndex-1));	
					ndis.flush();
					outFMStream.write(outBlock2, 0, outBlock2.length);
				}
				else{
					System.out.println("Block " + (blockIndex-1) + " has not changed.");
				}
			}//end of else

		}
		inFile.close();

		fNode.setNumBlocks(blockIndex); 	//Total number of encrypted blocks
		System.out.println("Total number of encrypted blocks: " + blockIndex);

		if(oldFile){		
			System.out.println("oldFile: Calling FileDirectory.updateNode()");
			if(!FileDirectory.updateNode(fNode)){
				System.out.println("FileDirectory.updateNode(fNode) returned false");
			}
		}
		outFMStream.flush();
		outFMStream.close();

		}
		catch(IOException ioe){
		      ioe.printStackTrace(); 
		}

	return fNode.fileId;

	}//end of submitFile	

  /***
	*	Precondition: 	User has chosen to refresh state on DT. 
	*	Postcondition:	Sends a request for refreshing state to FM.
	***/
	public void fmRequestFileListing(){

			System.out.println("Going to request File Listing");
		try{

			DataOutputStream ndis = new DataOutputStream(outFMStream);

			ndis.write((byte)oi.idLength);
			ndis.write(oi.idArr); //the ownerId

			//OwnerProtocolHH contains the passwd
			
			OwnerProtocolDT oProt = new OwnerProtocolDT(oi.password,(byte)3); 	
			byte[] oProtArr = oProt.byteEncode();
			int oLen = oProtArr.length;
			ndis.writeInt(oLen);
			try{
				ndis.flush();
			} catch(Exception ed){
				System.out.println(ed.getMessage());
			}
			
			ndis.write(oProtArr);

			ndis.flush();
			ndis.close();
			outFMStream.close();

		}catch (Exception e){
			System.out.println("4.could not open output stream");
		}

	}


	/***
	*	Precondition:	This function is called only when the user wants 
	*								to refresh the FileDirectory state.
	*
	*	Postcondition:Refreshes FileDirectory from the FM
	***/

	public void fmReceiveFileListing(){

		try{

			inFMStream = connection.getInputStream();	

			DataInputStream inDIS = new DataInputStream(inFMStream);
			int FLLen;				//length of File Listing	
			try{

				FLLen = inDIS.readInt();
				byte[] fListingArr = new byte[FLLen];
				inDIS.readFully(fListingArr);
				inFMStream.close();	//close input stream from FM

				boolean available = true;
			
				ByteArrayInputStream bais = new ByteArrayInputStream(fListingArr);
				DataInputStream badis = new DataInputStream(bais);
				int bFNodeLen = 0;
				while(available){
					//decode BriefFNode(s)
					try{
							bFNodeLen = badis.readInt();
							byte[] bFNodeArr = new byte[bFNodeLen];
							badis.readFully(bFNodeArr);	
							//Make a BriefFNode and add to FileDirectory
							BriefFNode bFN = new BriefFNode();
							bFN.byteDecode(bFNodeArr);
							FNode fn = new FNode(bFN);
							if(!FileDirectory.addNode(fn))
									FileDirectory.updateNode(fn);
					}
					catch(Exception e){
						available = false;
						System.out.println("No more BriefFNode(s)");
					}
				}//end of available

			}catch(Exception e){
				System.out.println("Unable to retrieve file listing");
				return;
			}
	
		}catch (Exception e){
			System.out.println("1.Could not open input stream");
			return;
		}

	}//end of fmReceiveListing()


	/***
	*	Purpose: 	Compares 32 bits of SHA-1 signature sig2 to sig1
	*						Used for avoiding re-submissions of unchanged 
	*						blocks.
	*	Postcondition: returns 0 if sig2 matches sig1, -1 otherwise
	***/
	public int compareSHA(byte[] sig1, byte[] sig2){
		for(int sigByte=0; sigByte<8; sigByte++)
			if(sig2[sigByte] != sig1[sigByte]) return -1;	

		return 0;
	}

	/***
	*	Purpose: 	Compares 160 bits of SHA-1 signature sig2 to sig1
	*						Used for verifying sigs of retrieved blocks
	*	Postcondition: returns 0 if sig2 matches sig1, -1 otherwise
	***/
	public int compareSigs(byte[] sig1, byte[] sig2){
		try{
			for(int sigByte=0; sigByte<20; sigByte++)
				if(sig2[sigByte] != sig1[sigByte]) return -1;	
		}catch(Exception e){
			System.out.println("compareSig()-Exception: " + e.getMessage());
			return -1;
		}
		return 0;
	}

	//Debugging - prints out the byte values in the block
	public void displayBlock(byte[] block){
		System.out.println("Block contains following bytes:");

		for(int i=0; i<block.length; i++){
			System.out.print(block[i]+" ");
		}
	}

	/***
	 * Precondition: 	User has requested a file from Diskster.
	 * Postcondition:	Request for FS list sent to FM. 
	 * 								Returns fileId which can be passed to receiveFile()
	 ***/
	
	public String requestFile(String fName) throws Exception{
		
		//fName needs to be mapped to the fileId sending the request
		String fileId;

		//Use SHA-1 for calculating fileId
		MessageDigest md = MessageDigest.getInstance("SHA-1");
		md.update(fName.getBytes(DEF_ENCODING));
		md.update(oi.ownerId.getBytes(DEF_ENCODING));
		byte digest[]  = md.digest();
		fileId= new String(digest, DEF_ENCODING);

		outFMStream.write((byte)oi.idLength);

		outFMStream.write(oi.idArr);	//the ownerId

		//OwnerProtocolDT contains the passwd

		//Pass the password entered by the user	
		OwnerProtocolDT oProt = new OwnerProtocolDT(oi.password, fileId ,(byte)2);

		byte[] oProtArr = oProt.byteEncode();	

		DataOutputStream ndis = new DataOutputStream(outFMStream);

		int lenOwnerProt = 0;
		lenOwnerProt = oProtArr.length;
		if(!(lenOwnerProt > 0)){
			System.out.println("Invalid Owner Protocol. Quitting...");
			return null;
		}
		ndis.writeInt(lenOwnerProt);
		ndis.flush();
		outFMStream.write(oProtArr);
		outFMStream.flush();
		outFMStream.close();
		
		//Sent request for FS list to FM
					
		return fileId;	//Will be passed to receiveFile to save computation.
					
	}

	/***
	 * Purpose:	receives FS list from FM and then call downloadAndSend()
	 * Precondition: inFMStream is initialized
	 * Postcondition: calls downloadAndSend()
	 ***/
	public void receiveFile(String fileId){

		if(fileId == null){
			System.out.println("receiveFile(): fileId is null");
			return;
		}	

		String[] FSList = new String[5]; //FS IP addresses

		try{
		inFMStream = connection.getInputStream();	
		DataInputStream dis = new DataInputStream(inFMStream);
		int fsTotLength = dis.readInt();
		byte[] fsListArr = new byte[fsTotLength];
		dis.readFully(fsListArr);
		FSList = decodeFSList(fsListArr);  //Reads and initializes IP addresses
		}catch(Exception e){
			System.out.println("Could not read FS list from FM");
			System.out.println(e.getMessage());
		}

		//Now that we have the FS list, we add this info to the FNode

		try{
			FNode fn = (FNode)FileDirectory.getNode(fileId);
			fn.setFSList(FSList);
					
			int attempt = 0;
			for(attempt=0; attempt<5 ; attempt++){			
				
				if(downloadAndSend(fn)){
				 System.out.println(	"downloadAndSend succeeded in attempt: " 
												 			+ (attempt+1));
				 break;
				}				
				else{
					System.out.println("downloadAndSend() failed in attempt: " 
															+ (attempt+1));				
				}

			}
			if(attempt==5){
				System.out.println("downloadAndSend failed 5 times.");
				return;
			}

		}catch(Exception e){
			System.out.println("receiveFile() Exception: " + e.getMessage());
			return;
		}
	}//end of receiveFile()

/***
 * Precondition:	User has requested file deletion.
 * Postcondition: sends file delete request to FM
 ***/
public String deleteFile(String fName) throws Exception{

		//fName needs to be mapped to the fileId sending the request
		String fileId;

		//Use SHA-1 for calculating fileId
		MessageDigest md = MessageDigest.getInstance("SHA-1");
		md.update(fName.getBytes(DEF_ENCODING));
		md.update(oi.ownerId.getBytes(DEF_ENCODING));
		byte digest[]  = md.digest();
		fileId= new String(digest, DEF_ENCODING);
		outFMStream.write((byte)oi.idLength);

		outFMStream.write(oi.idArr);	//the ownerId

		//OwnerProtocolDT contains the passwd

		FNode ofNode = FileDirectory.getNode(fileId);

		if(ofNode == null){
			System.out.println("Could not retrieve FNode.");
			return null;
		}

		FileDescription fd = new FileDescription(ofNode, oi);
		fd.putId(ofNode);
		//No need for putEncFileSize()
		

		//Pass the password entered by the user	
		OwnerProtocolDT oProt = new OwnerProtocolDT(oi.password, fd,(byte)5);

		byte[] oProtArr = oProt.byteEncode();	

		DataOutputStream ndis = new DataOutputStream(outFMStream);

		int lenOwnerProt = 0;
		lenOwnerProt = oProtArr.length;
		if(!(lenOwnerProt > 0)){
			System.out.println("Invalid Owner Protocol. Quitting...");
			return null;
		}
		ndis.writeInt(lenOwnerProt);
		ndis.flush();
		outFMStream.write(oProtArr);
		outFMStream.flush();
		outFMStream.close();

		return fileId;	//Will be passed to receiveDeleteConf to save computation.
					
	}
	/***
	 * Precondition: file delete request has been sent to FM.
	 * Postcondition:receives delete confirmation from FM.
	 ***/
	public void receiveDeleteConf(String fileId) throws Exception{

		inFMStream = connection.getInputStream();	
		DataInputStream dis = new DataInputStream(inFMStream);
		try{
			int res = dis.readInt();

			if(res == RMFILE_CONF)
				System.out.println("Received RMFILE_CONF from FM.");
			else
				System.out.println("Did NOT receive RMFILE_CONF from FM.");

		System.out.println("Removing fileId from File Directory.");

		if(fileId != null)
			FileDirectory.removeNode(fileId);

		}catch(Exception e){
			System.out.println(e.getMessage());
		}

	}

  //Helper function, extracts FS IP addresses from message that FM
	//sends DT in response to file request.
	private String[]  decodeFSList(byte[] fsListArr) throws Exception{

		ByteArrayInputStream bais = new ByteArrayInputStream(fsListArr);
		DataInputStream badis = new DataInputStream(bais);

		String[] FS = new String[5];

		int len;

		//We know that there are 5 FS address Strings
		for(int fsIndex=0; fsIndex < 5; fsIndex++){

			len = badis.readInt();
			byte[] fsArr = new byte[len];
			badis.readFully(fsArr);
			FS[fsIndex] = new String(fsArr, DEF_ENCODING);

		}

		return FS;

	}

	//Helper function, to read user input strings.	
   public static String readString()
   {  int ch;
      String r = "";
      boolean done = false;
      while (!done)
      {  try
         {  ch = System.in.read();
            if (ch < 0 || (char)ch == '\n')
               done = true;
            else
               r = r + (char) ch;
         }
         catch(java.io.IOException e)
         {  done = true;
         }
      }
      return r;
   }

	   /***
			*  Precondition:   bFNode has been initialized with information required
			*                  to downloadAndSend() the file.
			*  Postcondition:  Get the list of FSs that have this file from bfNode.
			*                  Send block reqs to different FSs and receive and check
			*                  block sigs. Transmit blocks in proper order back to the
			*                  requesting HH
			***/
	 
	public boolean downloadAndSend(FNode bfNode) throws Exception{

			//Check for the condition where the total number of blocks is
			//less than 5
			int maxSeriesNum;
			if(bfNode.noBlocks<5) 
				maxSeriesNum = bfNode.noBlocks;
			else
				maxSeriesNum = 5;

			//Five RetrieveFS objects are instantiated in five threads.
	 ThreadGroup retrieveFSGroup = new ThreadGroup("RetrievefromFS");

	switch(maxSeriesNum){

		case 5: 
	 		RetrieveFS firstFS = 	new RetrieveFS("FirstFS", retrieveFSGroup, 
										 				bfNode, 5/*Series Num*/);
		case 4:
	 		RetrieveFS secondFS = new RetrieveFS("SecondFS", retrieveFSGroup,
										 				bfNode, 4);
		case 3:
	 		RetrieveFS thirdFS = 	new RetrieveFS("ThirdFS", retrieveFSGroup,
										 				bfNode, 3);
		case 2:
	 		RetrieveFS fourthFS = new RetrieveFS("FourthFS", retrieveFSGroup,
											 			bfNode, 2);
		case 1:
	 		RetrieveFS fifthFS = 	new RetrieveFS("FifthFS", retrieveFSGroup,
									 					bfNode, 1);
	 		break;

		default:
	 		System.out.println("Wrong maxSeriesNum: Shouldn't be getting this");
			return false;

		
	}//end of switch

	  /***
		 * Iterate through the 5 threads, blocking if necessary, to retrieve
		 * the file blocks in proper order
		 ***/
	 Thread groupFSThread[] = new Thread[retrieveFSGroup.activeCount()];
	 retrieveFSGroup.enumerate(groupFSThread);

		boolean available = true;

		Encryptor dTor = new Encryptor((oi.DESKey).getBytes());

		BufferedOutputStream outFile = null;
		String fName = bfNode.fileName;

	  try{
			outFile = new BufferedOutputStream(new FileOutputStream(fName));
			System.out.println("Opened file " + fName + " to write to disk.");	
		}
		catch(FileNotFoundException fnf){
			System.out.println("File Not Found: " + fName);
			return false;
		}	

		int count=0;

		/***
		 *	groupFSThread[0] corresponds to series 4
		 *	groupFSThread[4] corresponds to series 0
		 ***/ 

		boolean error = false;

		MessageDigest md = MessageDigest.getInstance("SHA-1");

	 	while(/*more blocks available*/ available){
	 		for (int seriesNum=maxSeriesNum-1; seriesNum>=0; seriesNum--){
				//May block if necessary
				//Read from each of the 5 buffers in order.
				try{
				byte[] blockBytes = ((RetrieveFS)groupFSThread[seriesNum]).getBlock();			

				if(blockBytes == null){
					System.out.println("Finished retrieving all file blocks");
					available=false;
					break;			//Careful - will 'break' out of for only
				}


				boolean badSig = false;			
				md.update(blockBytes);		

				byte[] newSig = md.digest();	//this also resets the digest

				//Check signature of retrieved block
				if(compareSigs(newSig, bfNode.blockSig[count]) != 0) 
						badSig = true;

				if(badSig){

					int badFSIndex = ((RetrieveFS)(groupFSThread[seriesNum])).FSIndex; 
					System.out.println("Blacklisting following FS: " + bfNode.FS[badFSIndex]);
					bfNode.FS[badFSIndex] = null;
					throw new SigException();
				}
							//decrypt the block
							try{
								byte[] dBlock = dTor.decrypt(blockBytes, blockBytes.length);
								//send block to disk 

								outFile.write(dBlock, 0, dBlock.length);
							}catch(Exception de){
								System.out.println("ERROR: Decrypting or writing-"+de.getMessage());
								outFile.flush();
								outFile.close();
								return false;
							}
					}
					catch(SigException se){
						System.out.println("downloadAndSend()- caught SigException...");	
						outFile.flush();
						outFile.close();
						return false;
					}
					catch(Exception eh){
						System.out.println("downloadAndSend() Exception: " + eh.getMessage());
						System.out.println("Unable to retrieve entire file. Please try again.");
						error=true;
						available=false;
						break;
					}
				count++;			//One more block
				System.out.println("Retrieved block: " + (count-1));
	 		}//end of for

		}//end of while

		System.out.println("\nTotal number of blocks retrieved: " + count);
		System.out.println("");

		if(error){
			outFile.flush();
			outFile.close();
			return false;
		}
	
		try{
			outFile.flush();
			System.out.println("Finished retrieving all blocks. Closing file.");
			outFile.close();
		}catch (Exception eee){
			System.out.println("3.downAndSend Exception: " + eee.getMessage());
			System.exit(1);
		}

		return true;					//All well
	
	}//end of downloadAndSend()


	/***
	 * Purpose: Helper function. Converts a byte array to String.
	 * Postcondition: Returns String representation of byte array.
	 ***/
	private String stringBlock(byte[] block){

		ByteArrayOutputStream baos = new ByteArrayOutputStream();

		baos.write(block,0,block.length);
		String blockString = baos.toString();
		return blockString;
	}

	//Saves OwnerInfo state to file.
	public void saveOwnerInfo(OwnerInfo oi) throws Exception{

		FileOutputStream fos = new FileOutputStream("owner.properties");
		ObjectOutputStream oos = new ObjectOutputStream(fos);
	
		oos.writeObject(oi);
		oos.flush();
		oos.close();
		fos.close();
	}

}//end of class DeskTop


