// JOBPOOL.H

// Declaration of classes
// Member functions are defined in JOBPOOL.CC

#ifndef JOBPOOL_H
#define JOBPOOL_H

#define MAXAGENTNUMBER 512
#define MAXMACHINENUMBER 128
#define MAXJOBNUMBER  32768
#define LENGTHOFHOSTNAME 64
#define LENGTHOFJOBNAME 128
#define MAXDATASECTIONNUMBER 64
#define MAXSTRLENGTH 256

#define MAINPROGRAMADDRS -9999
#define REQUEST 0
#define RESULT 1
#define CONFIRMATION 2

typedef int Info;
typedef int Tid;
typedef int JobID;
typedef int JobNumber;
typedef int ResultID;
typedef int AgentID;
typedef int MachineID;
typedef int Msgtag;

// PackageType represents the type of package: request package or
// result package.
typedef int PackageType;
 
// CheckTime controls the time interval in seconds between which the living
// status is checked for all of the machines in a virtual machine.
typedef int CheckTime;

// StorageType decides whether Buffer is created in place or somewhere else.
enum    StorageType { OUTOF_PLACE, IN_PLACE };

// JobStatus represents the status of a job in the jobpool. 
enum    JobStatus   { INITIAL, SENT, FINISHED, SENTTOSELF };

// RunType decides how many JobAgents are spawned in a single machine.  
enum    RunType     { NICE, NORMAL, FAST };

// MachineStatus denotes the status of a machine in the 'hostname' file.
enum    MachineStatus { ALIVE, SPAWNED, DEAD };

// Reliability decides the tolerance for crashing of machines or networks.  
enum    Reliability { LOW, HIGH };

// RunMode controls the mode of starting a virtual machine. In MANUAL mode,
// the virtual machine has to be started before an application runs.
enum    RunMode { MANUAL, AUTOSTART };

// AddMode controls the mode of add a host into a group. In NEW mode, only
// a new host is added into group. In ANY mode, any host can be added into
// group. (AddMode is only used in Class GroupManager)
enum    AddMode { NEW, ANY };
 
// Buffer is used to temporarily store data to be delivered.
class Buffer{
      friend void PackBuffer(Buffer *);
      friend void UnpackBuffer(Buffer *);
   public:
      Buffer();
      Buffer( StorageType );
      void PackInt(int *, int);
      void PackFloat(float *, int);
      void PackString(char *);
      void UnpackInt(int *, int);
      void UnpackFloat(float *, int);
      void UnpackString(char *);

   protected:
      StorageType BufferType;
      int *intptr[MAXDATASECTIONNUMBER];
      int intnum[MAXDATASECTIONNUMBER];
      int intcnt;
      int intrev;
      float *floatptr[MAXDATASECTIONNUMBER];
      int floatnum[MAXDATASECTIONNUMBER];
      int floatcnt; 
      int floatrev;
      char *strptr[MAXDATASECTIONNUMBER];
      int strnum[MAXDATASECTIONNUMBER];
      int strcnt; 
      int strrev; 
};

// Machine is for the infomation about host machines.
class Machine{
     friend class JobManager;
     friend class GroupManager;
   public:
     Machine();
     void  InsertJobID(JobID);
     int  RemoveJobID(JobID);

   protected:
     char *name;
     MachineID  machineID;
     int  jobID[MAXJOBNUMBER];
     MachineStatus status;
     int  jobCount;
};


// Agent records the information about a remote JobAgent. 
class AgentInfo{
      friend class JobManager;
      friend class GroupManager;
   protected:
      char *hostName;
      char *hostArch;
      int  hostSpeed;
      Tid  myTid;
      Tid  parentId;
      JobID  jobID;
};

class Package{
      friend class JobManager;
      friend class GroupManager;
      friend class Distributor;
      friend class JobAgent;
   public:
      Package();
      void SetPackageType(PackageType);
      void SetSource(Tid);
      void SetDestination(Tid);
      PackageType GetPackageType();
      Tid GetSource();
      Tid GetDestination();

   protected:
      PackageType type;
      Tid         source;
      Tid         destination;
};

// Job encapsulates all data and other information about a task to be 
// parallelized.
class Job : public Package{
      friend class JobManager;
      friend class GroupManager;
      friend class Distributor;
      friend class JobPool;
      friend class JobAgent;
   public:
      Job();
      Job( StorageType );
      void SetJobName(char *);
      void SetLocalJobID( JobID );
      JobID  GetJobID();
      JobID  GetLocalJobID();
      char* GetJobName();
      
      Buffer *broadcastReceiveBuffer;
      Buffer *privateReceiveBuffer;
      Buffer *privateSendBuffer;


   protected:
      JobStatus jobStatus;
      AgentID    agentID;
      MachineID machineID;
      JobID jobID;
      JobID localJobID;
      char *jobName;

      void SetJobID( JobID );
};

// Result contains the results from a requested job.
class Result : public Package {
      friend class JobAgent;
   public:
      Result();
      Result( StorageType ); 
      void SetResultName(char *);
      void SetResultID(JobID);
      void SetLocalResultID(JobID);
      ResultID  GetResultID();
      ResultID  GetLocalResultID();
      char *GetResultName();
      
      Buffer *broadcastReceiveBuffer;
      Buffer *privateReceiveBuffer;
      Buffer *privateSendBuffer;

   protected:
      ResultID resultID;
      ResultID localResultID;
      char *resultName;

};

// JobPool is a collection of jobs to be parallelized. 
class JobPool{
   public:
      JobPool( Job *, int );
      Info AddJob( Job * );
      void RemoveJob( JobID );
      JobNumber GetJobNumber();
      JobNumber GetTotalJobNumber();
      JobNumber GetUserJobNumber();
      Job *GetJob(JobNumber);
      void ReduceJobNumberByOne();
      void ReduceUserJobNumberByOne();


   protected:
      JobID     nextJobID;
      JobNumber jobNumber;
      JobNumber totalJobNum;
      JobNumber userJobNum;
      Job *job[MAXJOBNUMBER];
};

// JobManager administrates the whole parallelizing computation process. 
class JobManager{
   public:
      JobManager(Job *, JobNumber, Buffer *, RunType, Reliability, RunMode, CheckTime);
      ~JobManager();
      Result* Run();

   protected:
      JobPool     *jobPool;
      RunType     runType;
      Reliability reliability;
      AgentID     nextAgentID;
      MachineID   nextMachineID;
      Buffer      *broadcastBuffer;
      AgentInfo   *agent[MAXAGENTNUMBER];
      char        *hostName[MAXMACHINENUMBER];
      Result      *result;
      JobID       resendJobID;
      Machine     *machines;
      RunMode     runMode;
      CheckTime   checktime;

      int         SendToNewHost(MachineID, int, JobStatus);
      int         Resend(MachineID, AgentID, JobID, JobStatus);
};

// JobAgent is a remote representative sitting in a worker machine and
// delivers data between JobManager and its Worker.
class JobAgent{
   public:
      JobAgent();
      void Run( Result* (*Worker)(Job *) );

   protected:
      Tid myTid;
      Tid parentTid;
      PackageType currentType;
      Tid currentSource;
      Tid currentDestination;
      JobID currentJobID;
      JobID localJobID;
      
      Job* Receive();
      void Send(Result *);
};

// GroupManager administrates the parallelizing computation process of
// one group. 
class GroupManager{
   public:
      GroupManager(Job *, JobNumber, Buffer *, RunType, AddMode);
      Result* Run();

   protected:
      JobPool     *jobPool;
      RunType     runType;
      AgentID     nextAgentID;
      MachineID   nextMachineID;
      Buffer      *broadcastBuffer;
      AgentInfo   *agent[MAXAGENTNUMBER];
      char        *hostName[MAXMACHINENUMBER];
      Result      *result;
      Machine     *machines;
      AddMode     addMode;
};

class Distributor{
   public:
      Distributor(Job *, JobNumber, Buffer *);
      Result* Run();

   protected:
      JobPool     *jobPool;
      Buffer      *broadcastBuffer;
      Result      *result;
};


// Utility function for packing Buffer into PVM buffer.
void PackBuffer(Buffer *);

// Utility function for unpacking Buffer from PVM buffer.
void UnpackBuffer(Buffer *);

#endif 
