//JOBPOOL.CC

// Member function definitions for classes

#include <iostream.h>
#include <fstream.h>
#include <iomanip.h>
#include <stdlib.h>
#include <string.h>
#include "pvm3.h"
#include "jobpool.h"
#include "machinelist.h"


// Utility Function to pack Buffer to PVM buffer
void PackBuffer(Buffer *bufptr)
{

   int i;

// Pack integers
   pvm_pkint(&(bufptr->intcnt), 1, 1);
   if(bufptr->intcnt != 0)
      for(i=0; i<bufptr->intcnt; i++)
      {
         pvm_pkint(&(bufptr->intnum[i]), 1, 1);
	 pvm_pkint(bufptr->intptr[i], bufptr->intnum[i], 1);
      }

// Pack floats
   pvm_pkint(&(bufptr->floatcnt), 1, 1);
   if(bufptr->floatcnt != 0)
      for(i=0; i<bufptr->floatcnt; i++)
      {
         pvm_pkint(&(bufptr->floatnum[i]), 1, 1);
	 pvm_pkfloat(bufptr->floatptr[i], bufptr->floatnum[i], 1);
      }

// Pack strings
   pvm_pkint(&(bufptr->strcnt), 1, 1);
   if(bufptr->strcnt != 0)
      for(i=0; i<bufptr->strcnt; i++)
      {
         pvm_pkstr(bufptr->strptr[i]);
      }
}

// Utility Function to unpack Buffer from PVM buffer
void UnpackBuffer(Buffer *bufptr)
{
   int i;

// Unpack integers
   pvm_upkint(&(bufptr->intcnt), 1, 1);
   if(bufptr->intcnt != 0)
       for(i=0; i<bufptr->intcnt; i++)
       {
          pvm_upkint(&(bufptr->intnum[i]), 1, 1);
          bufptr->intptr[i] = new int[bufptr->intnum[i]];
	  pvm_upkint(bufptr->intptr[i], bufptr->intnum[i], 1);
       }

// Unpack floats
       pvm_upkint(&(bufptr->floatcnt), 1, 1);
       if(bufptr->floatcnt != 0)
          for(i=0; i<bufptr->floatcnt; i++)
          {
             pvm_upkint(&(bufptr->floatnum[i]), 1, 1);
             bufptr->floatptr[i] = new float[bufptr->floatnum[i]];
	     pvm_upkfloat(bufptr->floatptr[i], bufptr->floatnum[i], 1);
          }

// Unpack strings
       pvm_upkint(&(bufptr->strcnt), 1, 1);
       if(bufptr->strcnt != 0)
          for(i=0; i<bufptr->strcnt; i++)
          {
             bufptr->strptr[i] = new char[MAXSTRLENGTH];
             pvm_upkstr(bufptr->strptr[i]);
          }
}

// Default constructor of class Buffer
Buffer::Buffer(){
   BufferType = OUTOF_PLACE;
   intcnt = 0;
   intrev = 0;
   floatcnt = 0;
   floatrev = 0;
   strcnt = 0;
   strrev = 0;
}

// Constructor of class Buffer with optional storage type
Buffer::Buffer(StorageType buffertype){
   BufferType = buffertype;
   intcnt = 0;
   intrev = 0;
   floatcnt = 0;
   floatrev = 0;
   strcnt = 0;
   strrev = 0;
}

// Pack integers into Buffer
void Buffer::PackInt(int *np, int cnt){
   if(BufferType == IN_PLACE)
   {
      intptr[intcnt] = np;
   }
   else
   {
      intptr[intcnt] = new int[cnt];
      for(int i=0; i<cnt; i++)
      {
	 intptr[intcnt][i] = *np;
	 np++;
      }
   }
   intnum[intcnt] = cnt;
   intcnt++;
}

// Pack floats into Buffer
void Buffer::PackFloat(float *fp, int cnt){
   if(BufferType == IN_PLACE)
   {
      floatptr[floatcnt] = fp;
   }
   else
   {
      floatptr[floatcnt] = new float[cnt];
      for(int i=0; i<cnt; i++)
      {
	 floatptr[floatcnt][i] = *fp;
	 fp++;
      }
   }
   floatnum[floatcnt] = cnt;
   floatcnt++;
}

// Pack strings into Buffer
void Buffer::PackString(char *cp){
   int n = 0;
   if(BufferType == IN_PLACE)
   {
      strptr[strcnt] = cp;
   }
   else
   {
     // How mang characters?
      while(cp[n] != '\0')
      {
	 n++;
      }
      strptr[strcnt] = new char[n+1];
      for(int i=0; i<n; i++)
      {
	 strptr[strcnt][i] = *cp;
	 cp++;
      }
      strptr[strcnt][n] = '\0';
   }
   strnum[strcnt] = n;
   strcnt++;
}

// Unpack intergers from Buffer
void Buffer::UnpackInt(int *np, int cnt){
   for(int i=0; i<cnt; i++)
   {
      np[i] = intptr[intrev][i];
   }
   intrev++;
}

// Unpack floats from Buffer
void Buffer::UnpackFloat(float *fp, int cnt){
   for(int i=0; i<cnt; i++)
   {
      fp[i] = floatptr[floatrev][i];
   }
   floatrev++;
}

// Unpack strings from Buffer
void Buffer::UnpackString(char *cp){
   int i = 0;
   while(strptr[strrev][i] != '\0')
   {
      cp[i] = strptr[strrev][i];
      i++;
   }
   cp[i] = '\0';
   strrev++;
}

// Constructor of Class Agent
AgentInfo::AgentInfo(){
}

// Constructor of Class Package
Package::Package(){
   type = REQUEST;
   source = MAINPROGRAMADDRS;
   destination = MAINPROGRAMADDRS;
}

// Set and Get functions of Class Package
void Package::SetPackageType(PackageType pType){ type = pType; }
void Package::SetSource(Tid src){ source = src; }
void Package::SetDestination(Tid dst){ destination = dst; }
PackageType Package::GetPackageType(){ return type; }
Tid Package::GetSource(){ return source; }
Tid Package::GetDestination(){ return destination; }

// Default constructor of Class Job
Job::Job(){
   Package();
   jobID = -1;
   jobName = "";
   jobStatus = INITIAL;
   broadcastReceiveBuffer = new Buffer();
   privateReceiveBuffer = new Buffer();
   privateSendBuffer = new Buffer();
}

// Constructor of Class Job with optional storage type
Job::Job( StorageType bufferType ){
   jobID = -1;
   jobName = "";
   jobStatus = INITIAL;
   broadcastReceiveBuffer = new Buffer(bufferType);
   privateReceiveBuffer = new Buffer(bufferType);
   privateSendBuffer = new Buffer(bufferType);
}

// Set and Get functions for Class Job
void Job::SetJobID(JobID jobid){ jobID = jobid; }

void Job::SetLocalJobID(JobID jobid){ localJobID = jobid; }

void Job::SetJobName(char *jobname){ jobName = jobname; }

JobID Job::GetJobID(){ return jobID; }

JobID Job::GetLocalJobID(){ return localJobID; }

char* Job::GetJobName(){ return jobName; }

// Default constructor of Class Result
Result::Result(){
   Package();
   resultID = -1;
   resultName = "";
   broadcastReceiveBuffer = new Buffer();
   privateReceiveBuffer = new Buffer();
   privateSendBuffer = new Buffer();
}

// Constructor of Class Result with optional storage type
Result::Result(StorageType bufferType){
   resultID = -1;
   resultName = "";
   broadcastReceiveBuffer = new Buffer(bufferType);
   privateReceiveBuffer = new Buffer(bufferType);
   privateSendBuffer = new Buffer(bufferType);
}

// Set and Get functions of Class Result
void Result::SetResultID(JobID jobID){ resultID = jobID; }

void Result::SetLocalResultID(JobID jobID){ localResultID = jobID; }

void Result::SetResultName(char *jobname){ resultName = jobname; }

ResultID Result::GetLocalResultID(){ return localResultID; }

ResultID Result::GetResultID(){ return resultID; }

char* Result::GetResultName(){ return resultName; }

// Constructor of Class JobPool
JobPool::JobPool( Job *job0, int jobnum)
{
   jobNumber = 0;
   nextJobID = 0;
   totalJobNum = 0;

// add Jobs into JobPool
   for(int i=0; i<jobnum; i++)
   {
      AddJob(&job0[i]);
   }
   userJobNum = totalJobNum;   
}

// Get functions of Class JobPool
JobNumber JobPool::GetJobNumber(){ return jobNumber; }

JobNumber JobPool::GetTotalJobNumber(){ return totalJobNum; }
JobNumber JobPool::GetUserJobNumber(){ return userJobNum; }


Job* JobPool::GetJob(JobNumber i){ return job[i]; }

// Add one more Job into JobPool 
int JobPool::AddJob( Job *jobptr ){

   job[nextJobID] = new Job;
   job[nextJobID]->type = jobptr->type;
   job[nextJobID]->source = jobptr->source;
   job[nextJobID]->destination = jobptr->destination;
   job[nextJobID]->localJobID = jobptr->localJobID;
   job[nextJobID]->SetJobID( nextJobID );
   job[nextJobID]->SetJobName( jobptr->GetJobName() );
   job[nextJobID]->privateSendBuffer = jobptr->privateSendBuffer;

   jobNumber++;
   totalJobNum++;
   nextJobID++;
}

// Remove one Job from JobPool
void JobPool::RemoveJob( JobID jobid )
{
   delete job[jobid];
}

// Reduce Job counter by one
void JobPool::ReduceJobNumberByOne(){ jobNumber--; }
void JobPool::ReduceUserJobNumberByOne(){ userJobNum--; }

// Constructor of Class JobManager
JobManager::JobManager(Job *job, JobNumber jobnum, Buffer *bcbuffer, 
      RunType runtype, Reliability reliable, RunMode runmode, CheckTime dtime)
{
   jobPool = new JobPool(job, jobnum);
   result = new Result[jobnum];
   broadcastBuffer = bcbuffer;
   machines = new Machine[MAXMACHINENUMBER];

   runType = runtype;
   reliability = reliable;
   resendJobID = 0;
   runMode = runmode;
   checktime = dtime;
   
}

JobManager::~JobManager()
{
   if(runMode == AUTOSTART)
   {
      cout << "Information: pvm has halted." << endl;
      pvm_halt();
   }
}

Result *JobManager::Run()
{
// variables about hostname
   char tmpname[LENGTHOFHOSTNAME], *tmp1;
   int len;

// variables about starting virtual machine and spawning jobs
   Tid mytid, tid;
   Tid tids[MAXJOBNUMBER];
   Info info;
   Info status[MAXAGENTNUMBER];
   int actjobNum;
   int numberOfSpawnedJobs;
   int numberOfAgents; // Number of JobAgents on one single machine
   int tmpnumberOfAgents;
   int knext;

// variables about virtual machine 
   int machineNum;
   MachineID currentMachineID;
   MachineList avamachineList;
   MachineList unavamachineList;
   AgentID currentAgentID;
   PackageType currentType;
   Tid currentSource;
   Tid currentDestination;

// variables about Job and JobPool
   JobID currentJobID;
   JobID nextJobID;
   JobID sendJobID;
   JobID tmpnextJobID;
   Buffer *bufptr;
   int signal;
   Job *jobptr;
   int localJobID;
   char jobName[LENGTHOFJOBNAME];

// variables about timer 
   time_t starttime;
   time_t endtime;
   double diftime;
   int times = 0;

// variables about getting OS name 
   FILE *pp;
   char osname[20];

// variables about "ping"
   char *cmd;
	  
   int i, j;

// get host names from file "hostname".

// open file "hostname"
   ifstream filein("hostname", ios::in);

   if(!filein)   
   {
      cerr << "file 'hostname' could not be opened" << endl;
      exit(1);
   }

// read machine names from "hostname"
   machineNum = 0;
   while(!filein.eof())
   {
      filein >> tmpname;
      len = strlen((char *)tmpname);
      if(len != 0)
      {
	 tmp1 = new char[len + 1];
	 hostName[machineNum] = strncpy(tmp1, tmpname, len+1);
	 machineNum++;
      }
   }
   filein.close();

// print out machine names
   cout << endl << "The hostname file includes the following hosts:" 
	<< endl;
   for(i=0; i<machineNum; i++)
   {
      cout << hostName[i] << endl;
   }
      cout <<  endl;

// detect the name of operating system
   pp = popen("uname -s", "r");
   fscanf(pp,"%s",&osname);
   pclose(pp);
   printf("The Operating System is %s\n\n", osname);


// "ping" all of machines
   for(i=0; i<machineNum; i++)
   {
      char cmdline[LENGTHOFHOSTNAME+10] = "ping ";
      cmd = strcat(cmdline, hostName[i]);
      if(strcmp(osname, "HP-UX") == 0)
         cmd = strcat(cmdline, " 8 1");

      info = system(cmd);

      machines[i].name = hostName[i];
      machines[i].machineID = i;

      if(info == 0)
      {
         machines[i].status = ALIVE;
	 // insert one machine into available machine list
	 avamachineList.addnew(machines[i]);
      }
      else
      {
         machines[i].status = DEAD;
	 // insert one machine into unavailable machine list
	 unavamachineList.addnew(machines[i]);
      }
    }
 
 // detect whether pvm has been started
   mytid = pvm_mytid();

// if runmode is AUTUSTART, pvm can be started here.
  if(runMode == AUTOSTART )  
  {
    if(mytid < 0)
    {
       cout << "    " << endl;
       cout << " Don't worry. I will start pvmd for you." << endl;
       cout << "    " << endl;

       // start pvm
       info = pvm_start_pvmd(0, (char **)0, 1);

       if(info != 0)
       {
          cout << "pvm_start_pvmd() fails " << endl;
          exit(0);
       }

       mytid = pvm_mytid();
    }

// add machines into pvm
   info = pvm_addhosts((char **)hostName, machineNum, status);

   if(info < 1)
      cout << "Warning: No hosts are added" << endl;
   else 
      cout << "Actual added host number: " << info+1 << endl;
  }

// How many jobs are spawned on one single machine simultaneously?
   switch(runType)
   {
      case NICE:
	 numberOfAgents = 1;
	 break;
      case NORMAL:
	 numberOfAgents = 2;
	 break;
      case FAST:
	 numberOfAgents = 3;
	 break;
      default:
	 numberOfAgents = 2;
	 break;
    }

   // spawn jobs
   actjobNum = 0;
   for(j=0; j<numberOfAgents; j++)
   {
      for(i=0; i<machineNum; i++)
      {
         if(actjobNum >= jobPool->GetTotalJobNumber()) break;
         if(machines[i].status != DEAD)
	 {
            numberOfSpawnedJobs = 
		       pvm_spawn(jobPool->GetJob(actjobNum)->GetJobName(),
		       (char **)0, PvmTaskHost,
		       machines[i].name, 1, &tid);
            if( numberOfSpawnedJobs == 1)
            {
	       machines[i].InsertJobID(actjobNum);
	       machines[i].status = SPAWNED;
	       machines[i].jobCount++;

               agent[actjobNum] = new AgentInfo;
               agent[actjobNum]->myTid = tid;
               agent[actjobNum]->hostName = hostName[i];
               agent[actjobNum]->jobID = actjobNum;
	       jobPool->GetJob(actjobNum)->agentID = actjobNum;
	       jobPool->GetJob(actjobNum)->machineID = i;
	       tids[actjobNum] = tid;
	       actjobNum++;
            }
	 }
      }
   }

   // next job to be sent
   nextJobID = actjobNum;

 // Broadcast data if any
 pvm_initsend(PvmDataDefault);

 if(broadcastBuffer != NULL)
 {
   // there is broadcast buffer
   signal = 1;
   pvm_pkint(&signal, 1, 1);
   PackBuffer(broadcastBuffer);
 }
 else
 {
   // there is not broadcast buffer
   signal = -1;
   pvm_pkint(&signal, 1, 1);
 } 
 pvm_mcast(tids, actjobNum, 0);

 // Send private data 
   for(j=0; j<actjobNum; j++)
   {
      pvm_initsend(PvmDataDefault);

      jobptr = jobPool->GetJob(j);
      pvm_pkint(&(jobptr->type), 1, 1);
      pvm_pkint(&(jobptr->source), 1, 1);
      pvm_pkint(&(jobptr->destination), 1, 1);

      pvm_pkint(&j, 1, 1);

      localJobID = jobptr->GetLocalJobID();
      pvm_pkint(&localJobID, 1, 1);
      pvm_pkstr(jobptr->jobName);

      bufptr = jobPool->GetJob(j)->privateSendBuffer;

      PackBuffer(bufptr);

      info = pvm_send(agent[j]->myTid, 1);
      if(info < 0)
      {
	 cout << " Warning: It is fail to send private data to "
	      << agent[j]->hostName << endl;
      }
      else
      {
	 jobPool->GetJob(j)->jobStatus = SENT;
      }

   }

// Receive data from JobAgents
// initialize timer for checking machine status
  starttime = time(0);

  while(jobPool->GetUserJobNumber() > 0)  // there is jobs uncompleted
  {
    int arrived;
    arrived = pvm_nrecv(-1, 2); // detect whether there are data received
    if(arrived <= 0)  
    {
       // there are not data received so check status of machines 
       Machine tmpmachine;
       MachineID tmpmachineID;
       int tmpjobCount;

       endtime = time(0);
       if(difftime(endtime, starttime) < (double)checktime)
	  continue; // time interval is less than checktime so skip

// check status of machines
       if(!avamachineList.isEmpty())
       {
	  // check list of available machines
          char cmdline[LENGTHOFHOSTNAME+10] = "ping ";

          // remove a machine from the available list 
          avamachineList.Remove(tmpmachine);
	  tmpmachineID = tmpmachine.machineID;

          cmd = strcat(cmdline, tmpmachine.name);
	  if(strcmp(osname, "HP-UX") == 0)  // for HP-UX machines
	     cmd = strcat(cmdline, " 8 1");

          // "ping" a machine
          info = system(cmd);
	  if(info == 0)
	  {
	     // the machine is still alive
	     avamachineList.addnew(machines[tmpmachineID]);
          }
	  else
	  {
	     // the machine was died so add the machine to the unavailable list
	     unavamachineList.addnew(machines[tmpmachineID]);

	     // find which jobs should be reassigned in the JobPool
	     tmpjobCount = machines[tmpmachineID].jobCount;
	     for(i=0; i<tmpjobCount; i++)
	     {
	        jobPool->GetJob(machines[tmpmachineID].jobID[i])->jobStatus = INITIAL;
		machines[tmpmachineID].jobID[i] = -1;
             }
	     machines[tmpmachineID].jobCount = 0;
	     machines[tmpmachineID].status = DEAD;
          }
       }
       if(!unavamachineList.isEmpty())
       {
	  // check the unavailable list of machines
	  // the checktime for unavailable list is 10 times of that for 
	  // available list
	  times++;
	  if(times > 10)
	     times = 0;
          else
	     continue; // it is not a check time so skip

          char cmdline[LENGTHOFHOSTNAME+10] = "ping ";

	  // remove a machine from the list of unavailable machines
          unavamachineList.Remove(tmpmachine);
	  tmpmachineID = tmpmachine.machineID;

          cmd = strcat(cmdline, tmpmachine.name);
	  if(strcmp(osname, "HP-UX") == 0)
	     cmd = strcat(cmdline, " 8 1");  // for HP-UX machines

	  // "ping" a machine
          info = system(cmd);
	  if(info != 0)
	  {
	     // the machine is still unavailable
	     unavamachineList.addnew(machines[tmpmachineID]);
          }
	  else
	  {
	     // the machine was born   
	     machines[tmpmachineID].status = ALIVE;
	     machines[tmpmachineID].jobCount = 0;
	     // add the machine to the list of available machines
	     avamachineList.addnew(machines[tmpmachineID]);

	     // Send an initial job to the new-born machine
	     SendToNewHost(tmpmachineID, numberOfAgents, INITIAL);
	     tmpnumberOfAgents = numberOfAgents - machines[tmpmachineID].jobCount;
	     if(tmpnumberOfAgents > 0)
	     {
	        // Send sent jobs to the new-born machine
	        SendToNewHost(tmpmachineID, tmpnumberOfAgents, SENT);
             }
          }
       }

       // Initialize timer 
       starttime = time(0);

    }
    else
    {
      // there are data received

      pvm_upkint(&currentType, 1, 1);
      pvm_upkint(&currentSource, 1, 1);
      pvm_upkint(&currentDestination, 1, 1);

    if(currentType == RESULT || currentType == CONFIRMATION)
    {
      // find the current JobID
      pvm_upkint(&currentJobID, 1, 1);
      pvm_upkint(&localJobID, 1, 1);
      pvm_upkstr(jobName);

      // find the current AgentID
      currentAgentID = jobPool->GetJob(currentJobID)->agentID;

      // find the current MachineID
      currentMachineID = jobPool->GetJob(currentJobID)->machineID;
     
      // print out JobID and MachineID
      cout << "Received: JobID = " << currentJobID 
	   << "  MachineID = " << currentMachineID << endl;

      if(jobPool->GetJob(currentJobID)->jobStatus == SENT)
      {
	 if(currentType == RESULT)
	 {
            // the current job is waiting for the results
            bufptr = result[currentJobID].privateReceiveBuffer;

	    // unpack pvm buffer to the received Buffer of Result
            UnpackBuffer(bufptr);

	    jobPool->ReduceUserJobNumberByOne();
         }

         machines[currentMachineID].RemoveJobID(currentJobID);

	 // set the status of the current job FINISHED 
         jobPool->GetJob(currentJobID)->jobStatus = FINISHED;

	 // The size of JobPool is reduced by one
	 jobPool->ReduceJobNumberByOne();

	 if(jobPool->GetJobNumber() > 0)
	 {
	    // there is jobs unfinished
	    if(nextJobID < jobPool->GetTotalJobNumber())
	    {
	       tmpnextJobID = nextJobID;
	       while(jobPool->GetJob(nextJobID)->jobStatus != INITIAL)
	       {
		  // Is there any unsent jobs whose ID is larger than
		  // next jobID?
		  nextJobID++;
		  if(nextJobID >= jobPool->GetTotalJobNumber())
		  {
		     nextJobID = tmpnextJobID;
                     break;
                  }
               }

	       sendJobID = nextJobID;
	       knext = 1;
	       for(i=0; i<nextJobID; i++)
	       {
		  if(jobPool->GetJob(i)->jobStatus == INITIAL)
		  {
		     // There is a killed jobs whose ID is less than
		     // next jobID. So send it first.
		     sendJobID = i;
		     knext = 0;
		     break;
                  }
               }

	       if(strcmp(jobPool->GetJob(currentJobID)->GetJobName(),
                         jobPool->GetJob(sendJobID)->GetJobName()) != 0)
	       {
		  // The job name to be sent is different from that of
		  // the current job so a new program to be spawned
		  pvm_kill(agent[currentAgentID]->myTid);
                  numberOfSpawnedJobs = 
                            pvm_spawn(jobPool->GetJob(sendJobID)->GetJobName(),
		                   (char **)0, PvmTaskHost,
		                   agent[currentAgentID]->hostName,
				   1, &tid);
                  if(numberOfSpawnedJobs == 1)
                  {
                     agent[currentAgentID]->myTid = tid;
                  }
		  else
		  {
		     cout << "spawn " << jobPool->GetJob(sendJobID)->GetJobName()
			  << " on " << agent[currentAgentID]->hostName 
			  << " fails." << endl;
                     break;
                  }
               }
                    
                // send broadcast data if any
                pvm_initsend(PvmDataDefault);

                if(broadcastBuffer != NULL)
                {
                   signal = 1; // there is a broadcast Buffer
                   pvm_pkint(&signal, 1, 1);
                   PackBuffer(broadcastBuffer);
                }
                else
                {
                   signal = -1; // there is not any broadcast Buffer
                   pvm_pkint(&signal, 1, 1);
                } 
                info = pvm_send(agent[currentAgentID]->myTid, 0);
                if(info < 0)
                {
	           cout << " Warning: It is fail to send broadcast data to "
	                << agent[currentAgentID]->hostName << endl;
		   break;
                }

               // send private data
               pvm_initsend(PvmDataDefault);

               jobptr = jobPool->GetJob(sendJobID);
               pvm_pkint(&(jobptr->type), 1, 1);
               pvm_pkint(&(jobptr->source), 1, 1);
               pvm_pkint(&(jobptr->destination), 1, 1);

               pvm_pkint(&sendJobID, 1, 1);

	       localJobID = jobptr->GetLocalJobID();
               pvm_pkint(&localJobID, 1, 1);
               pvm_pkstr(jobptr->jobName);

	       bufptr = jobPool->GetJob(sendJobID)->privateSendBuffer;

               PackBuffer(bufptr);

               info = pvm_send(agent[currentAgentID]->myTid, 1);
               if(info < 0)
               {
	          cout << " Warning: It is fail to send private data to "
	               << agent[currentAgentID]->hostName << endl;
               }
               else
               {
	          machines[currentMachineID].InsertJobID(sendJobID);
	          machines[currentMachineID].status = SPAWNED;

	          jobPool->GetJob(sendJobID)->jobStatus = SENT;
	          jobPool->GetJob(sendJobID)->agentID = currentAgentID;
	          jobPool->GetJob(sendJobID)->machineID = currentMachineID;
	          agent[currentAgentID]->jobID = sendJobID;
		  if(knext == 1)
		     nextJobID++; // The sent job is not a unsent job so
				  // nextJobID is incresed by one.
               }
            }
	    else
	    {
	       if(Resend(currentMachineID, currentAgentID, 
			 currentJobID, INITIAL) == 0)
               {
		  // All of jobs have been sent so reduce job number on
		  // each machine to one.
	          if(machines[currentMachineID].jobCount > 1)
	          {
		     pvm_kill(agent[currentAgentID]->myTid);
		     machines[currentMachineID].jobCount--;
                  }
	          else
	          {
		     // Resend sent jobs if high reliability is expected.
		     if(reliability == HIGH) 
		        Resend(currentMachineID, currentAgentID, 
			 currentJobID, SENT);
                  }
	       }
            }
         }
      }
     }
     }
     arrived = pvm_nrecv(-1, 3);
     if(arrived > 0)  
     {
	pvm_upkint(&currentType, 1, 1);
	if(currentType == REQUEST)
	{
	   jobptr = new Job;

	   pvm_upkint(&currentSource, 1, 1);
	   pvm_upkint(&currentDestination, 1, 1);
	   pvm_upkint(&currentJobID, 1, 1);
	   pvm_upkint(&localJobID, 1, 1);

	   jobptr->type = currentType;
	   jobptr->source = currentSource;
	   jobptr->destination = currentDestination;

	   jobptr->SetLocalJobID(localJobID);

	   pvm_upkstr(jobName);
	   jobptr->SetJobName(jobName);

	   bufptr = jobptr->privateSendBuffer;
	   UnpackBuffer(bufptr);

	   // add the requested job into JobPool
	   jobPool->AddJob(jobptr);
        }
     }
  }

   // kill all of agents
   for(i=0; i<nextAgentID; i++)
      pvm_kill(agent[i]->myTid);

   // delete all of machines in the parallel virtual machine
   if(runMode == AUTOSTART)
      info = pvm_delhosts((char **)hostName, machineNum, status);

  // exit from pvm
   pvm_exit();

  // remove jobs from JobPool
   for(i=0; i<actjobNum; i++)
      jobPool->RemoveJob((JobID)i);

   delete jobPool;
      
   return result; 
    
}

int JobManager::SendToNewHost(MachineID currentMachineID, int numberOfAgents,
			      JobStatus sendJobStatus)
{
   int i, j;
   Tid tid;
   int numberOfSpawnedJobs;
   Buffer *bufptr;
   Info info;
   AgentID currentAgentID;
   int signal;
   Job *jobptr;
   JobID localJobID;

// spawn job in a new-born machine
  for(j=0; j<numberOfAgents; j++)
  {
    for(i=0; i<jobPool->GetTotalJobNumber(); i++)
    {
       // find the job to be sent
       resendJobID = resendJobID%jobPool->GetTotalJobNumber();
       if(jobPool->GetJob(resendJobID)->jobStatus == sendJobStatus)
       {
	  // spawn job
          numberOfSpawnedJobs = 
                      pvm_spawn(jobPool->GetJob(resendJobID)->GetJobName(),
	              (char **)0, PvmTaskHost,
	              machines[currentMachineID].name,
	              1, &tid);
          if(numberOfSpawnedJobs == 1)
          {
	     currentAgentID = nextAgentID;
	     nextAgentID++;
	     agent[currentAgentID] = new AgentInfo;
             agent[currentAgentID]->myTid = tid;
             agent[currentAgentID]->hostName = machines[currentMachineID].name;
          }
	  else
	  {
	     cout << "spawn " << jobPool->GetJob(resendJobID)->GetJobName()
	          << " on " << machines[currentMachineID].name 
		  << " fails." << endl;
             return 0;
          }
	  
         // send broadcast data if any
         pvm_initsend(PvmDataDefault);

         if(broadcastBuffer != NULL)
         {
            signal = 1;
            pvm_pkint(&signal, 1, 1);
            PackBuffer(broadcastBuffer);
         }
         else
         {
            signal = -1;
            pvm_pkint(&signal, 1, 1);
         } 
         info = pvm_send(agent[currentAgentID]->myTid, 0);
         if(info < 0)
         {
	    cout << " Warning: It is fail to send broadcast data to "
	         << agent[currentAgentID]->hostName << endl;
	    break;
         }

 // send private data
          pvm_initsend(PvmDataDefault);

          jobptr = jobPool->GetJob(resendJobID);
          pvm_pkint(&(jobptr->type), 1, 1);
          pvm_pkint(&(jobptr->source), 1, 1);
          pvm_pkint(&(jobptr->destination), 1, 1);

          pvm_pkint(&resendJobID, 1, 1);
          localJobID = jobptr->GetLocalJobID();
          pvm_pkint(&localJobID, 1, 1);
          pvm_pkstr(jobptr->jobName);

	  bufptr = jobPool->GetJob(resendJobID)->privateSendBuffer;

          PackBuffer(bufptr);

          info = pvm_send(agent[currentAgentID]->myTid, 1);
          if(info < 0)
          {
	     cout << " Warning: It is fail to send private data to "
	          << agent[currentAgentID]->hostName << endl;
          }
          else
          {
	     machines[currentMachineID].InsertJobID(resendJobID);
	     machines[currentMachineID].status = SPAWNED;
	     machines[currentMachineID].jobCount++;

	     jobPool->GetJob(resendJobID)->agentID = currentAgentID;
	     jobPool->GetJob(resendJobID)->machineID = currentMachineID;
	     agent[currentAgentID]->jobID = resendJobID;
	     resendJobID++;
          }
          break;
       }
       resendJobID++;
    }
  }
  return 1;
}
   
 
int JobManager::Resend(MachineID currentMachineID, AgentID currentAgentID,
			JobID currentJobID, JobStatus resendJobStatus)
{
   int i;
   Tid tid;
   int numberOfSpawnedJobs;
   Buffer *bufptr;
   Info info;
   int signal;
   Job *jobptr;
   JobID localJobID;

// find the job to be sent
   for(i=0; i<jobPool->GetTotalJobNumber(); i++)
   {
       resendJobID = resendJobID%jobPool->GetTotalJobNumber();
       if(jobPool->GetJob(resendJobID)->jobStatus == resendJobStatus)
       {
	  if(strcmp(jobPool->GetJob(currentJobID)->GetJobName(),
              jobPool->GetJob(resendJobID)->GetJobName()) != 0)
	  {
	     pvm_kill(agent[currentAgentID]->myTid);
             numberOfSpawnedJobs = 
                         pvm_spawn(jobPool->GetJob(resendJobID)->GetJobName(),
	                 (char **)0, PvmTaskHost,
	                 agent[currentAgentID]->hostName,
		         1, &tid);
             if(numberOfSpawnedJobs == 1)
             { 
                agent[currentAgentID]->myTid = tid;
             }
	     else
	     {
	        cout << "spawn " << jobPool->GetJob(resendJobID)->GetJobName()
		     << " on " << agent[currentAgentID]->hostName 
		     << " fails." << endl;
                break;
             }
          }
	  
         // send broadcast data if any
         pvm_initsend(PvmDataDefault);

         if(broadcastBuffer != NULL)
         {
            signal = 1;
            pvm_pkint(&signal, 1, 1);
            PackBuffer(broadcastBuffer);
         }
         else
         {
            signal = -1;
            pvm_pkint(&signal, 1, 1);
         } 
         info = pvm_send(agent[currentAgentID]->myTid, 0);
         if(info < 0)
         {
	    cout << " Warning: It is fail to send broadcast data to "
	         << agent[currentAgentID]->hostName << endl;
	    break;
         }

 // send private data
          pvm_initsend(PvmDataDefault);

          jobptr = jobPool->GetJob(resendJobID);
          pvm_pkint(&(jobptr->type), 1, 1);
          pvm_pkint(&(jobptr->source), 1, 1);
          pvm_pkint(&(jobptr->destination), 1, 1);

          pvm_pkint(&resendJobID, 1, 1);
          localJobID = jobptr->GetLocalJobID();
          pvm_pkint(&localJobID, 1, 1);
          pvm_pkstr(jobptr->jobName);

	  bufptr = jobPool->GetJob(resendJobID)->privateSendBuffer;

          PackBuffer(bufptr);

          info = pvm_send(agent[currentAgentID]->myTid, 1);
          if(info < 0)
          {
	     cout << " Warning: It is fail to send private data to "
	          << agent[currentAgentID]->hostName << endl;
          }
          else
          {
	     machines[currentMachineID].InsertJobID(resendJobID);
	     machines[currentMachineID].status = SPAWNED;
	     jobPool->GetJob(resendJobID)->agentID = currentAgentID;
	     jobPool->GetJob(resendJobID)->machineID = currentMachineID;
	     agent[currentAgentID]->jobID = resendJobID;
	     resendJobID++;
          }
          return 1; // resending a job succeeded
       }
       resendJobID++;
    }
    return 0; // no jobs need to be resent
}
   
 
// Constructor of Class JobAgent
JobAgent::JobAgent()
{
   // get my tid
   myTid = pvm_mytid();

   // get parent's tid
   parentTid = pvm_parent();
}

Job* JobAgent::Receive()
{
   Job *job;
   Buffer *bufptr;
   int i;
   int info;

   job = new Job;


   //receive broadcast data
      pvm_recv(parentTid, 0);

      pvm_upkint(&info, 1, 1);

      if(info != -1)
      {
         // therte is broadcast data
         bufptr = job->broadcastReceiveBuffer;

         UnpackBuffer(bufptr);
      }

      // receive private data
      pvm_recv(parentTid, 1);

      pvm_upkint(&(job->type), 1, 1);
      pvm_upkint(&(job->source), 1, 1);
      pvm_upkint(&(job->destination), 1, 1);

      pvm_upkint(&currentJobID, 1, 1);
      pvm_upkint(&localJobID, 1, 1);
      pvm_upkstr(job->jobName);

      bufptr = job->privateReceiveBuffer;

      UnpackBuffer(bufptr);

      if(job->source != MAINPROGRAMADDRS)
	 currentDestination = job->source;
      else
	 currentDestination = parentTid;

      return job;
}

void JobAgent::Send(Result *result)
{
   int i;
   Buffer *bufptr;
   ResultID resultID = result->GetResultID();

   currentType = RESULT;
   currentSource = result->source;

   pvm_initsend(PvmDataDefault);

   pvm_pkint(&currentType, 1, 1);
   pvm_pkint(&currentSource, 1, 1);
   pvm_pkint(&currentDestination, 1, 1);

   pvm_pkint(&currentJobID, 1, 1);
   pvm_pkint(&localJobID, 1, 1);
   pvm_pkstr(result->resultName);

   bufptr = result->privateSendBuffer;

   PackBuffer(bufptr);

   pvm_send(currentDestination, 2);
}

void JobAgent::Run(Result* (*Worker)(Job *job))
{
   Job *jobptr;
   Result *resultptr;
   while(1)
   {
      // receive job from Manager
      jobptr = Receive();
      if(currentJobID == -1)
      {
	 // no more jobs
         delete jobptr;
         delete resultptr;
      }
      else
      {
	 // let Worker do its work
         resultptr = Worker(jobptr);

	 resultptr->SetResultID(jobptr->GetJobID());
	 resultptr->SetResultName(jobptr->GetJobName());

	 // send results to Manager
         Send(resultptr);

         delete jobptr;
         delete resultptr;
      }
   }
}

// Constructor of Class Machine
Machine::Machine()
{
    status = DEAD;
    jobCount = 0;
    machineID = -1;
    name = NULL;
    for(int i=0; i < MAXJOBNUMBER; i++)
       jobID[i] = -1;
}

// Insert a JobID
void Machine::InsertJobID(JobID currentJobID)
{
    jobID[jobCount] = currentJobID;
    jobCount++;
}

// Remove a JobID
int Machine::RemoveJobID(JobID currentJobID)
{
   int i;
   int tmpCount = 0;
   JobID tmpJobID;

   for(i=0; i<jobCount; i++)
   {
      tmpJobID = jobID[i];
      if(tmpJobID != currentJobID)  // remove currentJobID
      {
	 jobID[tmpCount] = tmpJobID; // put back jobID
	 tmpCount++;
      }
   }
   if(jobCount == tmpCount)
   {
      // tmpCount should be less than jobCount
      cout << "Something is wrong with Machine.jobCount" << endl;
      return 0;
   }
   else
   {
      jobCount = tmpCount; // update jobCount
      return 1;
   }
}

// Constructor of Class GroupManager
GroupManager::GroupManager(Job *job, JobNumber jobnum,
			   Buffer *bcbuffer, RunType runtype,
			   AddMode addmode)
{
   jobPool = new JobPool(job, jobnum);
   result = new Result[jobnum];
   broadcastBuffer = bcbuffer;
   machines = new Machine[MAXMACHINENUMBER];
   runType = runtype;
   addMode = addmode;
}

Result *GroupManager::Run()
{
// variables about hostname
   char tmpname[LENGTHOFHOSTNAME], *tmp1;
   int len;
   int nofile = 0;
   FILE *pp;

// variables about starting virtual machine and spawning jobs
   Tid mytid, tid;
   Tid tids[MAXJOBNUMBER];
   Info info;
   Info status[MAXAGENTNUMBER];
   int actjobNum;
   int numberOfSpawnedJobs;
   int numberOfAgents; // Number of JobAgents on one single machine
   int tmpnumberOfAgents;

// variables about virtual machine 
   int machineNum;
   int tmpMachineNum;
   MachineID currentMachineID;
   MachineList avamachineList;
   MachineList unavamachineList;
   AgentID currentAgentID;
   PackageType currentType;
   Tid currentSource;
   Tid currentDestination;
   JobID localJobID;

// variables about Job and JobPool
   JobID currentJobID;
   JobID nextJobID;
   JobID sendJobID;
   JobID tmpnextJobID;
   Buffer *bufptr;
   int signal; 
   int arrived;
   Job *jobptr;
   char jobName[LENGTHOFJOBNAME];
	  
   int i, j;

// get host names from file "hostname".
   machineNum = 0;

// open file "hostname"
   ifstream filein("hostname", ios::in);

   if(!filein)   
   {
      nofile = 1;
   }
   else
   {
      // read machine names from "hostname"
      while(!filein.eof())
      {
         filein >> tmpname;
         len = strlen((char *)tmpname);
         if(len != 0)
         {
	    tmp1 = new char[len + 1];
	    hostName[machineNum] = strncpy(tmp1, tmpname, len+1);
	    machineNum++;
         }
      }
      filein.close();
   }

   //get my name
   pp = popen("hostname", "r");
   fscanf(pp, "%s", &tmpname);
   pclose(pp);
   len = strlen((char *)tmpname);
   if(len != 0)
   {
      tmp1 = new char[len + 1];
      hostName[machineNum] = strncpy(tmp1, tmpname, len+1);
      machineNum++;
   }

// don't count myself twice and put myself at the end
   tmpMachineNum = machineNum - 1;
   machineNum = 0;
   for(i=0; i<tmpMachineNum; i++)
   {
      if(strcmp(hostName[i], hostName[tmpMachineNum]) != 0)
      {
         hostName[machineNum] = hostName[i];	 
	 machineNum++;
     }
   }
   hostName[machineNum] = hostName[tmpMachineNum];
   machineNum++;
 
// add machines into pvm
   for(i=0; i<machineNum - 1; i++)
   {
      info = pvm_addhosts((char **)&hostName[i], 1, status);

      if(info == 1 || (info == 0 && addMode == ANY))
      {
	 machines[i].status = ALIVE;
	 machines[i].name = hostName[i];
      }
      else
      {
	 machines[i].status = DEAD;
	 machines[i].name = hostName[i];
      }
   }

// add myself into pvm
      machines[machineNum-1].status = ALIVE;
      machines[machineNum-1].name = hostName[machineNum-1];
   

// How many jobs are spawned on one single machine simultaneously?
   switch(runType)
   {
      case NICE:
	 numberOfAgents = 1;
	 break;
      case NORMAL:
	 numberOfAgents = 2;
	 break;
      case FAST:
	 numberOfAgents = 3;
	 break;
      default:
	 numberOfAgents = 2;
	 break;
    }

   // spawn jobs
   actjobNum = 0;
   for(j=0; j<numberOfAgents; j++)
   {
      for(i=0; i<machineNum; i++)
      {
         if(actjobNum >= jobPool->GetTotalJobNumber()) break;
         if(machines[i].status != DEAD)
	 {
            numberOfSpawnedJobs = 
		       pvm_spawn(jobPool->GetJob(actjobNum)->GetJobName(),
		       (char **)0, PvmTaskHost,
		       machines[i].name, 1, &tid);
            if( numberOfSpawnedJobs == 1)
            {
	       machines[i].InsertJobID(actjobNum);
	       machines[i].status = SPAWNED;
	       machines[i].jobCount++;

               agent[actjobNum] = new AgentInfo;
               agent[actjobNum]->myTid = tid;
               agent[actjobNum]->hostName = hostName[i];
               agent[actjobNum]->jobID = actjobNum;
	       jobPool->GetJob(actjobNum)->agentID = actjobNum;
	       jobPool->GetJob(actjobNum)->machineID = i;
	       tids[actjobNum] = tid;
	       actjobNum++;
            }
	 }
      }
   }

   // next job to be sent
   nextJobID = actjobNum;

 // send data 
   for(j=0; j<actjobNum; j++)
   {
      // send broadcast data
      pvm_initsend(PvmDataDefault);

      if(broadcastBuffer != NULL)
      {
         // there is broadcast buffer
         signal = 1;
         pvm_pkint(&signal, 1, 1);
         PackBuffer(broadcastBuffer);
      }
      else
      {
         // there is not broadcast buffer
         signal = -1;
         pvm_pkint(&signal, 1, 1);
      } 

      info = pvm_send(agent[j]->myTid, 0);

      // send private data
      pvm_initsend(PvmDataDefault);

      jobptr = jobPool->GetJob(j);
      pvm_pkint(&(jobptr->type), 1, 1);
      pvm_pkint(&(jobptr->source), 1, 1);
      pvm_pkint(&(jobptr->destination), 1, 1);

      pvm_pkint(&j, 1, 1);
      localJobID = jobptr->GetLocalJobID();
      pvm_pkint(&localJobID, 1, 1);
      pvm_pkstr(jobptr->jobName);

      bufptr = jobPool->GetJob(j)->privateSendBuffer;

      PackBuffer(bufptr);

      info = pvm_send(agent[j]->myTid, 1);
      if(info < 0)
      {
//	 cout << " Warning: It is fail to send private data to "
//	      << agent[j]->hostName << endl;
      }
      else
      {
	 jobPool->GetJob(j)->jobStatus = SENT;
      }

   }

// Receive data from JobAgents

  while(jobPool->GetJobNumber() > 0)  // there is jobs uncompleted
  {
    arrived = pvm_nrecv(-1, 2); // detect whether there are data received
    if(arrived > 0)  
    {
      // there are data received
      pvm_upkint(&currentType, 1, 1);
      pvm_upkint(&currentSource, 1, 1);
      pvm_upkint(&currentDestination, 1, 1);

    if(currentType == RESULT || currentType == CONFIRMATION)
    {
      // find the current JobID
      pvm_upkint(&currentJobID, 1, 1);
      pvm_upkint(&localJobID, 1, 1);
      pvm_upkstr(jobName);

      // find the current AgentID
      currentAgentID = jobPool->GetJob(currentJobID)->agentID;

      // find the current MachineID
      currentMachineID = jobPool->GetJob(currentJobID)->machineID;
     
      // print out JobID and MachineID
       cout << "Received: JobID = " << currentJobID 
      	   << "  MachineID = " << currentMachineID << endl;

      if(jobPool->GetJob(currentJobID)->jobStatus == SENT)
      {
	 if(currentType == RESULT)
	 {
            // the current job is waiting for the results
            bufptr = result[currentJobID].privateReceiveBuffer;

	    // unpack pvm buffer to the received Buffer of Result
            UnpackBuffer(bufptr);
         }

         machines[currentMachineID].RemoveJobID(currentJobID);

	 // set the status of the current job FINISHED 
         jobPool->GetJob(currentJobID)->jobStatus = FINISHED;

	 // The size of JobPool is reduced by one
	 jobPool->ReduceJobNumberByOne();

	 if(jobPool->GetJobNumber() > 0)
	 {
	    // there is jobs unfinished
	    if(nextJobID < jobPool->GetTotalJobNumber())
	    {
	       sendJobID = nextJobID;

	       if(strcmp(jobPool->GetJob(currentJobID)->GetJobName(),
                         jobPool->GetJob(sendJobID)->GetJobName()) != 0)
	       {
		  // The job name to be sent is different from that of
		  // the current job so a new program to be spawned
		  pvm_kill(agent[currentAgentID]->myTid);
                  numberOfSpawnedJobs = 
                            pvm_spawn(jobPool->GetJob(sendJobID)->GetJobName(),
		                   (char **)0, PvmTaskHost,
		                   agent[currentAgentID]->hostName,
				   1, &tid);
                  if(numberOfSpawnedJobs == 1)
                  {
                     agent[currentAgentID]->myTid = tid;
                  }
		  else
		  {
//		     cout << "spawn " << jobPool->GetJob(sendJobID)->GetJobName()
//			  << " on " << agent[currentAgentID]->hostName 
//			  << " fails." << endl;
                     break;
                  }
               }
                    
                // send broadcast data if any
                pvm_initsend(PvmDataDefault);

                if(broadcastBuffer != NULL)
                {
                   signal = 1; // there is a broadcast Buffer
                   pvm_pkint(&signal, 1, 1);
                   PackBuffer(broadcastBuffer);
                }
                else
                {
                   signal = -1; // there is not any broadcast Buffer
                   pvm_pkint(&signal, 1, 1);
                } 
                info = pvm_send(agent[currentAgentID]->myTid, 0);
                if(info < 0)
                {
//	           cout << " Warning: It is fail to send broadcast data to "
//	                << agent[currentAgentID]->hostName << endl;
		   break;
                }

               // send private data
               pvm_initsend(PvmDataDefault);

               jobptr = jobPool->GetJob(sendJobID);
               pvm_pkint(&(jobptr->type), 1, 1);
               pvm_pkint(&(jobptr->source), 1, 1);
               pvm_pkint(&(jobptr->destination), 1, 1);

               pvm_pkint(&sendJobID, 1, 1);
               localJobID = jobptr->GetLocalJobID();
               pvm_pkint(&localJobID, 1, 1);
               pvm_pkstr(jobptr->jobName);

	       bufptr = jobPool->GetJob(sendJobID)->privateSendBuffer;

               PackBuffer(bufptr);

               info = pvm_send(agent[currentAgentID]->myTid, 1);
               if(info < 0)
               {
//	          cout << " Warning: It is fail to send private data to "
//	               << agent[currentAgentID]->hostName << endl;
               }
               else
               {
	          machines[currentMachineID].InsertJobID(sendJobID);
	          machines[currentMachineID].status = SPAWNED;

	          jobPool->GetJob(sendJobID)->jobStatus = SENT;
	          jobPool->GetJob(sendJobID)->agentID = currentAgentID;
	          jobPool->GetJob(sendJobID)->machineID = currentMachineID;
	          agent[currentAgentID]->jobID = sendJobID;
		  nextJobID++; // The sent job is not a unsent job so
		   	       // nextJobID is incresed by one.
               }
            }
	    else
	    {
	       pvm_kill(agent[currentAgentID]->myTid);
	       machines[currentMachineID].jobCount--;
            }
         }
      }
     }
     }
     arrived = pvm_nrecv(-1, 3);
     if(arrived > 0)  
     {
	pvm_upkint(&currentType, 1, 1);
	if(currentType == REQUEST)
	{
	   jobptr = new Job;

	   pvm_upkint(&currentSource, 1, 1);
	   pvm_upkint(&currentDestination, 1, 1);
	   pvm_upkint(&currentJobID, 1, 1);
	   pvm_upkint(&localJobID, 1, 1);

	   jobptr->type = currentType;
	   jobptr->source = currentSource;
	   jobptr->destination = currentDestination;

	   jobptr->SetLocalJobID(localJobID);

	   pvm_upkstr(jobName);
	   jobptr->SetJobName(jobName);

	   bufptr = jobptr->privateSendBuffer;
	   UnpackBuffer(bufptr);

	   // add the requested job into JobPool
	   jobPool->AddJob(jobptr);
        }
     }
  }

   // kill all of agents
   for(i=0; i<nextAgentID; i++)
      pvm_kill(agent[i]->myTid);

  // exit from pvm
  // pvm_exit();

  // remove jobs from JobPool
   for(i=0; i<actjobNum; i++)
      jobPool->RemoveJob((JobID)i);

   delete jobPool;
      
   return result; 
    
}

// Constructor of Class GroupManager
Distributor::Distributor(Job *job, JobNumber jobnum, Buffer *bcbuffer)
{
   jobPool = new JobPool(job, jobnum);
   result = new Result[jobnum];
   broadcastBuffer = bcbuffer;
}

Result *Distributor::Run()
{
    int jobCounter;
    int totalJobNum;
    Job *jobptr;
    Buffer *bufptr;
    int signal;
    Info info;
    PackageType currentType;
    Tid currentSource;
    Tid currentDestination;
    JobID currentJobID;
    JobID currentLocalJobID;
    JobID nextJobID;
    int arrived;
    char jobName[LENGTHOFJOBNAME];

    // variables about hostname
    char *hostName;
    char tmpname[LENGTHOFHOSTNAME], *tmp1;
    int  len;
    FILE *pp;

    Tid  tidOfMyWorker;

    Tid source = pvm_mytid();
    Tid destination = pvm_parent();

    totalJobNum = jobPool->GetTotalJobNumber();

    if(totalJobNum > 1)
    {
       for(jobCounter=0; jobCounter<totalJobNum-1; jobCounter++)
       {
          jobptr = jobPool->GetJob(jobCounter);
          jobptr->SetLocalJobID(jobCounter);
          jobptr->type = REQUEST;
          jobptr->source = source;
          jobptr->destination = destination;

          // send private data
          pvm_initsend(PvmDataDefault);

          pvm_pkint(&(jobptr->type), 1, 1);
          pvm_pkint(&(jobptr->source), 1, 1);
          pvm_pkint(&(jobptr->destination), 1, 1);

          pvm_pkint(&jobptr->jobID, 1, 1);
          pvm_pkint(&jobptr->localJobID, 1, 1);
          pvm_pkstr(jobptr->jobName);

          bufptr = jobptr->privateSendBuffer;

          PackBuffer(bufptr);

          info = pvm_send(destination, 3);
          if(info < 0)
          {
             cout << " Warning: It fails to send private data out " << endl;
          }
          else
          {
	     jobptr->jobStatus = SENT;
          }
       }
    }

   //get my name
   pp = popen("hostname", "r");
   fscanf(pp, "%s", &tmpname);
   pclose(pp);
   len = strlen((char *)tmpname);
   if(len != 0)
   {
      tmp1 = new char[len + 1];
      hostName = strncpy(tmp1, tmpname, len+1);
   }
 
   // spawn jon on self machine
   info = pvm_spawn(jobPool->GetJob(totalJobNum-1)->GetJobName(),
		    (char **)0, PvmTaskHost,
		    hostName, 1, &tidOfMyWorker);
   if( info != 1)
   {
       cout << "error when spawning job on self machine" << endl;
   }

   // set job properties
   jobptr = jobPool->GetJob(totalJobNum-1);
   jobptr->SetLocalJobID(totalJobNum-1);
   jobptr->type = REQUEST;
   jobptr->source = source;
   jobptr->destination = tidOfMyWorker;

   // send broadcast data if any
   pvm_initsend(PvmDataDefault);

   if(broadcastBuffer != NULL)
   {
        signal = 1; // there is a broadcast Buffer
        pvm_pkint(&signal, 1, 1);
        PackBuffer(broadcastBuffer);
   }
   else
   {
        signal = -1; // there is not any broadcast Buffer
        pvm_pkint(&signal, 1, 1);
   } 
   info = pvm_send(tidOfMyWorker, 0);
   if(info < 0)
   {
     cout << " Warning: It is fail to send broadcast data to "
          << hostName << endl;
   }

   // send private data
   pvm_initsend(PvmDataDefault);

   pvm_pkint(&(jobptr->type), 1, 1);
   pvm_pkint(&(jobptr->source), 1, 1);
   pvm_pkint(&(jobptr->destination), 1, 1);

   pvm_pkint(&jobptr->jobID, 1, 1);
   pvm_pkint(&jobptr->localJobID, 1, 1);
   pvm_pkstr(jobptr->jobName);

   bufptr = jobptr->privateSendBuffer;

   PackBuffer(bufptr);

   info = pvm_send(tidOfMyWorker, 1);
   if(info < 0)
   {
      cout << " Warning: It fails to send private data out " << endl;
   }
   else
   {
      jobptr->jobStatus = SENTTOSELF;
   }
  
	  
  nextJobID = 0;

  while(jobPool->GetJobNumber() > 0)  //there is jobs uncompleted
  {
     arrived = pvm_nrecv(-1, 2); //detect whether there are data received
     if(arrived > 0)  
     {
       // there are data received
       pvm_upkint(&currentType, 1, 1);
       pvm_upkint(&currentSource, 1, 1);
       pvm_upkint(&currentDestination, 1, 1);

       if(currentDestination == source && currentType == RESULT)
       {
	  pvm_upkint(&currentJobID, 1, 1);

	  pvm_upkint(&currentLocalJobID, 1, 1);
	  pvm_upkstr(jobName);

	  jobptr = jobPool->GetJob(currentLocalJobID);
	  if(jobptr->jobStatus == SENT || jobptr->jobStatus == SENTTOSELF)
	  {
	     bufptr = result[currentLocalJobID].privateReceiveBuffer;

	     UnpackBuffer(bufptr);

	     jobptr->jobStatus = FINISHED;

	     jobPool->ReduceJobNumberByOne();

	     if(jobptr->jobStatus == SENT)
	     {
                // send confirmation to JobManager
	        currentType = CONFIRMATION;

	        pvm_initsend(PvmDataDefault);

                pvm_pkint(&currentType, 1, 1);
                pvm_pkint(&source, 1, 1);
                pvm_pkint(&destination, 1, 1);

	        pvm_pkint(&currentJobID, 1, 1);

	        pvm_pkint(&currentLocalJobID, 1, 1);
	        pvm_pkstr(jobName);

	        info = pvm_send(destination, 2);
                if(info < 0)
                {
                   cout << " Warning: It fails to send CONFIRMATION out "<< endl;
                }
              }
              else
	      {
		 jobptr = jobPool->GetJob(nextJobID);
		 if(jobptr->jobStatus == SENT)
		 {
                    // set job properties
                    jobptr->destination = tidOfMyWorker;

	            if(strcmp(jobPool->GetJob(currentLocalJobID)->GetJobName(),
                         jobPool->GetJob(nextJobID)->GetJobName()) != 0)
	            {
		        // The job name to be sent is different from that of
		        // the current job so a new program to be spawned
		        pvm_kill(tidOfMyWorker);
                       info = pvm_spawn(jobPool->GetJob(nextJobID)->GetJobName(),
		                        (char **)0, PvmTaskHost,
		                        hostName, 1, &tidOfMyWorker);
                       if(info != 1)
		       {
		          cout << "error when spawning a job" << endl;
                       }
                    }

                    // send broadcast data if any
                    pvm_initsend(PvmDataDefault);

                    if(broadcastBuffer != NULL)
                    {
                       signal = 1; // there is a broadcast Buffer
                       pvm_pkint(&signal, 1, 1);
                       PackBuffer(broadcastBuffer);
                    }
                    else
                    {
                       signal = -1; // there is not any broadcast Buffer
                       pvm_pkint(&signal, 1, 1);
                    } 
                    info = pvm_send(tidOfMyWorker, 0);
                    if(info < 0)
                    {
                        cout << " Warning: It is fail to send broadcast data to "
                             << hostName << endl;
                    }

                    // send private data
                    pvm_initsend(PvmDataDefault);

                    pvm_pkint(&(jobptr->type), 1, 1);
                    pvm_pkint(&(jobptr->source), 1, 1);
                    pvm_pkint(&(jobptr->destination), 1, 1);

                    pvm_pkint(&jobptr->jobID, 1, 1);
                    pvm_pkint(&jobptr->localJobID, 1, 1);
                    pvm_pkstr(jobptr->jobName);

                    bufptr = jobptr->privateSendBuffer;

                    PackBuffer(bufptr);

                    info = pvm_send(tidOfMyWorker, 1);
                    if(info < 0)
                    {
                       cout << " Warning: It fails to send private data out "
			    << endl;
                    }
                    else
                    {
                       jobptr->jobStatus = SENTTOSELF;
                    }
                 }
		 nextJobID = (nextJobID++)%totalJobNum;
	      } 
           }
         }
      }
   }
   return result;
}
