/* JOBPOOL.C */

/* function definitions */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "pvm3.h"
#include "jobpool_c.h"
#include "machinelist_c.h"


/* Utility Function to pack Buffer to PVM buffer */
void PackBuffer(Buffer *bufptr)
{

   int i;

/* Pack integers */
   pvm_pkint(&(bufptr->intcnt), 1, 1);
   if(bufptr->intcnt != 0)
      for(i=0; i<bufptr->intcnt; i++)
      {
         pvm_pkint(&(bufptr->intnum[i]), 1, 1);
	 pvm_pkint(bufptr->intptr[i], bufptr->intnum[i], 1);
      }

/* Pack floats */
   pvm_pkint(&(bufptr->floatcnt), 1, 1);
   if(bufptr->floatcnt != 0)
      for(i=0; i<bufptr->floatcnt; i++)
      {
         pvm_pkint(&(bufptr->floatnum[i]), 1, 1);
	 pvm_pkfloat(bufptr->floatptr[i], bufptr->floatnum[i], 1);
      }

/* Pack strings */
   pvm_pkint(&(bufptr->strcnt), 1, 1);
   if(bufptr->strcnt != 0)
      for(i=0; i<bufptr->strcnt; i++)
      {
         pvm_pkstr(bufptr->strptr[i]);
      }
}

/* Utility Function to unpack Buffer from PVM buffer */
void UnpackBuffer(Buffer *bufptr)
{
   int i;

/* Unpack integers */
   pvm_upkint(&(bufptr->intcnt), 1, 1);
   if(bufptr->intcnt != 0)
       for(i=0; i<bufptr->intcnt; i++)
       {
          pvm_upkint(&(bufptr->intnum[i]), 1, 1);
          bufptr->intptr[i] = malloc( bufptr->intnum[i]*sizeof(int));
	  pvm_upkint(bufptr->intptr[i], bufptr->intnum[i], 1);
       }

/* Unpack floats */
   pvm_upkint(&(bufptr->floatcnt), 1, 1);
   if(bufptr->floatcnt != 0)
      for(i=0; i<bufptr->floatcnt; i++)
      {
         pvm_upkint(&(bufptr->floatnum[i]), 1, 1);
         bufptr->floatptr[i] = malloc( bufptr->intnum[i]*sizeof(float));
         pvm_upkfloat(bufptr->floatptr[i], bufptr->floatnum[i], 1);
      }

/* Unpack strings */
   pvm_upkint(&(bufptr->strcnt), 1, 1);
   if(bufptr->strcnt != 0)
      for(i=0; i<bufptr->strcnt; i++)
      {
         bufptr->strptr[i] = malloc(MAXSTRLENGTH*sizeof(char));
         pvm_upkstr(bufptr->strptr[i]);
      }
}

/* Initialize Buffer with optional storage type */
void InitBuffer(Buffer *buffer, enum StorageType buffertype){
   buffer->BufferType = buffertype;
   buffer->intcnt = 0;
   buffer->intrev = 0;
   buffer->floatcnt = 0;
   buffer->floatrev = 0;
   buffer->strcnt = 0;
   buffer->strrev = 0;
}

/* Pack integers into Buffer */
void PackInt(Buffer *buffer, int *np, int cnt){
   int i;
   if(buffer->BufferType == IN_PLACE)
   {
      buffer->intptr[buffer->intcnt] = np;
   }
   else
   {
      buffer->intptr[buffer->intcnt] = malloc(cnt*sizeof(int));
      for(i=0; i<cnt; i++)
      {
	 buffer->intptr[buffer->intcnt][i] = *np;
	 np++;
      }
   }
   buffer->intnum[buffer->intcnt] = cnt;
   buffer->intcnt++;
}

/* Pack floats into Buffer */
void PackFloat(Buffer *buffer, float *fp, int cnt){
   int i;
   if(buffer->BufferType == IN_PLACE)
   {
      buffer->floatptr[buffer->floatcnt] = fp;
   }
   else
   {
      buffer->floatptr[buffer->floatcnt] = malloc(cnt*sizeof(float));
      for(i=0; i<cnt; i++)
      {
	 buffer->floatptr[buffer->floatcnt][i] = *fp;
	 fp++;
      }
   }
   buffer->floatnum[buffer->floatcnt] = cnt;
   buffer->floatcnt++;
}

/* Pack strings into Buffer */
void PackString(Buffer *buffer, char *cp){
   int i;
   int n = 0;
   if(buffer->BufferType == IN_PLACE)
   {
      buffer->strptr[buffer->strcnt] = cp;
   }
   else
   {
     // How mang characters?
      while(cp[n] != '\0')
      {
	 n++;
      }
      buffer->strptr[buffer->strcnt] = malloc((n+1)*sizeof(char));
      for(i=0; i<n; i++)
      {
	 buffer->strptr[buffer->strcnt][i] = *cp;
	 cp++;
      }
      buffer->strptr[buffer->strcnt][n] = '\0';
   }
   buffer->strnum[buffer->strcnt] = n;
   buffer->strcnt++;
}

/* Unpack intergers from Buffer */
void UnpackInt(Buffer *buffer, int *np, int cnt){
   int i;
   for(i=0; i<cnt; i++)
   {
      np[i] = buffer->intptr[buffer->intrev][i];
   }
   buffer->intrev++;
}

/* Unpack floats from Buffer */
void UnpackFloat(Buffer *buffer, float *fp, int cnt){
   int i;
   for(i=0; i<cnt; i++)
   {
      fp[i] = buffer->floatptr[buffer->floatrev][i];
   }
   buffer->floatrev++;
}

/* Unpack strings from Buffer */
void UnpackString(Buffer *buffer, char *cp){
   int i = 0;
   while(buffer->strptr[buffer->strrev][i] != '\0')
   {
      cp[i] = buffer->strptr[buffer->strrev][i];
      i++;
   }
   cp[i] = '\0';
   buffer->strrev++;
}

/* Initialize Job with optional storage type */
void InitJob( Job *job, enum StorageType bufferType ){
   job->type = REQUEST;
   job->source = MAINPROGRAMADDRS;
   job->destination = MAINPROGRAMADDRS;
   job->jobID = -1;
   job->jobName = "";
   job->jobStatus = INITIAL;
   job->broadcastReceiveBuffer = malloc(sizeof(Buffer));
   InitBuffer(job->broadcastReceiveBuffer, bufferType);
   job->privateReceiveBuffer = malloc(sizeof(Buffer));
   InitBuffer(job->privateReceiveBuffer, bufferType);
   job->privateSendBuffer = malloc(sizeof(Buffer));
   InitBuffer(job->privateSendBuffer, bufferType);
}

/* Set and Get functions of Job */
void SetJobID(Job *job, JobID jobid){ job->jobID = jobid; }

void SetLocalJobID(Job *job, JobID jobid){ job->localJobID = jobid; }

void SetJobName(Job *job, char *jobname){ job->jobName = jobname; }

JobID GetJobID(Job *job){ return job->jobID; }

JobID GetLocalJobID(Job *job){ return job->localJobID; }

char* GetJobName(Job *job){ return job->jobName; }

/* Initialize Result with optional storage type */
void InitResult(Result *result, enum StorageType bufferType){
   result->type = REQUEST;
   result->source = MAINPROGRAMADDRS;
   result->destination = MAINPROGRAMADDRS;
   result->resultID = -1;
   result->resultName = "";
   result->broadcastReceiveBuffer = malloc(sizeof(Buffer));
   InitBuffer(result->broadcastReceiveBuffer, bufferType);
   result->privateReceiveBuffer = malloc(sizeof(Buffer));
   InitBuffer(result->privateReceiveBuffer, bufferType);
   result->privateSendBuffer = malloc(sizeof(Buffer));
   InitBuffer(result->privateSendBuffer, bufferType);
}

// Set and Get functions of Class Result
void SetResultID(Result *result, JobID jobID){ result->resultID = jobID; }

void SetLocalResultID(Result *result, JobID jobID){result->localResultID = jobID;}

void SetResultName(Result *result, char *jobname){ result->resultName = jobname; }

ResultID GetLocalResultID(Result *result){ return result->localResultID; }

ResultID GetResultID(Result *result){ return result->resultID; }

char* GetResultName(Result *result){ return result->resultName; }

/* Initialize JobPool */
void InitJobPool(JobPool *jobPool, Job *job0, int jobnum,
		 enum StorageType buffertype)
{
   int i;

   jobPool->jobNumber = 0;
   jobPool->nextJobID = 0;
   jobPool->totalJobNum = 0;

/* add Jobs into JobPool */
   for(i=0; i<jobnum; i++)
   {
      AddJob(jobPool, &job0[i], buffertype);
   }
   jobPool->userJobNum = jobPool->totalJobNum;   
}

/* Get functions of JobPool */
JobNumber GetJobNumber(JobPool *jobPool){ return jobPool->jobNumber; }
JobNumber GetTotalJobNumber(JobPool *jobPool){ return jobPool->totalJobNum; }
JobNumber GetUserJobNumber(JobPool *jobPool){ return jobPool->userJobNum; }
Job* GetJob(JobPool *jobPool, JobNumber i){ return jobPool->job[i]; }

/* Add one more Job into JobPool */ 
int AddJob( JobPool *jobPool, Job *jobptr, enum StorageType buffertype ){

   jobPool->job[jobPool->nextJobID] = malloc(sizeof(Job));
   InitJob(jobPool->job[jobPool->nextJobID], buffertype);
   jobPool->job[jobPool->nextJobID]->type = jobptr->type;
   jobPool->job[jobPool->nextJobID]->source = jobptr->source;
   jobPool->job[jobPool->nextJobID]->destination = jobptr->destination;
   jobPool->job[jobPool->nextJobID]->localJobID = jobptr->localJobID;
   SetJobID(jobPool->job[jobPool->nextJobID], jobPool->nextJobID );
   SetJobName(jobPool->job[jobPool->nextJobID], GetJobName(jobptr) );
   jobPool->job[jobPool->nextJobID]->privateSendBuffer
					  = jobptr->privateSendBuffer;

   jobPool->jobNumber++;
   jobPool->totalJobNum++;
   jobPool->nextJobID++;
}

/* Remove one Job from JobPool */
void RemoveJob(JobPool *jobPool, JobID jobid )
{
   free(jobPool->job[jobid]);
}

/* Reduce Job counter by one */
void ReduceJobNumberByOne(JobPool *jobPool){ jobPool->jobNumber--; }
void ReduceUserJobNumberByOne(JobPool *jobPool){ jobPool->userJobNum--; }

/* Initialize JobManager */
void InitJobManager(JobManager *jobManager, Job *job, JobNumber jobnum, 
		    Buffer *bcbuffer, enum RunType runtype,  
                    enum Reliability reliable, enum RunMode runmode, 
		    CheckTime dtime)
{
   int counter;

   jobManager->jobPool = malloc(sizeof(JobPool));
   InitJobPool(jobManager->jobPool, job, jobnum, OUTOF_PLACE);
   jobManager->result = malloc(sizeof(Result)*jobnum);
   for(counter=0; counter<jobnum; counter++)
      InitResult(&(jobManager->result[counter]), OUTOF_PLACE); 
   jobManager->broadcastBuffer = bcbuffer;
   jobManager->machines = malloc(sizeof(Machine)*MAXMACHINENUMBER);
   for(counter=0; counter<MAXMACHINENUMBER; counter++)
      InitMachine(&(jobManager->machines[counter]));
   jobManager->runType = runtype;
   jobManager->reliability = reliable;
   jobManager->resendJobID = 0;
   jobManager->runMode = runmode;
   jobManager->checktime = dtime;
}

void EndJobManager(JobManager *jobManager)
{
   if(jobManager->runMode == AUTOSTART)
   {
      printf("Information: pvm has halted.\n");
      pvm_halt();
   }
}

Result* RunJobManager(JobManager *jobManager)
{
/* variables about hostname */
   FILE *fp;
   char tmpname[LENGTHOFHOSTNAME], *tmp1;
   int len;

/* variables about starting virtual machine and spawning jobs */
   Tid mytid, tid;
   Tid tids[MAXJOBNUMBER];
   Info info;
   Info status[MAXAGENTNUMBER];
   int actjobNum;
   int numberOfSpawnedJobs;
   int numberOfAgents; // Number of JobAgents on one single machine
   int tmpnumberOfAgents;
   int knext;

/* variables about virtual machine  */
   int machineNum;
   MachineID currentMachineID;
   MachineList *avamachineList;
   MachineList *unavamachineList;
   AgentID currentAgentID;
   PackageType currentType;
   Tid currentSource;
   Tid currentDestination;

/* variables about Job and JobPool */
   JobID currentJobID;
   JobID nextJobID;
   JobID sendJobID;
   JobID tmpnextJobID;
   Buffer *bufptr;
   int signal;
   Job *jobptr;
   int localJobID;
   char jobName[LENGTHOFJOBNAME];

/* variables about timer */ 
   time_t starttime;
   time_t endtime;
   double diftime;
   int times = 0;

/* variables about getting OS name */ 
   FILE *pp;
   char osname[20];

/* variables about "ping" */
   char *cmd;
	  
   int i, j;

/* get host names from file "hostname". */
/* open file "hostname" */
   fp = fopen("hostname", "r");

   if(ferror(fp))   
   {
      printf("file 'hostname' could not be opened \n");
      exit(1);
   }

/* read machine names from "hostname" */
   machineNum = 0;
   while(!feof(fp))
   {
      fscanf(fp, "%s", tmpname);
      len = strlen((char *)tmpname);
      if(len != 0)
      {
	 tmp1 = malloc(sizeof(char)*(len + 1));
	 jobManager->hostName[machineNum] = 
			     strncpy(tmp1, tmpname, len+1);
	 machineNum++;
      }
   }
   fclose(fp);

   machineNum--;

/* print out machine names */
   printf("The hostname file includes the following hosts:\n"); 
   for(i=0; i<machineNum; i++)
   {
      printf("%s\n", jobManager->hostName[i]);
   }
      printf("\n");

/* detect the name of operating system */
   pp = popen("uname -s", "r");
   fscanf(pp,"%s",&osname);
   pclose(pp);
   printf("The Operating System is %s\n\n", osname);

   avamachineList = (MachineList *) malloc(sizeof(MachineList));
   InitMachineList(avamachineList);
   unavamachineList = (MachineList *) malloc(sizeof(MachineList));
   InitMachineList(unavamachineList);

/* "ping" all of machines */
   for(i=0; i<machineNum; i++)
   {
      char cmdline[LENGTHOFHOSTNAME+10] = "ping ";
      cmd = strcat(cmdline, jobManager->hostName[i]);
      if(strcmp(osname, "HP-UX") == 0)
         cmd = strcat(cmdline, " 8 1");

      info = system(cmd);

      jobManager->machines[i].name = jobManager->hostName[i];
      jobManager->machines[i].machineID = i;

      if(info == 0)
      {
         jobManager->machines[i].status = ALIVE;
	 /* insert one machine into available machine list */
	 AddNewMachine(avamachineList, &(jobManager->machines[i]));
      }
      else
      {
         jobManager->machines[i].status = DEAD;
	 /* insert one machine into unavailable machine list */
	 AddNewMachine(unavamachineList, &(jobManager->machines[i]));
      }
    }
 
 /* detect whether pvm has been started */
   mytid = pvm_mytid();

/* if runmode is AUTUSTART, pvm can be started here. */
  if(jobManager->runMode == AUTOSTART )  
  {
    if(mytid < 0)
    {
       printf("\n Don't worry. I will start pvmd for you.\n\n");

       // start pvm
       info = pvm_start_pvmd(0, (char **)0, 1);

       if(info != 0)
       {
          printf("pvm_start_pvmd() fails \n");
          exit(0);
       }

       mytid = pvm_mytid();
    }

/* add machines into pvm */
   info = pvm_addhosts((char **)jobManager->hostName, 
		       machineNum, status);

   if(info < 1)
      printf("Warning: No hosts are added\n");
   else 
      printf("Actual added host number: %d \n", info+1);
  }

/* How many jobs are spawned on one single machine simultaneously? */
   switch(jobManager->runType)
   {
      case NICE:
	 numberOfAgents = 1;
	 break;
      case NORMAL:
	 numberOfAgents = 2;
	 break;
      case FAST:
	 numberOfAgents = 3;
	 break;
      default:
	 numberOfAgents = 2;
	 break;
    }

   /* spawn jobs */
   actjobNum = 0;
   for(j=0; j<numberOfAgents; j++)
   {
      for(i=0; i<machineNum; i++)
      {
         if(actjobNum >= GetTotalJobNumber(jobManager->jobPool)) break;
         if(jobManager->machines[i].status != DEAD)
	 {
            numberOfSpawnedJobs = 
		 pvm_spawn(GetJobName(jobManager->jobPool->job[actjobNum]),
		       (char **)0, PvmTaskHost,
		       jobManager->machines[i].name, 1, &tid);
            if( numberOfSpawnedJobs == 1)
            {
	       InsertJobID(&(jobManager->machines[i]), actjobNum);
	       jobManager->machines[i].status = SPAWNED;
	       jobManager->machines[i].jobCount++;

               jobManager->agent[actjobNum] = 
				  (AgentInfo *) malloc(sizeof(AgentInfo));
               jobManager->agent[actjobNum]->myTid = tid;
               jobManager->agent[actjobNum]->hostName = jobManager->hostName[i];
               jobManager->agent[actjobNum]->jobID = actjobNum;
	       jobManager->jobPool->job[actjobNum]->agentID = actjobNum;
	       jobManager->jobPool->job[actjobNum]->machineID = i;
	       tids[actjobNum] = tid;
	       actjobNum++;
            }
	 }
      }
   }

   /* next job to be sent */
   nextJobID = actjobNum;

 /* Broadcast data if any */
 pvm_initsend(PvmDataDefault);

 if(jobManager->broadcastBuffer != NULL)
 {
   /* there is broadcast buffer */
   signal = 1;
   pvm_pkint(&signal, 1, 1);
   PackBuffer(jobManager->broadcastBuffer);
 }
 else
 {
   /* there is not broadcast buffer */
   signal = -1;
   pvm_pkint(&signal, 1, 1);
 } 
 pvm_mcast(tids, actjobNum, 0);

 /* Send private data */ 
   for(j=0; j<actjobNum; j++)
   {
      pvm_initsend(PvmDataDefault);

      jobptr = jobManager->jobPool->job[j];
      pvm_pkint(&(jobptr->type), 1, 1);
      pvm_pkint(&(jobptr->source), 1, 1);
      pvm_pkint(&(jobptr->destination), 1, 1);

      pvm_pkint(&j, 1, 1);

      localJobID = jobptr->localJobID;
      pvm_pkint(&localJobID, 1, 1);
      pvm_pkstr(jobptr->jobName);

      bufptr = jobptr->privateSendBuffer;

      PackBuffer(bufptr);

      info = pvm_send(jobManager->agent[j]->myTid, 1);
      if(info < 0)
      {
	 printf(" Warning: It is fail to send private data to \n");
	 printf("%s \n", jobManager->agent[j]->hostName);
      }
      else
      {
	 jobptr->jobStatus = SENT;
      }

   }

/* Receive data from JobAgents */
/* initialize timer for checking machine status */
  starttime = time(0);

  while(GetUserJobNumber(jobManager->jobPool) > 0)
  {
    /* there is jobs uncompleted */
    int arrived;
    arrived = pvm_nrecv(-1, 2); /* detect whether there are data received */
    if(arrived <= 0)  
    {
       /* there are not data received so check status of machines  */
       Machine *tmpmachine;
       MachineID tmpmachineID;
       int tmpjobCount;

       endtime = time(0);
       if(difftime(endtime, starttime) < (double)jobManager->checktime)
	  continue; /* time interval is less than checktime so skip */

/* check status of machines */
       if(!isEmpty(avamachineList))
       {
	  /* check list of available machines */
          char cmdline[LENGTHOFHOSTNAME+10] = "ping ";

          /* remove a machine from the available list */ 
          tmpmachine = RemoveMachine(avamachineList);
	  tmpmachineID = tmpmachine->machineID;

          cmd = strcat(cmdline, tmpmachine->name);
	  if(strcmp(osname, "HP-UX") == 0)  /* for HP-UX machines */
	     cmd = strcat(cmdline, " 8 1");

          /* "ping" a machine */
          info = system(cmd);
	  if(info == 0)
	  {
	     /* the machine is still alive */
	     AddNewMachine(avamachineList, &(jobManager->machines[tmpmachineID]));
          }
	  else
	  {
	   /* the machine was died so add the machine to the unavailable list */
	     AddNewMachine(unavamachineList, &(jobManager->machines[tmpmachineID]));

	     /* find which jobs should be reassigned in the JobPool */
	     tmpjobCount = jobManager->machines[tmpmachineID].jobCount;
	     for(i=0; i<tmpjobCount; i++)
	     {
	        jobManager->jobPool->job[jobManager->
		       machines[tmpmachineID].jobID[i]]->jobStatus = INITIAL;
		jobManager->machines[tmpmachineID].jobID[i] = -1;
             }
	     jobManager->machines[tmpmachineID].jobCount = 0;
	     jobManager->machines[tmpmachineID].status = DEAD;
          }
       }
       if(!isEmpty(unavamachineList))
       {
          char cmdline[LENGTHOFHOSTNAME+10] = "ping ";

	  /* check the unavailable list of machines
	     the checktime for unavailable list is 10 times of that for 
	     available list  */
	  times++;
	  if(times > 10)
	     times = 0;
          else
	     continue; /* it is not a check time so skip */

	  /* remove a machine from the list of unavailable machines */
          tmpmachine = RemoveMachine(unavamachineList);
	  tmpmachineID = tmpmachine->machineID;

          cmd = strcat(cmdline, tmpmachine->name);
	  if(strcmp(osname, "HP-UX") == 0)
	     cmd = strcat(cmdline, " 8 1");  /* for HP-UX machines */

	  /* "ping" a machine */
          info = system(cmd);
	  if(info != 0)
	  {
	     /* the machine is still unavailable */
	     AddNewMachine(unavamachineList, &(jobManager->machines[tmpmachineID]));
          }
	  else
	  {
	     /* the machine was born */   
	     jobManager->machines[tmpmachineID].status = ALIVE;
	     jobManager->machines[tmpmachineID].jobCount = 0;
	     /* add the machine to the list of available machines */
	     AddNewMachine(avamachineList, &(jobManager->machines[tmpmachineID]));

	     /* Send an initial job to the new-born machine */
	     SendToNewHost(jobManager, tmpmachineID, numberOfAgents, INITIAL);
	     tmpnumberOfAgents = numberOfAgents - 
			jobManager->machines[tmpmachineID].jobCount;
	     if(tmpnumberOfAgents > 0)
	     {
	        /* Send jobs to the new-born machine */
	        SendToNewHost(jobManager, tmpmachineID, tmpnumberOfAgents, SENT);
             }
          }
       }

       // Initialize timer 
       starttime = time(0);
    }
    else
    {
      // there are data received
      pvm_upkint(&currentType, 1, 1);
      pvm_upkint(&currentSource, 1, 1);
      pvm_upkint(&currentDestination, 1, 1);

    if(currentType == RESULT || currentType == CONFIRMATION)
    {
      /* get the current JobID */
      pvm_upkint(&currentJobID, 1, 1);
      pvm_upkint(&localJobID, 1, 1);
      pvm_upkstr(jobName);

      /* find the current AgentID */
      currentAgentID = jobManager->jobPool->job[currentJobID]->agentID;

      /* find the current MachineID */
      currentMachineID = jobManager->jobPool->job[currentJobID]->machineID;
     
      /* print out JobID and MachineID */
      printf("Received: JobID = %d ", currentJobID); 
      printf("  MachineID = %d \n", currentMachineID);

      if(jobManager->jobPool->job[currentJobID]->jobStatus == SENT)
      {
	 if(currentType == RESULT)
	 {
            /* the current job is waiting for the results */
            bufptr = jobManager->result[currentJobID].privateReceiveBuffer;

	    /* unpack pvm buffer to the received Buffer of Result */
            UnpackBuffer(bufptr);

	    ReduceUserJobNumberByOne(jobManager->jobPool);
         }

         RemoveJobID(&(jobManager->machines[currentMachineID]), currentJobID);

	 /* set the status of the current job FINISHED */ 
         jobManager->jobPool->job[currentJobID]->jobStatus = FINISHED;

	 /* The size of JobPool is reduced by one */
	 ReduceJobNumberByOne(jobManager->jobPool);

	 if(GetJobNumber(jobManager->jobPool) > 0)
	 {
	    /* there is jobs unfinished */
	    if(nextJobID < GetTotalJobNumber(jobManager->jobPool))
	    {
	       tmpnextJobID = nextJobID;
	       while(jobManager->jobPool->job[nextJobID]->jobStatus != INITIAL)
	       {
		  /* Is there any unsent jobs whose ID is larger than
		     next jobID? */
		  nextJobID++;
		  if(nextJobID >= GetTotalJobNumber(jobManager->jobPool))
		  {
		     nextJobID = tmpnextJobID;
                     break;
                  }
               }

	       sendJobID = nextJobID;
	       knext = 1;
	       for(i=0; i<nextJobID; i++)
	       {
		  if(jobManager->jobPool->job[i]->jobStatus == INITIAL)
		  {
		     /* There is a killed jobs whose ID is less than
		        next jobID. So send it first. */
		     sendJobID = i;
		     knext = 0;
		     break;
                  }
               }

	       if(strcmp(GetJobName(jobManager->jobPool->job[currentJobID]),
                         GetJobName(jobManager->jobPool->job[sendJobID])) != 0)
	       {
		  /* The job name to be sent is different from that of
		     the current job so a new program to be spawned */
		  pvm_kill(jobManager->agent[currentAgentID]->myTid);
                  numberOfSpawnedJobs = 
                     pvm_spawn(GetJobName(jobManager->jobPool->job[sendJobID]),
		               (char **)0, PvmTaskHost,
		               jobManager->agent[currentAgentID]->hostName,
			       1, &tid);
                  if(numberOfSpawnedJobs == 1)
                  {
                     jobManager->agent[currentAgentID]->myTid = tid;
                  }
		  else
		  {
		     printf("spawn %s", GetJobName(jobManager->jobPool->
			    job[sendJobID]),
			    " on %s", jobManager->agent[currentAgentID]->
			    hostName,  
			    " fails. \n");
                     break;
                  }
               }
                    
                /* send broadcast data if any */
                pvm_initsend(PvmDataDefault);

                if(jobManager->broadcastBuffer != NULL)
                {
                   signal = 1; /* there is a broadcast Buffer */
                   pvm_pkint(&signal, 1, 1);
                   PackBuffer(jobManager->broadcastBuffer);
                }
                else
                {
                   signal = -1; /* there is not any broadcast Buffer */
                   pvm_pkint(&signal, 1, 1);
                } 
                info = pvm_send(jobManager->agent[currentAgentID]->myTid, 0);
                if(info < 0)
                {
	           printf(" Warning: It is fail to send broadcast data to %s",
	                  jobManager->agent[currentAgentID]->hostName);
		   break;
                }

               /* send private data */
               pvm_initsend(PvmDataDefault);

               jobptr = jobManager->jobPool->job[sendJobID];
               pvm_pkint(&(jobptr->type), 1, 1);
               pvm_pkint(&(jobptr->source), 1, 1);
               pvm_pkint(&(jobptr->destination), 1, 1);

               pvm_pkint(&sendJobID, 1, 1);

	       localJobID = GetLocalJobID(jobptr);
               pvm_pkint(&localJobID, 1, 1);
               pvm_pkstr(jobptr->jobName);

	       bufptr = jobManager->jobPool->job[sendJobID]->privateSendBuffer;

               PackBuffer(bufptr);

               info = pvm_send(jobManager->agent[currentAgentID]->myTid, 1);
               if(info < 0)
               {
	          printf(" Warning: It is fail to send private data to %s %n",
	                 jobManager->agent[currentAgentID]->hostName);
               }
               else
               {
	          InsertJobID(&(jobManager->machines[currentMachineID]),
			      sendJobID);
	          jobManager->machines[currentMachineID].status = SPAWNED;

	          jobManager->jobPool->job[sendJobID]->jobStatus = SENT;
	          jobManager->jobPool->job[sendJobID]->agentID = currentAgentID;
	          jobManager->jobPool->job[sendJobID]->machineID 
							 = currentMachineID;
	          jobManager->agent[currentAgentID]->jobID = sendJobID;
		  if(knext == 1)
		     nextJobID++; /* The sent job is not a unsent job so */
				  /* nextJobID is incresed by one. */
               }
            }
	    else
	    {
	       if(Resend(jobManager, currentMachineID, currentAgentID, 
			 currentJobID, INITIAL) == 0)
               {
		  /* All of jobs have been sent so reduce job number on
		     each machine to one. */
	          if(jobManager->machines[currentMachineID].jobCount > 1)
	          {
		     pvm_kill(jobManager->agent[currentAgentID]->myTid);
		     jobManager->machines[currentMachineID].jobCount--;
                  }
	          else
	          {
		     /* Resend sent jobs if high reliability is expected. */
		     if(jobManager->reliability == HIGH) 
		        Resend(jobManager, currentMachineID, currentAgentID, 
			 currentJobID, SENT);
                  }
	       }
            }
         }
      }
     }
     }
     arrived = pvm_nrecv(-1, 3);
     if(arrived > 0)  
     {
	pvm_upkint(&currentType, 1, 1);
	if(currentType == REQUEST)
	{
	   jobptr = (Job *) malloc(sizeof(Job));
	   InitJob(jobptr, OUTOF_PLACE);

	   pvm_upkint(&currentSource, 1, 1);
	   pvm_upkint(&currentDestination, 1, 1);
	   pvm_upkint(&currentJobID, 1, 1);
	   pvm_upkint(&localJobID, 1, 1);

	   jobptr->type = currentType;
	   jobptr->source = currentSource;
	   jobptr->destination = currentDestination;

	   SetLocalJobID(jobptr, localJobID);

	   pvm_upkstr(jobName);
	   SetJobName(jobptr, jobName);

	   bufptr = jobptr->privateSendBuffer;
	   UnpackBuffer(bufptr);

	   /* add the requested job into JobPool */
	   AddJob(jobManager->jobPool, jobptr, 1);
        }
     }
  }

   /* kill all of agents */
   for(i=0; i<jobManager->nextAgentID; i++)
      pvm_kill(jobManager->agent[i]->myTid);

   /* delete all of machines in the parallel virtual machine */
   if(jobManager->runMode == AUTOSTART)
      info = pvm_delhosts((char **)jobManager->hostName, 
			  machineNum, status);

  /* exit from pvm */
   pvm_exit();

  /* remove jobs from JobPool */
   for(i=0; i<actjobNum; i++)
      RemoveJob(jobManager->jobPool, (JobID)i);

   free(jobManager->jobPool);
      
   return jobManager->result; 
    
}

int SendToNewHost(JobManager *jobManager, MachineID currentMachineID, 
		  int numberOfAgents, enum JobStatus sendJobStatus)
{
   int i, j;
   Tid tid;
   int numberOfSpawnedJobs;
   Buffer *bufptr;
   Info info;
   AgentID currentAgentID;
   int signal;
   Job *jobptr;
   JobID localJobID;

/* spawn job in a new-born machine */
  for(j=0; j<numberOfAgents; j++)
  {
    for(i=0; i<GetTotalJobNumber(jobManager->jobPool); i++)
    {
       /* find the job to be sent */ 
       jobManager->resendJobID = jobManager->resendJobID%
				    GetTotalJobNumber(jobManager->jobPool);
       if(jobManager->jobPool->job[jobManager->resendJobID]->jobStatus == 
	   sendJobStatus)
       {
	  /* spawn job */
          numberOfSpawnedJobs = 
                  pvm_spawn(GetJobName(jobManager->jobPool->
			    job[jobManager->resendJobID]),
		           (char **)0, PvmTaskHost,
		           jobManager->machines[currentMachineID].name,
			   1, &tid);
          if(numberOfSpawnedJobs == 1)
          {
	     currentAgentID = jobManager->nextAgentID;
	     jobManager->nextAgentID++;
	     jobManager->agent[currentAgentID] =
		      (AgentInfo *) malloc(sizeof(AgentInfo));
             jobManager->agent[currentAgentID]->myTid = tid;
             jobManager->agent[currentAgentID]->hostName = 
				  jobManager->machines[currentMachineID].name;
          }
	  else
	  {
	     printf("spawn %s", GetJobName(jobManager->jobPool->
			    job[jobManager->resendJobID]),
	           " on %s", jobManager->machines[currentMachineID].name, 
		   " fails. \n");
             return 0;
          }
	  
         /* send broadcast data if any */
         pvm_initsend(PvmDataDefault);

         if(jobManager->broadcastBuffer != NULL)
         {
            signal = 1;
            pvm_pkint(&signal, 1, 1);
            PackBuffer(jobManager->broadcastBuffer);
         }
         else
         {
            signal = -1;
            pvm_pkint(&signal, 1, 1);
         } 
         info = pvm_send(jobManager->agent[currentAgentID]->myTid, 0);
         if(info < 0)
         {
	    printf(" Warning: It is fail to send broadcast data to %s \n",
	           jobManager->agent[currentAgentID]->hostName);
	    break;
         }

 /* send private data */
          pvm_initsend(PvmDataDefault);

          jobptr = jobManager->jobPool->job[jobManager->resendJobID];
          pvm_pkint(&(jobptr->type), 1, 1);
          pvm_pkint(&(jobptr->source), 1, 1);
          pvm_pkint(&(jobptr->destination), 1, 1);

          pvm_pkint(&(jobManager->resendJobID), 1, 1);
          localJobID = GetLocalJobID(jobptr);
          pvm_pkint(&localJobID, 1, 1);
          pvm_pkstr(jobptr->jobName);

	  bufptr = jobptr->privateSendBuffer;

          PackBuffer(bufptr);

          info = pvm_send(jobManager->agent[currentAgentID]->myTid, 1);
          if(info < 0)
          {
	     printf(" Warning: It is fail to send private data to %s \n",
	            jobManager->agent[currentAgentID]->hostName);
          }
          else
          {
	     InsertJobID(&(jobManager->machines[currentMachineID]),
			      jobManager->resendJobID);
	     jobManager->machines[currentMachineID].status = SPAWNED;
	     jobManager->machines[currentMachineID].jobCount++;
	     jobManager->jobPool->job[jobManager->resendJobID]->agentID
						  = currentAgentID;
	     jobManager->jobPool->job[jobManager->resendJobID]->machineID 
					          = currentMachineID;
	     jobManager->agent[currentAgentID]->jobID = jobManager->resendJobID;
	     jobManager->resendJobID++;
          }
          break;
       }
       jobManager->resendJobID++;
    }
  }
  return 1;
}
   
 
int Resend(JobManager *jobManager, MachineID currentMachineID, 
	   AgentID currentAgentID,
	   JobID currentJobID, enum JobStatus resendJobStatus)
{
   int i;
   Tid tid;
   int numberOfSpawnedJobs;
   Buffer *bufptr;
   Info info;
   int signal;
   Job *jobptr;
   JobID localJobID;

// find the job to be sent
   for(i=0; i<GetTotalJobNumber(jobManager->jobPool); i++)
   {
       jobManager->resendJobID = jobManager->resendJobID%
			  GetTotalJobNumber(jobManager->jobPool);
       if(jobManager->jobPool->job[jobManager->resendJobID]->jobStatus ==
	     resendJobStatus)
       {
	  if(strcmp(GetJobName(jobManager->jobPool->job[currentJobID]),
                    GetJobName(jobManager->jobPool->
		    job[jobManager->resendJobID])) != 0)
	  {
	     pvm_kill(jobManager->agent[currentAgentID]->myTid);
             numberOfSpawnedJobs = 
                 pvm_spawn(GetJobName(jobManager->jobPool->
		       job[jobManager->resendJobID]),
	               (char **)0, PvmTaskHost,
		       jobManager->machines[currentAgentID].name,
		       1, &tid);
             if(numberOfSpawnedJobs == 1)
             { 
                jobManager->agent[currentAgentID]->myTid = tid;
             }
	     else
	     {
	        printf("spawn %s", GetJobName(jobManager->jobPool->
		       job[jobManager->resendJobID]),
		       " on %s", jobManager->agent[currentAgentID]->hostName, 
		       " fails. \n");
                break;
             }
          }
	  
         // send broadcast data if any
         pvm_initsend(PvmDataDefault);

         if(jobManager->broadcastBuffer != NULL)
         {
            signal = 1;
            pvm_pkint(&signal, 1, 1);
            PackBuffer(jobManager->broadcastBuffer);
         }
         else
         {
            signal = -1;
            pvm_pkint(&signal, 1, 1);
         } 
         info = pvm_send(jobManager->agent[currentAgentID]->myTid, 0);
         if(info < 0)
         {
	    printf(" Warning: It is fail to send broadcast data to %s \n",
	           jobManager->agent[currentAgentID]->hostName);
	    break;
         }

 /* send private data */
          pvm_initsend(PvmDataDefault);

          jobptr = jobManager->jobPool->job[jobManager->resendJobID];
          pvm_pkint(&(jobptr->type), 1, 1);
          pvm_pkint(&(jobptr->source), 1, 1);
          pvm_pkint(&(jobptr->destination), 1, 1);

          pvm_pkint(&(jobManager->resendJobID), 1, 1);
          localJobID = GetLocalJobID(jobptr);
          pvm_pkint(&localJobID, 1, 1);
          pvm_pkstr(jobptr->jobName);

	  bufptr = jobptr->privateSendBuffer;

          PackBuffer(bufptr);

          info = pvm_send(jobManager->agent[currentAgentID]->myTid, 1);
          if(info < 0)
          {
	     printf(" Warning: It is fail to send private data to %s /n",
	            jobManager->agent[currentAgentID]->hostName); 
          }
          else
          {
	     InsertJobID(&(jobManager->machines[currentMachineID]),
			      jobManager->resendJobID);
	     jobManager->machines[currentMachineID].status = SPAWNED;
	     jobManager->jobPool->job[jobManager->resendJobID]->agentID
						  = currentAgentID;
	     jobManager->jobPool->job[jobManager->resendJobID]->machineID 
					          = currentMachineID;
	     jobManager->agent[currentAgentID]->jobID = jobManager->resendJobID;
	     jobManager->resendJobID++;
          }
          return 1; /* resending a job succeeded */
       }
       jobManager->resendJobID++;
    }
    return 0; /* no jobs need to be resent */
}
   
 
/* Initialize JobAgent */
void InitJobAgent(JobAgent *jobAgent)
{
   /* get my tid */
   jobAgent->myTid = pvm_mytid();

   /* get parent's tid */
   jobAgent->parentTid = pvm_parent();
}

Job* Receive(JobAgent *jobAgent)
{
   Job *job;
   Buffer *bufptr;
   int i;
   int info;

   job = (Job *) malloc(sizeof(Job));
   InitJob(job, OUTOF_PLACE);


   /* receive broadcast data */
      pvm_recv(jobAgent->parentTid, 0);

      pvm_upkint(&info, 1, 1);

      if(info != -1)
      {
         /* therte is broadcast data */
         bufptr = job->broadcastReceiveBuffer;

         UnpackBuffer(bufptr);
      }

      /* receive private data */
      pvm_recv(jobAgent->parentTid, 1);

      pvm_upkint(&(job->type), 1, 1);
      pvm_upkint(&(job->source), 1, 1);
      pvm_upkint(&(job->destination), 1, 1);

      pvm_upkint(&jobAgent->currentJobID, 1, 1);
      pvm_upkint(&jobAgent->localJobID, 1, 1);
      pvm_upkstr(job->jobName);

      bufptr = job->privateReceiveBuffer;

      UnpackBuffer(bufptr);

      if(job->source != MAINPROGRAMADDRS)
	 jobAgent->currentDestination = job->source;
      else
	 jobAgent->currentDestination = jobAgent->parentTid;

      return job;
}

void Send(JobAgent *jobAgent, Result *result)
{
   int i;
   Buffer *bufptr;
   ResultID resultID = GetResultID(result);

   jobAgent->currentType = RESULT;
   jobAgent->currentSource = result->source;

   pvm_initsend(PvmDataDefault);

   pvm_pkint(&jobAgent->currentType, 1, 1);
   pvm_pkint(&jobAgent->currentSource, 1, 1);
   pvm_pkint(&jobAgent->currentDestination, 1, 1);

   pvm_pkint(&jobAgent->currentJobID, 1, 1);
   pvm_pkint(&jobAgent->localJobID, 1, 1);
   pvm_pkstr(result->resultName);

   bufptr = result->privateSendBuffer;

   PackBuffer(bufptr);

   pvm_send(jobAgent->currentDestination, 2);

}

void RunJobAgent(JobAgent *jobAgent, Result* (*Worker)(Job *job))
{
   Job *jobptr;
   Result *resultptr;
   while(1)
   {
      /* receive job from Manager */
      jobptr = Receive(jobAgent);
      if(jobAgent->currentJobID == -1)
      {
	 /* no more jobs */
         free(jobptr);
         free(resultptr);
      }
      else
      {
	 /* let Worker do its work */
         resultptr = Worker(jobptr);

	 SetResultID(resultptr, GetJobID(jobptr));
	 SetResultName(resultptr, GetJobName(jobptr));

	 /* send results to Manager */
         Send(jobAgent, resultptr);

         free(jobptr);
         free(resultptr);
      }
   }
}

/* iniltialize Machine */
void InitMachine(Machine *machine)
{
    int i;

    machine->status = DEAD;
    machine->jobCount = 0;
    machine->machineID = -1;
    machine->name = NULL;
    for(i=0; i < MAXJOBNUMBER; i++)
       machine->jobID[i] = -1;
}

/* Insert a JobID */
void InsertJobID(Machine *machine, JobID currentJobID)
{
    machine->jobID[machine->jobCount] = currentJobID;
    machine->jobCount++;
}

/* Remove a JobID */
int RemoveJobID(Machine *machine, JobID currentJobID)
{
   int i;
   int tmpCount = 0;
   JobID tmpJobID;

   for(i=0; i<machine->jobCount; i++)
   {
      tmpJobID = machine->jobID[i];
      if(tmpJobID != currentJobID)  /* remove currentJobID */
      {
	 machine->jobID[tmpCount] = tmpJobID; /* put back jobID */
	 tmpCount++;
      }
   }
   if(machine->jobCount == tmpCount)
   {
      /* tmpCount should be less than jobCount */
      printf("Something is wrong with Machine.jobCount\n");
      return 0;
   }
   else
   {
      machine->jobCount = tmpCount; /* update jobCount */
      return 1;
   }
}

/* Initialize GroupManager */
void InitGroupManager(GroupManager *groupManager, Job *job, JobNumber jobnum,
			   Buffer *bcbuffer, enum RunType runtype,
			   enum AddMode addmode)
{
   int counter;

   groupManager->jobPool = (JobPool *) malloc(sizeof(JobPool));
   InitJobPool(groupManager->jobPool, job, jobnum, OUTOF_PLACE);
   groupManager->result = (Result *) malloc(sizeof(Result)*jobnum);
   for(counter=0; counter<jobnum; counter++)
      InitResult(&(groupManager->result[counter]), OUTOF_PLACE); 
   groupManager->broadcastBuffer = bcbuffer;
   groupManager->machines = (Machine *)malloc(sizeof(Machine)*MAXMACHINENUMBER);
   for(counter=0; counter<MAXMACHINENUMBER; counter++)
      InitMachine(&(groupManager->machines[counter])); 

   groupManager->runType = runtype;
   groupManager->addMode = addmode;
}

Result* RunGroupManager(GroupManager *groupManager)
{
/* variables about hostname */
   char tmpname[LENGTHOFHOSTNAME], *tmp1;
   int len;
   int nofile = 0;
   FILE *pp, *fp;

/* variables about starting virtual machine and spawning jobs */
   Tid mytid, tid;
   Tid tids[MAXJOBNUMBER];
   Info info;
   Info status[MAXAGENTNUMBER];
   int actjobNum;
   int numberOfSpawnedJobs;
   int numberOfAgents; /* Number of JobAgents on one single machine */
   int tmpnumberOfAgents;

/* variables about virtual machine */
   int machineNum;
   int tmpMachineNum;
   MachineID currentMachineID;
   MachineList avamachineList;
   MachineList unavamachineList;
   AgentID currentAgentID;
   PackageType currentType;
   Tid currentSource;
   Tid currentDestination;
   JobID localJobID;

/* variables about Job and JobPool */
   JobID currentJobID;
   JobID nextJobID;
   JobID sendJobID;
   JobID tmpnextJobID;
   Buffer *bufptr;
   int signal; 
   int arrived;
   Job *jobptr;
   char jobName[LENGTHOFJOBNAME];
	  
   int i, j;

/* get host names from file "hostname". */
   machineNum = 0;

/* open file "hostname" */

   fp = fopen("hostname", "r");

   if(!fp)   
   {
      nofile = 1;
   }
   else
   {
      /* read machine names from "hostname" */
      while(!feof(fp))
      {
         fscanf(fp, "%s", tmpname);
         len = strlen((char *)tmpname);
         if(len != 0)
         {
	    tmp1 = (char *) malloc(sizeof(char)*(len + 1));
	    groupManager->hostName[machineNum] 
		  = strncpy(tmp1, tmpname, len+1);
	    machineNum++;
         }
      }
      fclose(fp);
   }
    
   machineNum--;

   /* get my name */
   pp = popen("hostname", "r");
   fscanf(pp, "%s", &tmpname);
   pclose(pp);
   len = strlen((char *)tmpname);
   if(len != 0)
   {
      tmp1 = (char *) malloc(sizeof(char)*(len + 1));
      groupManager->hostName[machineNum] 
		  = strncpy(tmp1, tmpname, len+1);
      machineNum++;
   }

/* don't count myself twice and put myself at the end */
   tmpMachineNum = machineNum - 1;
   machineNum = 0;
   for(i=0; i<tmpMachineNum; i++)
   {
      if(strcmp(groupManager->hostName[i], 
		       groupManager->hostName[tmpMachineNum]) != 0)
      {
         groupManager->hostName[machineNum]
		  = groupManager->hostName[i];	 
	 machineNum++;
     }
   }
   groupManager->hostName[machineNum]
	  = groupManager->hostName[tmpMachineNum];
   machineNum++;
 
/* add machines into pvm */
   for(i=0; i<machineNum - 1; i++)
   {
      info = pvm_addhosts((char **)&(groupManager->hostName[i]), 1, status);

      if(info == 1 || (info == 0 && groupManager->addMode == ANY))
      {
	 groupManager->machines[i].status = ALIVE;
	 groupManager->machines[i].name = groupManager->hostName[i];
      }
      else
      {
	 groupManager->machines[i].status = DEAD;
	 groupManager->machines[i].name = groupManager->hostName[i];
      }
   }

/* add myself into pvm */
      groupManager->machines[machineNum-1].status = ALIVE;
      groupManager->machines[machineNum-1].name = 
			  groupManager->hostName[machineNum-1];
   

/* How many jobs are spawned on one single machine simultaneously? */
   switch(groupManager->runType)
   {
      case NICE:
	 numberOfAgents = 1;
	 break;
      case NORMAL:
	 numberOfAgents = 2;
	 break;
      case FAST:
	 numberOfAgents = 3;
	 break;
      default:
	 numberOfAgents = 2;
	 break;
    }

   /* spawn jobs */
   actjobNum = 0;
   for(j=0; j<numberOfAgents; j++)
   {
      for(i=0; i<machineNum; i++)
      {
         if(actjobNum >= GetTotalJobNumber(groupManager->jobPool)) break;
         if(groupManager->machines[i].status != DEAD)
	 {
            numberOfSpawnedJobs = 
		  pvm_spawn(GetJobName(groupManager->jobPool->job[actjobNum]),
		       (char **)0, PvmTaskHost,
		       groupManager->machines[i].name, 1, &tid);
            if( numberOfSpawnedJobs == 1)
            {
	       InsertJobID(&(groupManager->machines[i]), actjobNum);
	       groupManager->machines[i].status = SPAWNED;
	       groupManager->machines[i].jobCount++;

               groupManager->agent[actjobNum] = 
				    (AgentInfo *) malloc(sizeof(AgentInfo));
               groupManager->agent[actjobNum]->myTid = tid;
               groupManager->agent[actjobNum]->hostName = groupManager->hostName[i];
               groupManager->agent[actjobNum]->jobID = actjobNum;
	       groupManager->jobPool->job[actjobNum]->agentID = actjobNum;
	       groupManager->jobPool->job[actjobNum]->machineID = i;
	       tids[actjobNum] = tid;
	       actjobNum++;
            }
	 }
      }
   }

   /* next job to be sent */
   nextJobID = actjobNum;

 /* send data  */
   for(j=0; j<actjobNum; j++)
   {
      /* send broadcast data */
      pvm_initsend(PvmDataDefault);

      if(groupManager->broadcastBuffer != NULL)
      {
         /* there is broadcast buffer */
         signal = 1;
         pvm_pkint(&signal, 1, 1);
         PackBuffer(groupManager->broadcastBuffer);
      }
      else
      {
         /* there is not broadcast buffer */
         signal = -1;
         pvm_pkint(&signal, 1, 1);
      } 

      info = pvm_send(groupManager->agent[j]->myTid, 0);

      // send private data
      pvm_initsend(PvmDataDefault);

      jobptr = groupManager->jobPool->job[j];
      pvm_pkint(&(jobptr->type), 1, 1);
      pvm_pkint(&(jobptr->source), 1, 1);
      pvm_pkint(&(jobptr->destination), 1, 1);

      pvm_pkint(&j, 1, 1);
      localJobID = GetLocalJobID(jobptr);
      pvm_pkint(&localJobID, 1, 1);
      pvm_pkstr(jobptr->jobName);

      bufptr = jobptr->privateSendBuffer;

      PackBuffer(bufptr);

      info = pvm_send(groupManager->agent[j]->myTid, 1);
      if(info < 0)
      {
	 printf(" Warning: It is fail to send private data to %s \n",
	      groupManager->agent[j]->hostName);
      }
      else
      {
	 jobptr->jobStatus = SENT;
      }

   }

/* Receive data from JobAgents */

  while(GetJobNumber(groupManager->jobPool) > 0) /* there is jobs uncompleted */
  {
    arrived = pvm_nrecv(-1, 2); /* detect whether there are data received */
    if(arrived > 0)  
    {
      /* there are data received */
      pvm_upkint(&currentType, 1, 1);
      pvm_upkint(&currentSource, 1, 1);
      pvm_upkint(&currentDestination, 1, 1);

    if(currentType == RESULT || currentType == CONFIRMATION)
    {
      /* find the current JobID */
      pvm_upkint(&currentJobID, 1, 1);
      pvm_upkint(&localJobID, 1, 1);
      pvm_upkstr(jobName);

      /* find the current AgentID */
      currentAgentID = groupManager->jobPool->job[currentJobID]->agentID;

      /* find the current MachineID */
      currentMachineID = groupManager->jobPool->job[currentJobID]->machineID;
     
      /* print out JobID and MachineID */
      printf("Received: JobID = %d ", currentJobID); 
      printf("  MachineID = %d \n", currentMachineID);


      if(groupManager->jobPool->job[currentJobID]->jobStatus == SENT)
      {
	 if(currentType == RESULT)
	 {
            /* the current job is waiting for the results */
            bufptr = groupManager->result[currentJobID].privateReceiveBuffer;

	    /* unpack pvm buffer to the received Buffer of Result */
            UnpackBuffer(bufptr);
         }

         RemoveJobID(&(groupManager->machines[currentMachineID]), currentJobID);

	 /* set the status of the current job FINISHED */ 
         groupManager->jobPool->job[currentJobID]->jobStatus = FINISHED;

	 /* The size of JobPool is reduced by one */
	 ReduceJobNumberByOne(groupManager->jobPool);

	 if(GetJobNumber(groupManager->jobPool) > 0)
	 {
	    /* there is jobs unfinished */
	    if(nextJobID < GetTotalJobNumber(groupManager->jobPool))
	    {
	       sendJobID = nextJobID;

	       if(strcmp(GetJobName(groupManager->jobPool->job[currentJobID]),
	                 GetJobName(groupManager->jobPool->job[sendJobID]))
                         != 0)
	       {
		  /* The job name to be sent is different from that of */
		  /* the current job so a new program to be spawned */
		  pvm_kill(groupManager->agent[currentAgentID]->myTid);
                  numberOfSpawnedJobs = 
		    pvm_spawn(GetJobName(groupManager->jobPool->job[sendJobID]),
		       (char **)0, PvmTaskHost,
		       groupManager->agent[currentAgentID]->hostName, 1, &tid);
                  if(numberOfSpawnedJobs == 1)
                  {
                     groupManager->agent[currentAgentID]->myTid = tid;
                  }
		  else
		  {
	              printf("spawn %s", GetJobName(groupManager->jobPool->
			    job[sendJobID]),
	           " on %s", groupManager->agent[currentAgentID]->hostName, 
		   " fails. \n");
                     break;
                  }
               }
                    
                /* send broadcast data if any */
                pvm_initsend(PvmDataDefault);

                if(groupManager->broadcastBuffer != NULL)
                {
                   signal = 1; // there is a broadcast Buffer
                   pvm_pkint(&signal, 1, 1);
                   PackBuffer(groupManager->broadcastBuffer);
                }
                else
                {
                   signal = -1; // there is not any broadcast Buffer
                   pvm_pkint(&signal, 1, 1);
                } 
                info = pvm_send(groupManager->agent[currentAgentID]->myTid, 0);
                if(info < 0)
                {
	          printf(" Warning: It is fail to send broadcast data to %s \n",
	                groupManager->agent[currentAgentID]->hostName);
		   break;
                }

               /* send private data */
               pvm_initsend(PvmDataDefault);

               jobptr = groupManager->jobPool->job[sendJobID];
               pvm_pkint(&(jobptr->type), 1, 1);
               pvm_pkint(&(jobptr->source), 1, 1);
               pvm_pkint(&(jobptr->destination), 1, 1);

               pvm_pkint(&sendJobID, 1, 1);
               localJobID = GetLocalJobID(jobptr);
               pvm_pkint(&localJobID, 1, 1);
               pvm_pkstr(jobptr->jobName);

	       bufptr = jobptr->privateSendBuffer;

               PackBuffer(bufptr);

               info = pvm_send(groupManager->agent[currentAgentID]->myTid, 1);
               if(info < 0)
               {
	          printf(" Warning: It is fail to send private data to %s \n",
	                groupManager->agent[currentAgentID]->hostName);
               }
               else
               {
	          InsertJobID(&(groupManager->machines[currentMachineID]), sendJobID);
	          groupManager->machines[currentMachineID].status = SPAWNED;

	          groupManager->jobPool->job[sendJobID]->jobStatus = SENT;
	          groupManager->jobPool->job[sendJobID]->agentID = currentAgentID;
	          groupManager->jobPool->job[sendJobID]->machineID = currentMachineID;
	          groupManager->agent[currentAgentID]->jobID = sendJobID;
		  nextJobID++; // The sent job is not a unsent job so
		   	       // nextJobID is incresed by one.
               }
            }
	    else
	    {
	       pvm_kill(groupManager->agent[currentAgentID]->myTid);
	       groupManager->machines[currentMachineID].jobCount--;
            }
         }
      }
     }
     }
     arrived = pvm_nrecv(-1, 3);
     if(arrived > 0)  
     {
	pvm_upkint(&currentType, 1, 1);
	if(currentType == REQUEST)
	{
	   jobptr = (Job *) malloc(sizeof(Job));
	   InitJob(jobptr, OUTOF_PLACE);

	   pvm_upkint(&currentSource, 1, 1);
	   pvm_upkint(&currentDestination, 1, 1);
	   pvm_upkint(&currentJobID, 1, 1);
	   pvm_upkint(&localJobID, 1, 1);

	   jobptr->type = currentType;
	   jobptr->source = currentSource;
	   jobptr->destination = currentDestination;

	   SetLocalJobID(jobptr, localJobID);

	   pvm_upkstr(jobName);
	   SetJobName(jobptr, jobName);

	   bufptr = jobptr->privateSendBuffer;
	   UnpackBuffer(bufptr);

	   /* add the requested job into JobPool */
	   AddJob(groupManager->jobPool,jobptr, 1);
        }
     }
  }

   /* kill all of agents */
   for(i=0; i<groupManager->nextAgentID; i++)
      pvm_kill(groupManager->agent[i]->myTid);

  /* exit from pvm */
  /* pvm_exit(); */

  /* remove jobs from JobPool */
   for(i=0; i<actjobNum; i++)
      RemoveJob(groupManager->jobPool, (JobID)i);

   free(groupManager->jobPool);
      
   return groupManager->result; 
    
}

/* Initialize Distributor */
void InitDistributor(Distributor *distributor, Job *job, 
		     JobNumber jobnum, Buffer *bcbuffer)
{
   int counter;

   distributor->jobPool = (JobPool *) malloc(sizeof(JobPool));
   InitJobPool(distributor->jobPool, job, jobnum, OUTOF_PLACE);
   distributor->result = (Result *) malloc(sizeof(Result)*jobnum);
   for(counter=0; counter<jobnum; counter++)
      InitResult(&(distributor->result[counter]), OUTOF_PLACE); 
   distributor->broadcastBuffer = bcbuffer;
}

Result* RunDistributor(Distributor *distributor)
{
    int jobCounter;
    int totalJobNum;
    Job *jobptr;
    Buffer *bufptr;
    int signal;
    Info info;
    PackageType currentType;
    Tid currentSource;
    Tid currentDestination;
    JobID currentJobID;
    JobID currentLocalJobID;
    JobID nextJobID;
    int arrived;
    char jobName[LENGTHOFJOBNAME];

    // variables about hostname
    char *hostName;
    char tmpname[LENGTHOFHOSTNAME], *tmp1;
    int  len;
    FILE *pp;

    Tid  tidOfMyWorker;

    Tid source = pvm_mytid();
    Tid destination = pvm_parent();

    totalJobNum = GetTotalJobNumber(distributor->jobPool);

    if(totalJobNum > 1)
    {
       for(jobCounter=0; jobCounter<totalJobNum-1; jobCounter++)
       {
          jobptr = distributor->jobPool->job[jobCounter];
	  SetLocalJobID(jobptr, jobCounter);
          jobptr->type = REQUEST;
          jobptr->source = source;
          jobptr->destination = destination;

          // send private data
          pvm_initsend(PvmDataDefault);

          pvm_pkint(&(jobptr->type), 1, 1);
          pvm_pkint(&(jobptr->source), 1, 1);
          pvm_pkint(&(jobptr->destination), 1, 1);

          pvm_pkint(&jobptr->jobID, 1, 1);
          pvm_pkint(&jobptr->localJobID, 1, 1);
          pvm_pkstr(jobptr->jobName);

          bufptr = jobptr->privateSendBuffer;

          PackBuffer(bufptr);

          info = pvm_send(destination, 3);
          if(info < 0)
          {
             printf(" Warning: It fails to send private data out \n");
          }
          else
          {
	     jobptr->jobStatus = SENT;
          }
       }
    }

   //get my name
   pp = popen("hostname", "r");
   fscanf(pp, "%s", &tmpname);
   pclose(pp);
   len = strlen((char *)tmpname);
   if(len != 0)
   {
      tmp1 = (char *) malloc(sizeof(char)*(len + 1));
      hostName = strncpy(tmp1, tmpname, len+1);
   }
 
   /* spawn jon on self machine */
   info = pvm_spawn(GetJobName(distributor->jobPool->job[totalJobNum-1]),
		    (char **)0, PvmTaskHost,
		    hostName, 1, &tidOfMyWorker);
   if( info != 1)
   {
      printf("error when spawning job on self machine \n" );
   }

   /* set job properties */
   jobptr = distributor->jobPool->job[totalJobNum-1];
   SetLocalJobID(jobptr, totalJobNum-1);
   jobptr->type = REQUEST;
   jobptr->source = source;
   jobptr->destination = tidOfMyWorker;

   // send broadcast data if any
   pvm_initsend(PvmDataDefault);

   if(distributor->broadcastBuffer != NULL)
   {
        signal = 1; /* there is a broadcast Buffer */
        pvm_pkint(&signal, 1, 1);
        PackBuffer(distributor->broadcastBuffer);
   }
   else
   {
        signal = -1; /* there is not any broadcast Buffer */
        pvm_pkint(&signal, 1, 1);
   } 
   info = pvm_send(tidOfMyWorker, 0);
   if(info < 0)
   {
     printf(" Warning: It is fail to send broadcast data to %s \n",
          hostName);
   }

   /* send private data */
   pvm_initsend(PvmDataDefault);

   pvm_pkint(&(jobptr->type), 1, 1);
   pvm_pkint(&(jobptr->source), 1, 1);
   pvm_pkint(&(jobptr->destination), 1, 1);

   pvm_pkint(&jobptr->jobID, 1, 1);
   pvm_pkint(&jobptr->localJobID, 1, 1);
   pvm_pkstr(jobptr->jobName);

   bufptr = jobptr->privateSendBuffer;

   PackBuffer(bufptr);

   info = pvm_send(tidOfMyWorker, 1);
   if(info < 0)
   {
      printf(" Warning: It fails to send private data out %s \n");
   }
   else
   {
      jobptr->jobStatus = SENTTOSELF;
   }
  
	  
  nextJobID = 0;

  while(GetJobNumber(distributor->jobPool) > 0)  /* there is jobs uncompleted */
  {
     arrived = pvm_nrecv(-1, 2); /* detect whether there are data received */
     if(arrived > 0)  
     {
       /* there are data received */
       pvm_upkint(&currentType, 1, 1);
       pvm_upkint(&currentSource, 1, 1);
       pvm_upkint(&currentDestination, 1, 1);

       if(currentDestination == source && currentType == RESULT)
       {
	  pvm_upkint(&currentJobID, 1, 1);

	  pvm_upkint(&currentLocalJobID, 1, 1);
	  pvm_upkstr(jobName);

	  jobptr = distributor->jobPool->job[currentLocalJobID];
	  if(jobptr->jobStatus == SENT || jobptr->jobStatus == SENTTOSELF)
	  {
	     bufptr = distributor->result[currentLocalJobID].privateReceiveBuffer;

	     UnpackBuffer(bufptr);

	     jobptr->jobStatus = FINISHED;

	     ReduceJobNumberByOne(distributor->jobPool);

	     if(jobptr->jobStatus == SENT)
	     {
                /* send confirmation to JobManager */
	        currentType = CONFIRMATION;

	        pvm_initsend(PvmDataDefault);

                pvm_pkint(&currentType, 1, 1);
                pvm_pkint(&source, 1, 1);
                pvm_pkint(&destination, 1, 1);

	        pvm_pkint(&currentJobID, 1, 1);

	        pvm_pkint(&currentLocalJobID, 1, 1);
	        pvm_pkstr(jobName);

	        info = pvm_send(destination, 2);
                if(info < 0)
                {
                   printf(" Warning: It fails to send CONFIRMATION out %s \n");
                }
              }
              else
	      {
		 jobptr = distributor->jobPool->job[nextJobID];
		 if(jobptr->jobStatus == SENT)
		 {
                    /* set job properties */
                    jobptr->destination = tidOfMyWorker;

	            if(strcmp(GetJobName(distributor->jobPool->
	                 job[currentLocalJobID]),
	                 GetJobName(distributor->jobPool->job[nextJobID]))
                         != 0)
	            {
		        /* The job name to be sent is different from that of
		         the current job so a new program to be spawned */
		        pvm_kill(tidOfMyWorker);
                       info = pvm_spawn(GetJobName(distributor->
					jobPool->job[nextJobID]),
		                        (char **)0, PvmTaskHost,
		                        hostName, 1, &tidOfMyWorker);
                       if(info != 1)
		       {
		          printf("error when spawning a job %s \n");
                       }
                    }

                    /* send broadcast data if any */
                    pvm_initsend(PvmDataDefault);

                    if(distributor->broadcastBuffer != NULL)
                    {
                       signal = 1; /* there is a broadcast Buffer */
                       pvm_pkint(&signal, 1, 1);
                       PackBuffer(distributor->broadcastBuffer);
                    }
                    else
                    {
                       signal = -1; /* there is not any broadcast Buffer */
                       pvm_pkint(&signal, 1, 1);
                    } 
                    info = pvm_send(tidOfMyWorker, 0);
                    if(info < 0)
                    {
                        printf(" Warning: It is fail to send broadcast 
			       data to %s \n",
                               hostName);
                    }

                    /* send private data */
                    pvm_initsend(PvmDataDefault);

                    pvm_pkint(&(jobptr->type), 1, 1);
                    pvm_pkint(&(jobptr->source), 1, 1);
                    pvm_pkint(&(jobptr->destination), 1, 1);

                    pvm_pkint(&jobptr->jobID, 1, 1);
                    pvm_pkint(&jobptr->localJobID, 1, 1);
                    pvm_pkstr(jobptr->jobName);

                    bufptr = jobptr->privateSendBuffer;

                    PackBuffer(bufptr);

                    info = pvm_send(tidOfMyWorker, 1);
                    if(info < 0)
                    {
                       printf(" Warning: It fails to send private data out %s \n");
                    }
                    else
                    {
                       jobptr->jobStatus = SENTTOSELF;
                    }
                 }
		 nextJobID = (nextJobID++)%totalJobNum;
	      } 
           }
         }
      }
   }
   return distributor->result;
}
